// ===================================================================
// noise.h
//	Header file for Noise classes.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

// ---------------------------------------------------------
// NoiseMaker
//	Implements solid noise functions.  Given a solid 
//	noise function (Noise), NoiseMaker implements a
//	vector-valued noise function (DNoise), a turbulence 
//	function, and a vector-valued turbulence function.
// ---------------------------------------------------------

// ---------------------------------------------------------
// This vector is used to formulate vector-valued noise 
// functions in terms of the scalar-valued noise function
// (see NoiseMaker::DNoise).
// ---------------------------------------------------------
#define BIGVECT Vector3D(1000.2, 1000.5, 1000.8)

class NoiseMaker {
public:
    NoiseMaker() { }
    virtual ~NoiseMaker() { }

    // Pure virtual Noise function.
    virtual float Noise(const Vector3D& x) = 0;

    // Vector3D-valued noise function, formulated in terms of the
    // scalar-valued noise function.
    virtual Vector3D DNoise(const Vector3D& x) {
	return Vector3D(Noise(x - BIGVECT), Noise(x), Noise(x + BIGVECT));
    }

    // Scalar and vector turbulence functions, formulated in terms of
    // the scalar-valued noise function.
    virtual float Turbulence(const Vector3D& x);
    virtual Vector3D DTurbulence(const Vector3D& x);
};

// ---------------------------------------------------------------
// WhiteNoise
//	Linearly interpolates between noise values at lattice
//	points
// ---------------------------------------------------------------
class WhiteNoise : public NoiseMaker {
public:
    WhiteNoise(int NumValues = 509);
    virtual ~WhiteNoise();
    virtual float Noise(const Vector3D& x);
private:
    int *P;	// Array containing random permutation of 0..NumValues-1.
    float *R;	// Table of n random values
    int n;

    int Phi(long x) {
	return P[labs(x) % n];
    }
};

// ---------------------------------------------------------
// PerlinNoise 
//	Implements the solid noise function described in
//	Ken Perlin's "Hypertexture," SIGGRAPH '89, pg. 253.
// ---------------------------------------------------------

#define NUM_NOISE_POINTS 4096

class PerlinNoise : public NoiseMaker {
public:
    PerlinNoise();
    PerlinNoise(int seed);
    virtual ~PerlinNoise();
    virtual float Noise(const Vector3D& x);
private:
    Vector3D *G;	// An array of unit vectors uniformly distributed
			// about the unit sphere.
    int *P;		// An array containing a random permutation of
			// 0..NUM_NOISE_POINTS-1.

    // Hermite curve interpolation function.
    float w(float t) {
	t = fabs(t);
	if (t < 1)
	    return t*t*(-3 + 2*t) + 1;
	else
	    return 0;
    }

    // Truncate x to fit within the randomly permuted table, then
    // return the corresponding randomly permuted value.
    int Phi(long x) {
	return P[labs(x) & (NUM_NOISE_POINTS - 1)];
    }

    // 3D hashing function; given the truncated values of a vector
    // in 3-space, return the randomized vector corresponding to the
    // position.
    Vector3D Gamma(long i, long j, long k) {
	return G[Phi(i+Phi(j+Phi(k)))];
    }

    // Do a Hermite interpolation in the three dimensions.
    float Omega(const Vector3D& A, long x, long y, long z) {
	float dot = DotProd(Gamma(x, y, z), Vector3D(fabs(A.x), fabs(A.y), fabs(A.z)));
	return w(A.x)*w(A.y)*w(A.z)*dot;
    }

    // Allocates and computes G and P.
    void Initialize();

};


