// ===================================================================
// object3d.h
//	Definitions required to implement the Object3D abstract class
//	and the hierarchy rooted by it.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

// Forward-declare some classes.
class Object3D;
class AxisAlignedBox;
class Surface;
class World;
class Ray;
class SpanList;

// ----------------------------------------------------------------
// SurfaceList
//	List of colors; derives from SimpleList and adds a function
//	to search for a given color and return nonzero if there.
// ----------------------------------------------------------------
class SurfaceList : public SimpleList<Surface *> {
public:
    int SurfaceThere(Surface *probe);
};

class ObjectList : public SimpleList<Object3D *> {
public:
    ObjectList(): SimpleList<Object3D *>() { }
    ObjectList(Object3D * & x): SimpleList<Object3D *>(x) { }
    void TransformList(const Matrix& tform);
    void TransformSurfaces(SurfaceList& clist, const Matrix& tform);
    int SingleObject() { return n == 1; }
};

// --------------------------------------------------------
// AxisAlignedBox class.
//	This is a box in 3-space, defined by min and max
//	corner vectors.  Many useful operations are defined
//	on this
// --------------------------------------------------------

class AxisAlignedBox {
public:
    // Constructors.
    AxisAlignedBox() { }
    AxisAlignedBox(const Vector3D& Min, const Vector3D& Max): min(Min), max(Max) { }

    // Read-only access to min and max vectors.
    Vector3D Min() const;
    Vector3D Max() const;

    // Return whether the box is unbounded.  Unbounded boxes appear
    // when unbounded objects such as quadric surfaces are included.
    int Unbounded() const;

    // Expand the axis-aligned box to include the given object.
    void Include(const Vector3D& newpt);
    void Include(const AxisAlignedBox& bbox);

    // Overlap returns 1 if the two axis-aligned boxes overlap.
    friend int Overlap(const AxisAlignedBox&, const AxisAlignedBox&);

    // Returns the smallest axis-aligned box that includes all points
    // inside the two given boxes.
    friend AxisAlignedBox Union(const AxisAlignedBox& x, const AxisAlignedBox& y);

    // Returns the intersection of two axis-aligned boxes.
    friend AxisAlignedBox Intersect(const AxisAlignedBox& x, const AxisAlignedBox& y);

    friend AxisAlignedBox Transform(const AxisAlignedBox& box, const Matrix& tform);

    virtual float SurfaceArea() const;
    int ComputeMinMaxT(Ray& ray, float *tmin, float *tmax) const;

    friend class Ray;

protected:
    Vector3D min, max;
};

inline int 
Overlap(const AxisAlignedBox& x, const AxisAlignedBox& y)
{
    if (x.max.x < y.min.x ||
	x.min.x > y.max.x ||
	x.max.y < y.min.y ||
	x.min.y > y.max.y ||
	x.max.z < y.min.z ||
	x.min.z > y.max.z) {
	return 0;
    }
    return 1;
}

inline AxisAlignedBox
Intersect(const AxisAlignedBox& x, const AxisAlignedBox& y)
{
    if (x.Unbounded())
	return y;
    else if (y.Unbounded())
	return x;
    AxisAlignedBox ret = x;
    if (Overlap(ret, y)) {
	Maximize(ret.min, y.min);
	Minimize(ret.max, y.max);
	return ret;
    }
    else	    // Null intersection.
	return AxisAlignedBox(Vector3D(0), Vector3D(0));
}

inline AxisAlignedBox
Union(const AxisAlignedBox& x, const AxisAlignedBox& y)
{
    Vector3D min = x.min;
    Vector3D max = x.max;
    Minimize(min, y.min);
    Maximize(max, y.max);
    return AxisAlignedBox(min, max);
}

inline AxisAlignedBox 
Transform(const AxisAlignedBox& box, const Matrix& tform)
{
    AxisAlignedBox ret(Vector3D(MAXFLOAT), Vector3D(-MAXFLOAT));
    ret.Include(tform * Vector3D(box.min.x, box.min.y, box.min.z));
    ret.Include(tform * Vector3D(box.min.x, box.min.y, box.max.z));
    ret.Include(tform * Vector3D(box.min.x, box.max.y, box.min.z));
    ret.Include(tform * Vector3D(box.min.x, box.max.y, box.max.z));
    ret.Include(tform * Vector3D(box.max.x, box.min.y, box.min.z));
    ret.Include(tform * Vector3D(box.max.x, box.min.y, box.max.z));
    ret.Include(tform * Vector3D(box.max.x, box.max.y, box.min.z));
    ret.Include(tform * Vector3D(box.max.x, box.max.y, box.max.z));
    return ret;
}


// Flags that can be attached to Object3D's.
#define BOUNDING	1
#define CSG		2
#define AGGREGATE	4

class Object3D {
public:
    // Constructors set flags and color.
    Object3D(int F = 0): surf(0), flags(F) { }
    Object3D(Surface *NewColor, int F):
    	surf(NewColor), flags(F) { }

    // Make destructor explicitly virtual so it can be overloaded.
    virtual ~Object3D() { }

    Surface *ReturnColor() const { return surf; }
    int Flags() const { return flags; }
    void OverrideColor(Surface *NewColor) { surf = NewColor; }

    // NearestInt returns 1 if the ray hits the object; it passes back
    // the parameter of intersection in t.  maxt specifies the maximum
    // allowable value for the parameter of intersection; if the only 
    // intersections with the object are of greater parameter than maxt,
    // no intersection is returned.
    virtual int NearestInt(Ray& ray, float& t, float maxt) = 0;

    // FindAllIntersections returns a pointer to a SpanList that
    // contains all the spans where the ray passes through the
    // object.  This is primarily used for CSG.
    virtual SpanList *FindAllIntersections(Ray& ray) = 0;

    // Returns the normal to the object at the given intersection point.
    virtual Vector3D FindNormal(const Vector3D& p) = 0;

    // Returns the sum of the object's ambient, diffuse, specular,
    // reflective, and transparent components.
    //
    // This was unneeded (the World class knew how to do this stuff
    // in FindColor) until hypertexture objects came along.  Since
    // these objects get shaded quite a bit differently from other
    // objects, ShadeObject has a default implementation for all
    // objects that gets overloaded by the SoftObject (hypertexture)
    // class.
    virtual RGBColor ShadeObject(World&, Ray&, Vector3D&, int);

    // Returns the number of children the object has.  This is defined
    // to be 0, except for bounding volumes.
    virtual int NumChildren() { return 0; }

    // Apply an arbitrary 3D 4x4 matrix to the object.
    virtual void ApplyTransform(const Matrix& tform) = 0;

    // Applies a given transform to the Surface structure associated
    // with the object.
    virtual void TransformSurface(SurfaceList& clist, const Matrix& tform);

    // Return a duplicate of the object.
    virtual Object3D *Dup() const = 0;

    // Return nonzero if the given point is inside the object, 0 if not.
    virtual int IsInside(Vector3D& v);

    // Return nonzero if the given ray intersects the object with
    // parameter of intersection less than maxt.  This is frequently
    // more efficient to implement than find-nearest-intersection.
    virtual int HitAtAll(Ray& ray, float maxt);

    // Return a measure of how far inside the object the given point is.
    // This distance is 0 on the object's surface, negative if inside the
    // object, or positive if outside the object.  If the number is
    // positive or negative, it should reflect how far inside or outside
    // the object the given point is.

    // This function is used strictly by the SoftObject hypertexture class.
    virtual float PtDistance(Vector3D& v) = 0;

    // Writes a brief description of the object, indenting by the given
    // number of spaces first.
    virtual void Describe(int ind) const = 0;

    // Returns bounding box of object.	Unbounded by default.
    virtual AxisAlignedBox BBox() const {
    	return AxisAlignedBox(Vector3D(-MAXFLOAT), Vector3D(MAXFLOAT));
    }

    virtual void CountMe() const = 0;

    friend World;
protected:
    Surface *surf;	// Object surface.
private:
    int flags;		// Flags: see just above for #define's
};

// -------------------------------------------------------------------
// BoundingVolume
//	This abstract class is derived from Object3D.  It contains a
//	list of other objects (those contained in the bounding
//	volume).  Member functions to manipulate the list are
//	provided.
// -------------------------------------------------------------------
class BoundingVolume: public Object3D {
public:
    BoundingVolume() : Object3D(BOUNDING) { }
    virtual int NumChildren() { return contents.ListCount(); }
    virtual Vector3D FindNormal(const Vector3D& intersection) = 0;
    virtual void ApplyTransform(const Matrix& tform) = 0;
    virtual void TransformSurface(SurfaceList& clist, const Matrix& tform);
    virtual SpanList *FindAllIntersections(Ray& ray) = 0;
    virtual Object3D *Dup() const = 0;
    virtual void Describe(int ind) const = 0;
    virtual float PtDistance(Vector3D& v) = 0;
    virtual AxisAlignedBox BBox() const = 0;
    virtual void CountMe() const = 0;

    virtual float SurfaceArea() const = 0;

    virtual void AddObject(const Object3D& NewObj); // add new instance of object
    virtual void AddObject(Object3D *NewObj);	    // put object in list
    virtual ObjectList *List() { return &contents; }

    friend World;

protected:
    ObjectList contents;	// Contents of bounding volume

};


// -------------------------------------------------------------------
// BoundingBox
//	Derived multiply from BoundingVolume and AxisAlignedBox, this
//	class encapsulates the axis-aligned bounding boxes used by
//	OORT.
// -------------------------------------------------------------------
class BoundingBox : public BoundingVolume, public AxisAlignedBox {
public:
    BoundingBox(): BoundingVolume(),
	AxisAlignedBox(Vector3D(MAXFLOAT), Vector3D(-MAXFLOAT)) { }
    BoundingBox(const Vector3D& Min, const Vector3D& Max): BoundingVolume(),
	AxisAlignedBox(Min, Max) { }
    BoundingBox(AxisAlignedBox& x): BoundingVolume(), AxisAlignedBox(x) { }

    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual int NearestInt(Ray& ray, float& t, float maxt);
    virtual Vector3D FindNormal(const Vector3D& intersection);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const;
    virtual int IsInside(Vector3D& v);
    virtual float PtDistance(Vector3D& v);
    virtual void Describe(int ind) const;
    virtual AxisAlignedBox BBox() const;
    virtual void CountMe() const { Statistics::Objects::BBox++; }
    virtual float SurfaceArea() const;

    virtual void AddObject(const Object3D&);
    virtual void AddObject(Object3D *);

    friend class Quadric;
    friend class Ray;
private:
    int ComputeMinMaxT(Ray& ray, float *tmin, float *tmax);
};

// -------------------------------------------------------------------
// Aggregate
//		Basically a collection of objects that get transformed
//		together.
// -------------------------------------------------------------------
class Aggregate : public Object3D {
public:
    Aggregate(): Object3D(AGGREGATE) { surf = 0; }
    Aggregate(const Aggregate& x): Object3D(AGGREGATE), list(x.list) { }
    Aggregate(const ObjectList&);

    virtual int NearestInt(Ray& ray, float& t, float maxt);
    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Vector3D FindNormal(const Vector3D& intersection);
    virtual void ApplyTransform(const Matrix& tform);
    virtual void TransformSurface(SurfaceList& clist, const Matrix& tform);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual float PtDistance(Vector3D& v);
    virtual AxisAlignedBox BBox() const;
    virtual void CountMe() const;

    ObjectList& List() { return list; }
    void AddObject(Object3D *);
    void AddObject(const Object3D&);

    void NewSurface(Surface *);
private:
    ObjectList list;
};

// This type of Object has a canonical form, so you "transform" it
// by transforming rays that you're intersecting with it.
class InverseRayObject : public Object3D {
public:
    InverseRayObject(int f = 0): Object3D(f) {
	mat = inv = IdentityMatrix();
    }
    InverseRayObject(Surface *clr, int f = 0): Object3D(clr, f) {
	mat = inv = IdentityMatrix();
    }

    // Pass along responsibility for intersection calculations.
    virtual int NearestInt(Ray& ray, float& t, float maxt) = 0;
    virtual SpanList *FindAllIntersections(Ray& ray) = 0;

    // Apply a transform to the object.  This function modifies
    // the mat and inv matrices to reflect the transformation.
    virtual void ApplyTransform(const Matrix& tform);

protected:

    // Apply inverse transformation (world space -> canonical space)
    Ray InvertRay(Ray&);
    Vector3D InvertPt(const Vector3D&);
    Vector3D InvRotatePt(const Vector3D&);

    // Apply forward transformation (canonical space -> world space)
    Ray TransformRay(Ray&);
    Vector3D TransformPt(const Vector3D&);
    Vector3D RotatePt(const Vector3D&);

    // Transform spans from canonical space to world space.

    Matrix mat, inv;
};

// -------------------------------------------------------------------
// Algebraic surface
//	Derived from Object3D.
//	This defines a generalized, implicit surface.
// -------------------------------------------------------------------

// forward-declare Polynomial class
class Polynomial;

class Algebraic : public InverseRayObject {
public:
    Algebraic(): InverseRayObject(0) { }
    Algebraic(Surface *clr): InverseRayObject(clr, 0) { }
    Algebraic(const Quadric& q);

    virtual int NearestInt(Ray& Ray, float& t, float maxt);
    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Vector3D FindNormal(const Vector3D& intersection);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual float PtDistance(Vector3D& v);
    virtual void CountMe() const { Statistics::Objects::Algebraic++; }

    void AddTerm(double c, int i, int j, int k);
protected:
    virtual Polynomial RayToPoly(Ray& ray);
private:
    class Term {
	double c;
	int i, j, k;
    public:
	Term(double _c, int _i, int _j, int _k): 
	    c(_c), i(_i), j(_j), k(_k) { }
	friend int operator==(const Term& A, const Term& B) {
	    return (A.i == B.i && A.j == B.j && A.k == B.k);
	}
	friend Algebraic;
    };

    class TermList : public SimpleList<Term *> { };

    TermList terms;
};

// -------------------------------------------------------------------
// Torus
//	Derived from Algebraic.
// -------------------------------------------------------------------
class Torus : public Algebraic {
public:
    Torus(float a, float b, float c, Surface *clr);
    virtual void ApplyTransform(const Matrix&);
    virtual AxisAlignedBox BBox() const;
private:
    BoundingBox bbox;
};


// -------------------------------------------------------------------
// Sphere
//		Derived from Object3D.
// -------------------------------------------------------------------

class Sphere : public Object3D {
public:
    Sphere(const Vector3D& NewCenter, float NewRadius, Surface *NewColor);

    virtual int NearestInt(Ray& Ray, float& t, float maxt);
    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Vector3D FindNormal(const Vector3D& intersection);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual int IsInside(Vector3D& v);
    virtual float PtDistance(Vector3D& v);
    virtual AxisAlignedBox BBox() const;
    virtual void CountMe() const { Statistics::Objects::Sphere++; }
private:
    Vector3D center;		// location of sphere's center
    float radius;		// radius
    float radiussq;		// radius^2 (precomputed for efficiency)
};

// -------------------------------------------------------------------
// Quadric
//	These are defined by the equation:
//	Ax^2 + By^2 + Cz^2 + 2Dxy + 2Eyz + 2Fxz + 2Gx + 2Hy + 2Jz + K = 0
//
//	It is useful to reformulate this into a 4x4 3D transformation
//	matrix for some operations, such as transformations.
// -------------------------------------------------------------------
class Quadric : public Object3D {
public:
    Quadric(float a, float b, float c, float d, float e, float f,
	    float g, float h, float j, float k, Surface *NewColor);

    virtual int NearestInt(Ray& Ray, float& t, float maxt);
    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Vector3D FindNormal(const Vector3D& Intersection);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual float PtDistance(Vector3D& v);
    virtual void CountMe() const { Statistics::Objects::Quadric++; }
private:
    Matrix mat; 			  // Quadric matrix
    double A, B, C, D, E, F, G, H, J;	  // Temporaries

    void Precompute();
    void ComputeABC(Ray& ray, double *a, double *b, double *c) const;

    friend Algebraic;
};

class AlgQuadric : public Algebraic {
public:
    AlgQuadric(float a, float b, float c, float d, float e, float f,
	    float g, float h, float j, float k, Surface *NewColor);
    virtual void ApplyTransform(const Matrix& tform);
    virtual SpanList *FindAllIntersections(Ray& ray);
private:
    Quadric q;
};

// -------------------------------------------------------------------
// Ellipsoid
//	These are quadrics that can be bounded by bounding boxes.
//	
// -------------------------------------------------------------------
class Ellipsoid : public Quadric {
public:
    Ellipsoid(float Xrad, float Yrad, float Zrad, Surface *NewColor);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual void CountMe() const { Statistics::Objects::Ellipsoid++; }
    virtual AxisAlignedBox BBox() const;
private:
    float a, b, c;
    AxisAlignedBox bbox;
};

// -------------------------------------------------------------------
// Planar
//	Root of a hierarchy of planar objects (infinite plane, 
//	polygon, ring).
// -------------------------------------------------------------------
class Planar : public Object3D {
public:
    Planar(Surface *NewColor);
    Planar(float A, float B, float C, float D, Surface *NewColor);
    Planar(const Vector3D& norm, float D, Surface *NewColor); 
    virtual ~Planar() { }

    virtual int NearestInt(Ray& Ray, float& t, float maxt);
    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Vector3D FindNormal(const Vector3D& Intersection);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const = 0;
    virtual void Describe(int ind) const = 0;
    virtual float PtDistance(Vector3D& v);
    virtual void CountMe() const = 0;
protected:
    Vector3D Normal;			// a, b, c of plane equation
    float d;				// d of plane equation
    int bad;				// nonzero if normal is of
					// zero length.
};

// -------------------------------------------------------------------
// Plane
//	Infinite plane.
// -------------------------------------------------------------------
class Plane : public Planar {
public:
    Plane(float A, float B, float C, float D, Surface *NewColor);
    Plane(const Vector3D& norm, float D, Surface *NewColor): 
        Planar(norm, D, NewColor) { }

    virtual int NearestInt(Ray& Ray, float& t, float maxt);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual void CountMe() const { Statistics::Objects::Plane++; }
};

// -------------------------------------------------------------------
// Ring
//	A circular, concentric ring.
// -------------------------------------------------------------------
class Ring : public Planar {
public:
    Ring(float A, float B, float C, float D, 
	 const Vector3D& _center, float _inner, float _outer, 
	 Surface *NewColor);
    Ring(const Vector3D& norm, float D, 
	 const Vector3D& _center, float _inner, float _outer, 
	 Surface *NewColor);
    virtual int NearestInt(Ray& ray, float& t, float maxt);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual void CountMe() const { Statistics::Objects::Ring++; }
private:
    Vector3D center;
    float inner, outer;
};

// -------------------------------------------------------------------
// Polygon
//	Derived from Planar.
//	Much of the information in this class is precomputed to speed
//	intersection calculations.
// -------------------------------------------------------------------
class Polygon : public Planar {
public:
    Polygon(): Planar(0) { }
    Polygon(int NewNumVerts, Vector3D *NewVertices, Surface *NewColor);
    Polygon(const Polygon& x);

    virtual ~Polygon();
    virtual int NearestInt(Ray& Ray, float& t, float maxt);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const;
    virtual void Describe(int ind) const;
    virtual AxisAlignedBox BBox() const;
    virtual void CountMe() const { Statistics::Objects::Polygon++; }
protected:
    int NumVerts;		// Number of vertices
    Vector3D *Vertices;		// Pointer to an array of NumVerts vertices

    // Projection information
    int which;			// Which coordinate to ignore in projection
    float umin, umax;		// Bounding box of projection
    float vmin, vmax;		//

    void Precompute();		// Computes projection and some other stuff
};

// ------------------------------------------------------------------
// CSGObject hierarchy
//	CSGObject and its descendents, CSGUnion, CSGIntersection and
//	CSGDifference, implement objects for constructive solid
//	geometry.
// ------------------------------------------------------------------

class Span {
public:
    Span() { }
    Span(float _tmin, Object3D *_omin, float _tmax, Object3D *_omax)
    {
	tmin = _tmin;  omin = _omin;
	tmax = _tmax;  omax = _omax;
    }
    friend int Compare(const Span& x, const Span& y) {
	if (x.tmin < y.tmin)
	    return -1;
	else if (y.tmin < x.tmin)
	    return 1;
	else return 0;
    }
    friend int Overlap(const Span& x, const Span& y) {
	if (x.tmin > y.tmax || x.tmax < y.tmin)
	    return 0;
	else return 1;
    }

    friend class CSGObject;
    friend class World;
    friend class InverseRayObject;

    friend class AlgQuadric;

    // These functions each take a pair of SpanLists, operate
    // on them and return a new SpanList according to the
    // operation (union, intersection, difference) being
    // performed.  The SpanList arguments are destroyed.
    friend SpanList *MergeUnion(SpanList *, SpanList *);
    friend SpanList *MergeIntersection(SpanList *, SpanList *);
    friend SpanList *MergeDifference(SpanList *, SpanList *);
private:
    float tmin, tmax;
    Object3D *omin, *omax;
};

class SpanList {
public:
    SpanList() { }
    SpanList(Span& x) { spans.Insert(x); }
    void AddSpan(const Span& x) { spans.Insert(x); }
    Span ExtractMinSpan() { return spans.ExtractMin(); }
    int NumSpans() const { return spans.N(); }
private:
    Heap<Span> spans;
};

class CSGObject: public Object3D {
public:
    CSGObject() : Object3D(CSG) { left = right = 0; }
    CSGObject(Object3D *Left, Object3D *Right);

    virtual int NearestInt(Ray& Ray, float& t, float maxt);
    virtual SpanList *FindAllIntersections(Ray& ray) = 0;
    virtual Vector3D FindNormal(const Vector3D& Intersection);
    virtual void ApplyTransform(const Matrix& tform);
    virtual Object3D *Dup() const = 0;
    virtual void Describe(int ind) const = 0;
    virtual IsInside(Vector3D& v) = 0;
    virtual float PtDistance(Vector3D& v) = 0;
    virtual AxisAlignedBox BBox() const;
    virtual void CountMe() const = 0;
protected:
    Object3D *left, *right;	// Left and right nodes

    Object3D *nearest;		// Last child object intersected.
};

// -------------------------------------------------------------------
// CSGUnion
//	Derived from CSGObject.
// -------------------------------------------------------------------
class CSGUnion : public CSGObject {
public:
    CSGUnion(Object3D *Left, Object3D *Right):
	CSGObject(Left, Right) { }

    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Object3D *Dup() const;
    virtual int IsInside(Vector3D& v);
    virtual int HitAtAll(Ray& ray, float maxt);
    virtual float PtDistance(Vector3D& v);
    virtual void Describe(int ind) const;
    virtual void CountMe() const;
};

// -------------------------------------------------------------------
// CSGIntersection
//	Derived from CSGObject.
// -------------------------------------------------------------------
class CSGIntersection : public CSGObject {
public:
    CSGIntersection(Object3D *Left, Object3D *Right):
	CSGObject(Left, Right) { }

    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Object3D *Dup() const;
    virtual int IsInside(Vector3D& v);
    virtual float PtDistance(Vector3D& v);
    virtual void Describe(int ind) const;
    virtual void CountMe() const;
    virtual AxisAlignedBox BBox() const;
};

// -------------------------------------------------------------------
// CSGDifference
//	Derived from CSGObject.
// -------------------------------------------------------------------
class CSGDifference : public CSGObject {
public:
    CSGDifference(Object3D *Left, Object3D *Right) :
	CSGObject(Left, Right) { }

    virtual SpanList *FindAllIntersections(Ray& ray);
    virtual Object3D *Dup() const;
    virtual int IsInside(Vector3D& v);
    virtual float PtDistance(Vector3D& v);
    virtual void Describe(int ind) const;
    virtual void CountMe() const;
};

#ifdef ASM
    extern "C" int plane_int(const float *, const float *, float *);
#endif

Aggregate *ReadGEOFile(char *, Surface *);

