// ================================================================
// oort.h
//	Main header file for OORT.  Including this should be
//	everything you need.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ================================================================

#include <malloc.h>
#include <ctype.h>
#include <fstream.h>
#include <iomanip.h>
#include <math.h>
#include <mem.h>
#include <stdio.h>
#include <conio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <values.h>

// For Vector3D class
#include "vector.h"

// For Matrix class
#include "matrix.h"

// For Heap (binary heap-based priority queue) class template
#include "heap.h"

// For SimpleList class template
#include "list.h"

// For SimpleStack class template
#include "stack.h"

// For Statistics
#include "stats.h"

// For Object3D class and its descendents
#include "object3d.h"

// For texture-mapping
#include "mapping.h"
#include "colormap.h"
#include "noise.h"
#include "texture.h"
#include "bump.h"
#include "surface.h"

// For Light class and its descendents
#include "light.h"

// For adaptive supersampling
#include "distrib.h"

// ----------------------------------------------------------------
// GlobalNoise.
//	For now, this class encapsulates the classes that I feel
//	may need to be passed into Textures and other places that
//	use them; different solid noise functions may be useful for
//	different Textures, for instance.  But for now, it is
//	simpler to have a global solid noise function, to which the
//	user can specify the random number seed.  All textures look
//	to GlobalNoise::Noise for solid noise, and they similarly
//	look to GlobalNoise::Waves and GlobalNoise::Freqs for waves
//	and frequencies.  I'll have to break these out sometime.
// ----------------------------------------------------------------
class GlobalNoise {
public:
  static NoiseMaker *Noise;
  static WaveSource *Waves;
  static FreqSource *Freqs;
};

// -----------------------------------------------
//	Returns true if the x and y are within
//	Limits::Small of each other.
// -----------------------------------------------
inline int
LessThanEpsilon(const float& x, const float& y)
{
    float diff = x - y;
    Statistics::EpsilonTotal++;
    if (diff >= 0)
      return 0;
    Statistics::EpsilonNeeded++;
    return -diff > Limits::Small;
//  return (diff < 0) && (-diff > Limits::Small);
}

// -------------------------------------------------------------------
// Ray class.  A ray is defined by a location and a direction.
// The direction is always normalized (length == 1).
// -------------------------------------------------------------------
class Ray {
public:
    // constructors
    Ray(const Vector3D& wherefrom, const Vector3D& whichdir) {
	loc = wherefrom;
	dir = Normalize(whichdir);
	Precompute();
    }

    // -----------------------------------
    // Extrapolate ray given parameter
    // -----------------------------------
    Vector3D Extrap(float t) const {
	return loc + dir * t;
    }

    // -----------------------------------
    // Return parameter given point on ray
    // -----------------------------------
    float Interp(Vector3D& x) const {
	for (int i = 0; i < 3; i++)
	    if (fabs(dir[i]) > Limits::Small)
		return (x[i] - loc[i]) / dir[i];
	return 0;
    }

    // -----------------------------------
    // Reflects direction of reflection for the ray, 
    // given the normal to the surface.
    // -----------------------------------
    Vector3D ReflectRay(const Vector3D& N) const {
	return N*2*DotProd(N, -dir) + dir;
    }

    void ApplyTransform(Matrix& tform) {
	loc = tform * loc;
	dir = RotateOnly(tform, dir);
	Precompute();
    }
private:
    Vector3D loc, dir;		// Describes ray
    Vector3D ll, ld, dd;	// Various vectors with corresponding
				// elements multiplied together.
    void Precompute() {
	ll = loc * loc;
	ld = loc * dir;
	dd = dir * dir;
    }

    friend class World;
    friend class AxisAlignedBox;
    friend class Object3D;
    friend class Sphere;
    friend class BoundingBox;
    friend class Planar;
    friend class Polygon;
    friend class Quadric;
    friend class Ellipsoid;
    friend class Algebraic;

    friend class HallSurface;
    friend class DistributedHall;
};

// ---------------------------------------------------------
// KayKajiyaNode
//	Contains the information needed to traverse the 
//	bounding volume hierarchy.  It is named after the 
//	type of bounding volume traversal OORT uses (invented 
//	by Kay and Kajiya).  KayKajiyaNodes exist to get 
//	Insert()'ed and ExtractMin()'ed from the heap used 
//	to traverse the hierarchy.
// ---------------------------------------------------------
class KayKajiyaNode {
public:
    // -------------------------------------------------------
    // Constructors.
    // -------------------------------------------------------
    KayKajiyaNode() { }
    KayKajiyaNode(Object3D *NewObj, float& NewT)
    {
	obj = NewObj;
	t = NewT;
    }

    // -------------------------------------------------------
    // Used by Heap<KayKajiyaNode> to order the nodes.
    // -------------------------------------------------------
    friend int Compare (KayKajiyaNode *x, KayKajiyaNode *y) {
	if (x->t < y->t)
	    return -1;
	else if (x->t > y->t)
	    return 1;
	else return 0;
    }
private:
    float t;		// Intersection parameter
    Object3D *obj;	// Pointer to object intersected

    friend World;
};

// -------------------------------------------------------------------
// Support functions
// -------------------------------------------------------------------

// -------------------------------------------------------------------
// Report (in main.cpp) reports ray tracer statistics
// -------------------------------------------------------------------
void Report(ostream&);

// -------------------------------------------------------------------
// Usage (in main.cpp) explains the command-line options
// available to the user.
// -------------------------------------------------------------------
void Usage();

// -------------------------------------------------------------------
// Indents to cout the number of spaces specified.
// This routine is located in main.cpp, for lack of a better place.
// -------------------------------------------------------------------
void indent(int ind);

// ---------------------------------------------------------
// RandomValue
//	Returns a random floating-point value between the two 
//	values.  Range is inclusive; the function should 
//	occasionally return exactly a or b.
// ---------------------------------------------------------
inline float
RandomValue(float a, float b)
{
    float range = fabs(a - b);
    return ((float) rand() / RAND_MAX) * range + ((a < b) ? a : b);
}

// ---------------------------------------------------------
// Swaps two values.
// ---------------------------------------------------------
template<class T>
inline void 
Swap(T& x, T& y)
{
    T temp = x;
    x = y;
    y = temp;
}

// ---------------------------------------------------------
// InRange
//	Returns nonzero if min <= candidate <= max.
// ---------------------------------------------------------
template<class T>
inline int 
InRange(T min, T max, T candidate)
{
    return candidate >= min && candidate <= max;
}

