// ===================================================================
// texture.h
//	Header file for texture implementations.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#ifndef __TEXTURE__

#include <iostream.h>

#define __TEXTURE__

// ---------------------------------------------------------
// Texture
//	Implements various methods of taking intersection 
//	points on objects and turning them into colors.
// ---------------------------------------------------------

class Texture {
public:
    Texture() { reference = IdentityMatrix(); }
    Texture(const Matrix& ref): reference(ref) { }
    virtual ~Texture() { }

    virtual void ApplyTransform(const Matrix& tform);
    virtual void PreMulTransform(const Matrix& tform);

    virtual RGBColor GetColor(const Vector3D&) const = 0;
    virtual Texture *Dup() const = 0;

protected:
    Matrix reference;	// Matrix to apply to intersection points

    virtual Vector3D PreprocessVector(const Vector3D& x) const;
};

// ---------------------------------------------------------
// TextureMapping
//	Implements a Texture that applies an inverse mapping
//	after applying the reference matrix.
// ---------------------------------------------------------
class TextureMapping : public Texture {
public:
    TextureMapping();
    TextureMapping(const Matrix &ref, Mapping *map = 0);

    virtual ~TextureMapping();
    virtual Texture *Dup() const = 0;
protected:
    Mapping *mapping;	// Mapping to apply while preprocessing
			// the vector
    virtual Vector3D PreprocessVector(const Vector3D& x) const;
};

// ---------------------------------------------------------
// PureColor
//	Special texture that just returns an RGB triple 
//	regardless of the intersection point.
// ---------------------------------------------------------

class PureColor : public Texture {
public:
    PureColor() { }
    PureColor(const RGBColor& clr): Color(clr) { }
    PureColor(float r, float g, float b): Color(RGBColor(r, g, b)) { }

    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
    virtual void ApplyTransform(const Matrix& tform) { }
private:
    RGBColor Color;	  // Color associated with this texture
};

// ---------------------------------------------------------
// Checkerboard.
// ---------------------------------------------------------
class Checkerboard : public TextureMapping {
public:
    Checkerboard(float XSize, float YSize,
		 Texture *Text1, Texture *Text2,
		 const Matrix& ref, Mapping *map = 0);
    RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    float xsize, ysize;
    Texture *text1, *text2;
};

// ---------------------------------------------------------
// ImageFile
//	Provides a uniform way to deal with various image 
//	file formats.  Only RAW files are supported right
//	now (for memory reasons--the RAW file can stay on
//	disk and you can still access the pixels).
// ---------------------------------------------------------

class ImageFile {
public:
    ImageFile() { }
    virtual ~ImageFile() { }
    virtual RGBColor GetPixel(int x, int y) = 0;
    virtual int Width() { return width; }
    virtual int Height() { return height; }
protected:
    int width, height;
};

// ---------------------------------------------------------
// RAWFile
//	Implements an ImageFile interface for ray tracer
//	output files.
// ---------------------------------------------------------
class RAWFile : public ImageFile {
public:
    RAWFile(char *filename);
    virtual ~RAWFile();
    RGBColor GetPixel(int x, int y);
private:
    FILE *fptr;			// Pointer to the open image file

    int cachedscanl;		// Index of the upper scanline cached
    unsigned char *scanl[2];	// Scanline cached

    void Seek(int y) {
	fseek(fptr, (long) y*(3* (long) width+2)+4, SEEK_SET);
    }

};

// ---------------------------------------------------------
// RAWFileInMemory
//	RAWFile, except the entire image is read into
//	memory on creation.  This is much faster, but much
//	more memory intensive as well.
// ---------------------------------------------------------
class RAWFileInMemory : public ImageFile {
public:
    RAWFileInMemory(char *filename);
    virtual ~RAWFileInMemory();
    virtual RGBColor GetPixel(int x, int y);
private:
    unsigned char **scanl;	// Array of pointers to scanline
};

ImageFile *OpenFile(char *filename);

// ---------------------------------------------------------
// ImageMap
//	Maps an image onto an object.  Bilinearly 
//	interpolating the colors is optional.
// ---------------------------------------------------------

class ImageMap : public TextureMapping {
public:
    ImageMap(const Matrix& ref, Mapping *map, int Interp, int Once, char *filename, float xscale = 1, float yscale = 1);
    ImageMap(const Matrix& ref, Mapping *map, int Interp, int Once, ImageFile *file, float xscale = 1, float yscale = 1);
    ImageMap(char *filename);

    virtual ~ImageMap() { }
    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    int interp; 	// Boolean--if true, bilinear interpolation
			// is performed.
    int once;		// Boolean--if true, the image is mapped once
			// rather than tiled.
    ImageFile *file;	// Image to map
    Vector3D scale;	// Scale by this vector after mapping.
};

// ---------------------------------------------------------
// SolidTexture
//	This class encapsulates the various solid textures
//	such as Bozo, Wood, and Marble.
// ---------------------------------------------------------
class SolidTexture : public Texture {
public:
    SolidTexture() { }
    SolidTexture(const Matrix& ref, ColorMap *CMap);
    virtual RGBColor GetColor(const Vector3D&) const = 0;
    virtual Texture *Dup() const = 0;
protected:
    ColorMap *cmap;
};

// ---------------------------------------------------------
// Bozo
//	A solid texture that maps the solid noise function 
//	directly into a color map.
// ---------------------------------------------------------
class Bozo : public SolidTexture {
public:
    Bozo(const Matrix& Ref, float Turbulence, ColorMap *CMap);
    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    float turbulence;
};

// ---------------------------------------------------------
// Wood
//	Solid texture to simulate wood.
// ---------------------------------------------------------
class Wood : public SolidTexture {
public:
    Wood(const Matrix& Ref, float Turbulence, ColorMap *CMap);
    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    float turbulence;
};

// ---------------------------------------------------------
// Marble
//	Solid texture to simulate marble.
// ---------------------------------------------------------
class Marble : public SolidTexture {
public:
    Marble(const Matrix& Ref, float Turbulence, ColorMap *CMap);
    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    float turbulence;
};

// ---------------------------------------------------------
// Gradient texture.
// ---------------------------------------------------------

class Gradient : public SolidTexture {
public:
    Gradient(const Matrix& Ref, float Turbulence, ColorMap *CMap);
    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    float turbulence;
};

// ---------------------------------------------------------
// WoodUpstill
//	Wood texture derived from Listing 16.15 in Steve 
//	Upstill's book, _The RenderMan Companion_.
//	(Addison-Wesley, 1990).
// ---------------------------------------------------------
class WoodUpstill : public SolidTexture {
public:
    WoodUpstill(const Matrix& ref = IdentityMatrix(),
		float _ringscale = 10, 
		const RGBColor& _light = RGBColor(0.3, 0.12, 0.03), 
		const RGBColor& _dark = RGBColor(0.05, 0.01, 0.005));
    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    float ringscale;
    RGBColor light, dark;
};

// ---------------------------------------------------------
// Granite texture.
//	Also derived from Upstill.
// ---------------------------------------------------------
class Granite : public SolidTexture {
public:
    Granite(const RGBColor& _K,
	    const Matrix& _reference = IdentityMatrix());
    virtual RGBColor GetColor(const Vector3D&) const;
    virtual Texture *Dup() const;
private:
    RGBColor K;
};

// ---------------------------------------------------------
// Some support functions for textures.
// ---------------------------------------------------------

// ---------------------------------------------------------
// posmod()
//	Returns ((x < 0) ? y - x%y : x%y)/y.  
//	Guaranteed to range [0, 1].
// ---------------------------------------------------------
inline float
posmod(float x, float y)
{
    float ret = fmod(x, y);
    if (ret < 0)
	ret += y;
    return ret / y;
}

// ---------------------------------------------------------
// cycloidal()
//	Maps the fractional portion of x to a sinusoid.
// ---------------------------------------------------------
inline float 
cycloidal(float x) {
    float temp = fmod(x, 1);
    if (temp < 0)
	temp += 1;
    return sin(temp * 2 * M_PI);
}

// ---------------------------------------------------------
// triangle_wave() 
//	Maps the fractional portion of x to a triangle wave.
// ---------------------------------------------------------
inline float 
triangle_wave(float x) {
    float offset = fmod(x, 1);
    if (offset < 0)
	offset += 1;
    if (offset > 0.5)
	offset = 1 - offset;
    return offset + offset;
}

// ---------------------------------------------------------
// MixColors
//	Linearly combines two colors according to a (a is
//	in the range [0, 1]).
// ---------------------------------------------------------
inline RGBColor
MixColors(const RGBColor& clr0, const RGBColor& clr1, float a)
{
    return RGBColor(a) * clr0 + RGBColor(1 - a) * clr1;
}

// ---------------------------------------------------------
// Hermite
//	Hermite curve interpolation function.
//	Returns 0 if val < min.  Returns 1 if val > max.
//	Otherwise, returns an interpolated value between 0 and 1.
// ---------------------------------------------------------
inline float 
Hermite(float min, float max, float val) 
{
    if (val < min)
	return 0;
    if (val > max)
	return 1;
    float t = (val - min) / (max - min);
    return t*t*(-3 + 2*t) + 1;
}

// ---------------------------------------------------------
// Texture functions implemented in-line.
// ---------------------------------------------------------
inline Vector3D
Texture::PreprocessVector(const Vector3D& x) const
{
    return reference * Vector3D(x);
}

// ---------------------------------------------------------
// ---------------------------------------------------------
inline void
Texture::ApplyTransform(const Matrix& tform)
{
    reference *= tform;
}

inline void
Texture::PreMulTransform(const Matrix& tform)
{
    reference = tform * reference;
}

// ---------------------------------------------------------
// TextureMapping functions implemented in-line.
// ---------------------------------------------------------
inline
TextureMapping::TextureMapping()
{
    mapping = 0;
}

inline
TextureMapping::TextureMapping(const Matrix& ref, Mapping *map):
    Texture(ref)
{
    mapping = map;
}

inline
TextureMapping::~TextureMapping()
{
    delete mapping;
}

inline Vector3D
TextureMapping::PreprocessVector(const Vector3D& x) const
{
    if (mapping)
	return mapping->Map3DTo2D(Texture::PreprocessVector(x));
    else
	return Texture::PreprocessVector(x);
}

// ---------------------------------------------------------
// PureColor functions implemented in-line.
// ---------------------------------------------------------
inline RGBColor
PureColor::GetColor(const Vector3D& x) const
{
    return Color;
}

inline Texture *
PureColor::Dup() const
{
    return new PureColor(Color);
}

#endif
