// ===================================================================
// vector.h
//	Header file for Vector3D class in OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#ifndef __VECTOR__

#define __VECTOR__

// Forward-declare some other classes.
class Matrix;
class ostream;

class Vector3D {

public:
    float x, y, z;

    // constructors
    Vector3D() { }
    Vector3D(float X, float Y, float Z) { x = X; y = Y; z = Z; }
    Vector3D(float X) { x = y = z = X; }
    Vector3D(const Vector3D& v) { x = v.x; y = v.y; z = v.z; }

    // Functions to get at the vector components
    float& operator[] (int inx) {
	if (inx == 0)      return x;
	else if (inx == 1) return y;
	else	       	   return z;
    }
    const float& operator[] (int inx) const {
	if (inx == 0)      return x;
	else if (inx == 1) return y;
	else	       	   return z;
    }
    void ExtractVerts(float *px, float *py, int which) const;

    friend float Magnitude(const Vector3D& v);
    friend Vector3D Normalize(const Vector3D& A);

    // Rotate a normal vector.
    friend Vector3D PlaneRotate(const Matrix&, const Vector3D&);

    // Unary operators
    Vector3D operator+ () const;
    Vector3D operator- () const;

    // Assignment operators
    Vector3D& operator+= (const Vector3D& A);
    Vector3D& operator-= (const Vector3D& A);
    Vector3D& operator*= (const Vector3D& A);
    Vector3D& operator*= (float A);
    Vector3D& operator/= (float A);

    // Binary operators
    friend Vector3D operator+ (const Vector3D& A, const Vector3D& B);
    friend Vector3D operator- (const Vector3D& A, const Vector3D& B);
    friend Vector3D operator* (const Vector3D& A, const Vector3D& B);
    friend Vector3D operator* (const Vector3D& A, float B);
    friend Vector3D operator* (float A, const Vector3D& B);
    friend Vector3D operator* (const Matrix&, const Vector3D&);
    friend Vector3D operator/ (const Vector3D& A, const Vector3D& B);

    friend int operator< (const Vector3D& A, const Vector3D& B);

    friend Vector3D operator/ (const Vector3D& A, float B);
    friend int operator== (const Vector3D& A, const Vector3D& B);
    friend float DotProd(const Vector3D& A, const Vector3D& B);
    friend Vector3D CrossProd (const Vector3D& A, const Vector3D& B);

    friend ostream& operator<< (const ostream& s, const Vector3D& A);

    friend void Minimize(Vector3D& min, const Vector3D& Candidate);
    friend void Maximize(Vector3D& max, const Vector3D& Candidate);
};

class RGBColor : public Vector3D {
public:
    RGBColor() { }
    RGBColor(float X, float Y, float Z) { x = X; y = Y; z = Z; }
    RGBColor(float X) { x = y = z = X; }
    RGBColor(const Vector3D& v) { x = v.x; y = v.y; z = v.z; }
    friend RGBColor NormalizeColor(const RGBColor& x);
};

inline float Magnitude(const Vector3D& v) 
{
    return sqrt(v.x*v.x + v.y*v.y + v.z*v.z);
}

inline Vector3D Normalize(const Vector3D& A)
{
    return A / Magnitude(A);
}

inline float DotProd (const Vector3D& A, const Vector3D& B)
{
    return A.x * B.x + A.y * B.y + A.z * B.z;
}

inline Vector3D Vector3D::operator+ () const
{
    return *this;
}

inline Vector3D Vector3D::operator- () const
{
    return Vector3D(-x, -y, -z);
}

inline Vector3D& Vector3D::operator+= (const Vector3D& A)
{
    x += A.x;  y += A.y;  z += A.z;
    return *this;
}

inline Vector3D& Vector3D::operator-= (const Vector3D& A)
{
    x -= A.x;  y -= A.y;  z -= A.z;
    return *this;
}

inline Vector3D& Vector3D::operator*= (float A)
{
    x *= A;  y *= A;  z *= A;
    return *this;
}

inline Vector3D& Vector3D::operator/= (float A)
{
    x /= A;  y /= A;  z /= A;
    return *this;
}

inline Vector3D& Vector3D::operator*= (const Vector3D& A)
{
    x *= A.x;  y *= A.y;  z *= A.z;
    return *this;
}

inline Vector3D operator+ (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x + B.x, A.y + B.y, A.z + B.z);
}

inline Vector3D operator- (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x - B.x, A.y - B.y, A.z - B.z);
}

inline Vector3D operator* (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x * B.x, A.y * B.y, A.z * B.z);
}

inline Vector3D operator* (const Vector3D& A, float B)
{
    return Vector3D(A.x * B, A.y * B, A.z * B);
}

inline Vector3D operator* (float A, const Vector3D& B)
{
    return B * A;
}

inline Vector3D operator/ (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.x / B.x, A.y / B.y, A.z / B.z);
}

inline Vector3D operator/ (const Vector3D& A, float B)
{
    return Vector3D(A.x / B, A.y / B, A.z / B);
}

inline int operator< (const Vector3D& A, const Vector3D& B)
{
    return A.x < B.x && A.y < B.y && A.z < B.z;
}

// Might replace floating-point == with comparisons of
// magnitudes, if needed.
inline int operator== (const Vector3D& A, const Vector3D& B)
{
    return (A.x == B.x) && (A.y == B.y) && (A.z == B.z);
}

inline Vector3D CrossProd (const Vector3D& A, const Vector3D& B)
{
    return Vector3D(A.y * B.z - A.z * B.y,
		    A.z * B.x - A.x * B.z,
		    A.x * B.y - A.y * B.x);
}

// Compute a normalized version of a color (same hue, but less bright).
inline RGBColor NormalizeColor(const RGBColor& clr)
{
    if (clr.x > 1.0 || clr.y > 1.0 || clr.z > 1.0) {
	float max = clr.x;
	if (clr.y > max)
	    max = clr.y;
	if (clr.z > max)
	    max = clr.z;
	if (max != 0.0)
	    return clr / max;
    }
    return clr;
}

#endif
