// ===================================================================
// mapping.cpp
//	Implementations of inverse mappings.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

// ---------------------------------------------------------
// CircularMapping
//	Maps onto a circle centered about the origin.
// ---------------------------------------------------------

CircularMapping::CircularMapping(float _radius)
{
    radius = _radius;
}

Mapping *
CircularMapping::Dup() const
{
    return new CircularMapping(radius);
}

Vector3D
CircularMapping::Map3DTo2D(const Vector3D& v) const
{
    float mag = posmod(hypot(v.x, v.y), radius);
    float u = acos(posmod(v.x, radius)) / (2 * M_PI);
    return Vector3D((v.y < 0) ? 1 - u : u,
		    mag,
		    0);
}

// ---------------------------------------------------------
// ConicalMapping
//	Maps points onto a cone centered about the Y axis.
// ---------------------------------------------------------
ConicalMapping::ConicalMapping(float _rad0, float _radh, float _height)
{
    rad0 = _rad0;
    radh = _radh;
    height = _height;
}

Mapping *
ConicalMapping::Dup() const
{
    return new ConicalMapping(rad0, radh, height);
}

Vector3D
ConicalMapping::Map3DTo2D(const Vector3D& v) const
{
    float tempv = posmod(v.y, height);
    float denom = rad0 + (radh - rad0) * tempv;

    if (fabs(denom) < Limits::Small)
        return Vector3D(0, tempv, 0);

    float u = acos(fmod(v.x, denom)) / (2 * M_PI);

    return Vector3D( (v.z < 0) ? 1 - u : u,
		     tempv,
		     0);
}

// ---------------------------------------------------------
// CylindricalMapping
//	Maps points onto a cylinder centered about the Y axis.
// ---------------------------------------------------------

CylindricalMapping::CylindricalMapping(float _radius, float _height)
{
    radius = _radius;
    height = _height;
}

Mapping *
CylindricalMapping::Dup() const
{
    return new CylindricalMapping(radius, height);
}

Vector3D
CylindricalMapping::Map3DTo2D(const Vector3D& v) const
{
    float u = acos(v.x / Magnitude(v)) / (2 * M_PI);
    return Vector3D( (v.y < 0) ? 1 - u : u,
		     posmod(v.y, height),
		     0);
/*
    float u = fmod(v.x, radius);
    u = acos(u / radius) / (2 * M_PI);
    return Vector( (v.y < 0) ? 1 - u : u,
		   posmod(v.y, height),
		   0);
*/
}

// ---------------------------------------------------------
// ProjectionMapping
//	Projects points onto the XY coordinate plane.
// ---------------------------------------------------------

Mapping *
ProjectionMapping::Dup() const
{
    return new ProjectionMapping;
}

Vector3D
ProjectionMapping::Map3DTo2D(const Vector3D& v) const
{
    return Vector3D(v.x, v.y, 0);
}

// ---------------------------------------------------------
// QuadrilateralMapping
//	Maps onto an arbitrary quadrilateral.
// ---------------------------------------------------------
QuadrilateralMapping::QuadrilateralMapping(const Vector3D *Pts)
{
    for (int i = 0; i < 4; i++)
	pts[i] = Pts[i];
    Vector3D Pa = pts[0] - pts[3] + pts[2] - pts[1];
    Vector3D Pb = pts[3] - pts[0];
    Vector3D Pc = pts[1] - pts[0];
    Vector3D Pd = pts[0];
    Vector3D Pn = -Normalize(CrossProd(pts[2] - pts[1],
				     pts[0] - pts[1]));
    Na = CrossProd(Pa, Pn);
    Nb = CrossProd(Pb, Pn);
    Nc = CrossProd(Pc, Pn);
    Nd = CrossProd(Pd, Pn);
    Du0 = DotProd(Nc, Pd);
    Du1 = DotProd(Na, Pd) + DotProd(Nc, Pb);
    Du2 = DotProd(Na, Pb);
    Dv0 = DotProd(Nb, Pd);
    Dv1 = DotProd(Na, Pd) + DotProd(Nb, Pc);
    Dv2 = DotProd(Na, Pc);
    uaxespar = fabs(Du2) < Limits::Small;
    if (! uaxespar) {
	Qux = Na / (2 * Du2);
	Dux = -Du1 / (2 * Du2);
	Quy = -Nc / Du2;
	Duy = Du0 / Du2;
    }
    vaxespar = fabs(Dv2) < Limits::Small;
    if (! vaxespar) {
	Qvx = Na / (2 * Dv2);
	Dvx = -Dv1 / (2 * Dv2);
	Qvy = -Nb / Dv2;
	Dvy = Dv0 / Dv2;
    }
}

Mapping *
QuadrilateralMapping::Dup() const
{
    return new QuadrilateralMapping(pts);
}

Vector3D
QuadrilateralMapping::Map3DTo2D(const Vector3D& x) const
{
    float u, v;
    if (uaxespar)
	u = (DotProd(Nc, x) - Du0) / (Du1 - DotProd(Na, x));
    else {
	float Ka = Dux + DotProd(Qux, x);
	float Kb = Duy + DotProd(Quy, x);
	if (Kb <= Ka*Ka) {
	    float u0 = Ka - sqrt(Ka*Ka - Kb);
	    float u1 = Ka + sqrt(Ka*Ka - Kb);
	    u = u0;
	    if (! InRange((float) 0, (float) 1, u0))
		u = u1;
	}
	else u = 0;
    }
    if (vaxespar)
	v = (DotProd(Nb, x) - Dv0) / (Dv1 - DotProd(Na, x));
    else {
	float Ka = Dvx + DotProd(Qvx, x);
	float Kb = Dvy + DotProd(Qvy, x);
	if (Kb <= Ka*Ka) {
	    float v0 = Ka - sqrt(Ka*Ka - Kb);
	    float v1 = Ka + sqrt(Ka*Ka - Kb);
	    v = v0;
	    if (! InRange((float) 0, (float) 1, v0))
		v = v1;
	}
	else v = 0;
    }
    return Vector3D(u, v, 0);
}

// ---------------------------------------------------------
// SphericalMapping
//	Maps onto a sphere centered about any point.
// ---------------------------------------------------------

SphericalMapping::SphericalMapping(Vector3D _center)
{
    center = _center;
}

Mapping *
SphericalMapping::Dup() const
{
    return new SphericalMapping(center);
}

Vector3D
SphericalMapping::Map3DTo2D(const Vector3D& v) const
{
    Vector3D unit = Normalize(v - center);
    float phi = acos(unit.y);
    float theta = unit.x / sqrt(1 - unit.y * unit.y);
    if (fabs(theta) > 1)
	return Vector3D(0, phi / M_PI, 0);
    theta = acos(theta) / (2 * M_PI);
    if (unit.z > 0)
	return Vector3D(theta, phi / M_PI, 0);
    else
	return Vector3D(1 - theta, phi / M_PI, 0);
}
