// ===================================================================
// plane.cpp
//	Plane support routines for OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

Planar::Planar(Surface *NewColor): Object3D(NewColor, 0)
{
    bad = 1;
}

Planar::Planar(float A, float B, float C, float D, Surface *NewColor):
    Object3D(NewColor, 0)
{
    Normal = Vector3D(A, B, C);
    d = D;
    float mag = Magnitude(Normal);
    bad = fabs(mag) < Limits::Small;
    if (! bad) {
	Normal /= mag;
	d /= mag;
    }
}

Planar::Planar(const Vector3D& norm, float D, Surface *NewColor):
    Object3D(NewColor, 0)
{
    Normal = norm;
    d = D;
    float mag = Magnitude(Normal);
    bad = fabs(mag) < Limits::Small;
    if (! bad) {
	Normal /= mag;
	d /= mag;
    }
}

int
Planar::NearestInt(Ray& ray, float& t, float maxt)
{
    if (bad)
	return 0;

    float dot = DotProd(Normal, ray.dir);

    // Watch for near-zero denominator
    if (fabs(dot) < Limits::Small) {
	return 0;
    }

    t = (-d - DotProd(Normal, ray.loc)) / dot;

    return t > Limits::Threshold && t < maxt;
}

SpanList *
Planar::FindAllIntersections(Ray& ray)
{
    float dot = DotProd(Normal, ray.dir);

    if (fabs(dot) == 0) {
	// Ray is parallel; if ray is in the plane, it intersects
	// the plane infinitely far away in both directions; otherwise
	// it intersects the ray not at all, because it is parallel.
	if (DotProd(Normal, ray.loc) + d < Limits::Small) {
	    SpanList *ret = new SpanList(Span(-Limits::Infinity, this,
					      Limits::Infinity, this));
	    return ret;
	}
	else return 0;
    }

    float t = (-d - DotProd(Normal, ray.loc)) / dot;

    if (dot < 0)
	return new SpanList(Span(t, this, Limits::Infinity, this));
    else {
	if (t < Limits::Threshold)
	    return 0;
	return new SpanList(Span(-Limits::Infinity, this, t, this));
    }
}

Vector3D
Planar::FindNormal(const Vector3D& v)
{
    return Normal;
}

void
Planar::ApplyTransform(const Matrix& tform)
{
    Vector3D v = Normal * -d;
    Normal = Normalize(RotateOnly(tform, Normal));
    d = -DotProd(tform * v, Normal);
}

float
Planar::PtDistance(Vector3D& v)
{
    return DotProd(v, Normal) + d;
}

// ===================================================================
// Plane implementations
// ===================================================================

Plane::Plane(float A, float B, float C, float D, Surface *NewColor):
    Planar(A, B, C, D, NewColor)
{ }

int
Plane::NearestInt(Ray& ray, float& t, float maxt)
{
    Statistics::Intersections::Plane += 1;
    return Planar::NearestInt(ray, t, maxt);
}


Object3D *
Plane::Dup() const
{
    return new Plane(Normal.x, Normal.y, Normal.z, d, surf);
}

void
Plane::Describe(int ind) const
{
    indent(ind);
    cout << "Plane: " << setprecision(3) << Normal.x << "x + ";
    cout << Normal.y << "y + " << Normal.z << "z + " << d << " = 0\n";
}

// ===================================================================
// Ring implementations
// ===================================================================

Ring::Ring(float A, float B, float C, float D, 
	   const Vector3D& _center, float _inner, float _outer, 
	   Surface *surf):
    Planar(A, B, C, D, surf), center(_center), inner(_inner), outer(_outer)
{
}

Ring::Ring(const Vector3D& norm, float D, 
	   const Vector3D& _center, float _inner, float _outer, 
	   Surface *surf):
    Planar(norm, D, surf), center(_center), inner(_inner), outer(_outer)
{
}

int
Ring::NearestInt(Ray& ray, float& t, float maxt)
{
    // Count ring intersection
    Statistics::Intersections::Ring += 1;

    // If we miss the plane, we miss the ring.
    if (Planar::NearestInt(ray, t, maxt)) {
	// If we hit the plane, we have to be between the two radii
	// to hit the ring.
	float dist = Magnitude(ray.Extrap(t) - center);
	return dist >= inner && dist <= outer;
    }
    return 0;
}

void
Ring::ApplyTransform(const Matrix& tform)
{
    Planar::ApplyTransform(tform);
    center = tform * center;
}

Object3D *
Ring::Dup() const
{
    return new Ring(Normal, d, center, inner, outer, surf);
}

void
Ring::Describe(int ind) const
{
    indent(ind);
    cout << "Ring: plane normal " << Normal << ", dist " << d;
    cout << ", inner rad " << inner << ", outer rad " << outer << '\n';
}

