// ===================================================================
// polygon.cpp
//	Polygon support routines for OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

// Polygon constructor
Polygon::Polygon(int NewNumVerts,	// number of vertices
		 Vector3D *NewVertices,	// vertices
		 Surface *NewSurface):  // polygon surface
    Planar(NewSurface)
{
    NumVerts = NewNumVerts;		  // copy given info
    Vertices = new Vector3D[NumVerts];
    for (int i = 0; i < NumVerts; i++)
	Vertices[i] = NewVertices[i];
    Precompute();			  // precompute
}

Polygon::Polygon(const Polygon& x): Planar(x.surf)
{
    Vertices = new Vector3D[NumVerts = x.NumVerts];
    for (int i = 0; i < NumVerts; i++)
	Vertices[i] = x.Vertices[i];
    Precompute();
}

// Polygon destructor
Polygon::~Polygon()
{
    delete[] Vertices;
}

// Precompute various parameters about the polygon:
//	The plane equation (including normal)
//	which component of the normal is dominant (needed
//		for intersection routine)
//	Bounding rectangle of projection (used by intersection
//		routine to eliminate intersections)
//	Overall bounding box of polygon (used by intersection
//		routine to intersect with ray bounding boxes)
void 
Polygon::Precompute()
{
    int i;			  // Index variable
    float max;			  // Maximum component of normal
    Vector3D minvert, maxvert;	  // Bounds of polygon

    // Compute the plane equation of the polygon.
    Vector3D xprod = CrossProd(Vertices[2] - Vertices[1],
			       Vertices[0] - Vertices[1]);
    if (fabs(Magnitude(xprod)) > Limits::Small) {
	bad = 0;
	Normal = Normalize(xprod);
    }
    else {
	bad = 1;
	return;
    }
    d = -DotProd(Normal, Vertices[0]);

    // Precompute which.  which is 0 if the normal is dominant
    // in the X direction, 1 if the Y direction, 2 if the Z direction.

    which = 0;			  // Assume we'll ignore X for now
    max = fabs(Normal.x);
    if (fabs(Normal.y) > max) {   // Check to see if we ignore Y
	max = fabs(Normal.y);	  // Yes, ignore Y instead
	which = 1;
    }
    if (fabs(Normal.z) > max)	  // Check to see if we ignore Z
	which = 2;		  // Yep

    // Compute bounding rectangle of projection.  This is fairly
    // successful at rejecting nonintersections.
    Vertices[0].ExtractVerts(&umin, &vmin, which);
    umax = umin;
    vmax = vmin;
    for (i = 1; i < NumVerts; i++) {
	float tempx, tempy;

	Vertices[i].ExtractVerts(&tempx, &tempy, which);
	if (tempx < umin)
	    umin = tempx;
	else if (tempx > umax)
	    umax = tempx;
	if (tempy < vmin)
	    vmin = tempy;
	else if (tempy > vmax)
	    vmax = tempy;
    }
}

#ifdef ASM
extern "C" int int_lineseg(float, float, float, float, float, float);
#else
// int_lineseg returns 1 if the given line segment intersects a 2D
// ray travelling in the positive X direction.	This is used in the
// Jordan curve computation for polygon intersection.
int
int_lineseg(float px, float py, float u1, float v1, float u2, float v2)
{
    float t;
    float ydiff;

    u1 -= px; u2 -= px; 	  // translate line
    v1 -= py; v2 -= py;

    if ((v1 > 0 && v2 > 0) ||
	(v1 < 0 && v2 < 0) ||
	(u1 < 0 && u2 < 0))
	 return 0;

    if (u1 > 0 && u2 > 0)
	return 1;

    ydiff = v2 - v1;
    if (fabs(ydiff) < Limits::Small) {	  // denominator near 0
	if (((fabs(v1) > Limits::Small) ||
	    (u1 > 0) || (u2 > 0)))
	    return 0;
	return 1;
    }

    t = -v1 / ydiff;		  // Compute parameter

    return (u1 + t * (u2 - u1)) > 0;
}
#endif

// Polygon intersection routine.
int
Polygon::NearestInt(Ray& ray, float& t, float maxt)
{
    int count;
    float u, v, u1, v1, u2, v2;
    int i;

    Statistics::Intersections::Polygon++;
    if (Planar::NearestInt(ray, t, maxt)) {
	// Project the intersection point onto the coordinate plane
	// specified by which.
	ray.Extrap(t).ExtractVerts(&u, &v, which);

	// Reject intersection if outside projected bounding rectangle
	if (u < umin || u > umax || v < vmin || v > vmax)
	    return 0;

	// We're stuck with the Jordan curve computation.  Count number
	// of intersections between the line segments the polygon comprises
	// with a ray originating at the point of intersection and
	// travelling in the positive X direction.

	count = 0;
	Vertices[NumVerts - 1].ExtractVerts(&u1, &v1, which);

	for (i = 0; i < NumVerts; i++) {
	    Vertices[i].ExtractVerts(&u2, &v2, which);
	    count += int_lineseg(u, v, u1, v1, u2, v2) != 0;

	    u1 = u2;
	    v1 = v2;
	}

	// We hit polygon if number of intersections is odd.
	return count & 1;
    }
    return 0;
}

// Apply transform to a polygon: transform each vertex, then
// re-precompute all the precomputed information.
void
Polygon::ApplyTransform(const Matrix& tform)
{
    for (int i = 0; i < NumVerts; i++)
	Vertices[i] = tform * Vertices[i];
    Precompute();
}

// Duplicate a polygon.
Object3D *
Polygon::Dup() const
{
    return new Polygon(NumVerts, Vertices, surf);
}

// Print out vertices of polygon.
void
Polygon::Describe(int ind) const
{
    int i;

    indent(ind);
    cout << "Polygon.  Vertices: ";
    for (i = 0; i < NumVerts; i++) {
	cout << Vertices[i];
	if (i < NumVerts-1)
	    cout << ", ";
    }
    cout << '\n';
}

AxisAlignedBox
Polygon::BBox() const
{
    int i;
    Vector3D min = Vertices[0];
    Vector3D max = Vertices[0];

    for (i = 1; i < NumVerts; i++) {
	Minimize(min, Vertices[i]);
	Maximize(max, Vertices[i]);
    }
    // Make sure the bounding box isn't flat.
    Vector3D diff = min - max;
    for (i = 0; i < 3; i++)
	if (fabs(diff[i]) < Limits::Threshold) {
	    min[i] -= Limits::Threshold;
	    max[i] += Limits::Threshold;
	}
    return AxisAlignedBox(min, max);
}
