// ===================================================================
// quadric.cpp
//	Quadric support routines for OORT.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

// Quadric constructor.  For now, we just take the equation and color
// and plug them into the matrix.  Later, more sophisticated
// methods of making quadrics such as Ellipsoids may be added.
Quadric::Quadric(float a, float b, float c, float d, float e, float f,
		 float g, float h, float j, float k, Surface *clr) :
		 Object3D (clr, 0)
{
    mat = QuadricMatrix(a, b, c, d, e, f, g, h, j, k);
    Precompute();
}

void
Quadric::ComputeABC(Ray& ray, double *a, double *b, double *c) const
{
    *a = mat.x[0][0] * ray.dd.x +
	 mat.x[1][1] * ray.dd.y +
	 mat.x[2][2] * ray.dd.z +
	 D * ray.dir.x * ray.dir.y +
	 E * ray.dir.y * ray.dir.z +
	 F * ray.dir.x * ray.dir.z;
    *b = A * ray.ld.x +
	 B * ray.ld.y +
	 C * ray.ld.z +
	 D * (ray.loc.x * ray.dir.y + ray.loc.y * ray.dir.x) +
	 E * (ray.loc.y * ray.dir.z + ray.loc.z * ray.dir.y) +
	 F * (ray.loc.x * ray.dir.z + ray.loc.z * ray.dir.x) +
	 G * ray.dir.x +
	 H * ray.dir.y +
	 J * ray.dir.z;
    *c = mat.x[0][0] * ray.ll.x +
	 mat.x[1][1] * ray.ll.y +
	 mat.x[2][2] * ray.ll.z +
	 D * ray.loc.x * ray.loc.y +
	 E * ray.loc.y * ray.loc.z +
	 F * ray.loc.x * ray.loc.z +
	 G * ray.loc.x +
	 H * ray.loc.y +
	 J * ray.loc.z +
	 mat.x[3][3];
}

// Quadric intersection routine.  Find the nearest intersection,
// if any.  Slow and ugly, but serviceable.
int
Quadric::NearestInt(Ray& ray, float& t, float maxt)
{
    double a, b, c, d;
    float t1, t2;

    Statistics::Intersections::Quadric++;
    ComputeABC(ray, &a, &b, &c);

    d = b * b - 4 * a * c;
    if (d <= 0)
	return 0;

    d = sqrt(d);
    a += a;
    if (fabs(a) < Limits::Small)
	return 0;
    t1 = (-b + d) / a;
    if (t1 >= maxt)
	t1 = -1;
    t2 = (-b - d) / a;
    if (t2 >= maxt)
	t2 = -1;

    if (t1 < 0) {
	if (t2 < 0)
	    return 0;
	t = t2;
    }
    else {
	if (t2 < 0)
	    t = t1;
	else
	    t = (t1 < t2) ? t1 : t2;
    }
    return t > Limits::Threshold;
}

SpanList *
Quadric::FindAllIntersections(Ray& ray)
{
    double a, b, c, d;
    float t1, t2;

    Statistics::Intersections::Quadric += 1;
    ComputeABC(ray, &a, &b, &c);
    d = b * b - 4 * a * c;
    if (d <= 0)
	return 0;
    d = sqrt(d);
    a += a;
    if (fabs(a) < Limits::Small)
    	return 0;
    t1 = (-b + d) / a;
    t2 = (-b - d) / a;

    // If both parameters are less than threshold, return no-intersection.
    if (t1 < Limits::Threshold && t2 < Limits::Threshold)
	return 0;

    // And if either parameter is near zero, assume self-intersection
    // and return no-intersection.
    if (fabs(t1) < Limits::Threshold || fabs(t2) < Limits::Threshold)
	return 0;
    if (t1 > t2)
	Swap(t1, t2);
    if (t1 < 0) {
	if (IsInside(ray.loc))
	    return new SpanList(Span(t1, this, t2, this));
	else
	    return new SpanList(Span(t2, this, Limits::Infinity, this));
    }
    return new SpanList(Span(t1, this, t2, this));
}

// Quadric normal-finder.  Again, slow and ugly.
Vector3D
Quadric::FindNormal(const Vector3D& intersection)
{
    Vector3D Direction(	A * intersection.x +
			D * intersection.y +
			F * intersection.z + G,
			B * intersection.y +
			D * intersection.x +
			E * intersection.z + H,
			C * intersection.z +
			F * intersection.x +
			E * intersection.y + J);
    return Normalize(Direction);
}

// Apply a transform to a quadric.
void
Quadric::ApplyTransform(const Matrix& tform)
{
    Matrix Inverted = Invert(tform);
    Matrix Transposed = Transpose(Inverted);

    mat = Inverted * mat * Transposed;

    Precompute();
}

// Precompute some useful values related to the quadric.
void
Quadric::Precompute()
{
    int i, j;
    float min = 1;

    for (i = 0; i < 4; i++)
	for (j = 0; j < 4; j++)
	    if (fabs(mat.x[i][j]) > 1e-6 &&
		fabs(mat.x[i][j]) < min /*&&
		fabs(mat.x[i][j] > 0.001)*/)
	      min = fabs(mat.x[i][j]);
    mat *= 1 / min;
    A = 2 * mat.x[0][0];
    B = 2 * mat.x[1][1];
    C = 2 * mat.x[2][2];
    D = mat.x[0][1] + mat.x[1][0];
    E = mat.x[1][2] + mat.x[2][1];
    F = mat.x[2][0] + mat.x[0][2];
    G = mat.x[3][0] + mat.x[0][3];
    H = mat.x[1][3] + mat.x[1][3];
    J = mat.x[2][3] + mat.x[3][2];
}

// Duplicate the quadric.
Object3D *
Quadric::Dup() const
{
    return new Quadric(*this);
}

float
Quadric::PtDistance(Vector3D& v)
{
    return mat.x[0][0] * v.x * v.x +
	   mat.x[1][1] * v.y * v.y +
	   mat.x[2][2] * v.z * v.z +
	   D * v.x * v.y +
	   E * v.y * v.z +
	   F * v.x * v.z +
	   G * v.x +
	   H * v.y +
	   J * v.z + mat.x[3][3];
}

// Describe a quadric.	Indent by the given amount first.
void
Quadric::Describe(int ind) const
{
    indent(ind);
    cout << setprecision(7) << "Quadric: " << A << ", " << B << ", ";
    cout << C << ", " << D << ", " << E << ", " << F << ", " << G;
    cout << ", " << H << ", " << J << ", " << mat.x[3][3] << '\n';
}


// ==================================================================
// Ellipsoid
//	Derived from Quadric.  However, because ellipsoids are
//	bounded, we can surround our ellipsoid with a loose bounding
//	volume and move the bounding volume around whenever the
//	ellipsoid is transformed.
// ==================================================================

Ellipsoid::Ellipsoid(float Xrad, float Yrad, float Zrad, Surface *NewColor):
    Quadric(1/(Xrad*Xrad), 1/(Yrad*Yrad), 1/(Zrad*Zrad),
	0, 0, 0, 0, 0, 0, -1, NewColor)
{
    a = Xrad;	// Needed only when Dup() is called.
    b = Yrad;
    c = Zrad;
    bbox = AxisAlignedBox(Vector3D(-a, -b, -c), Vector3D(a, b, c));
}

void
Ellipsoid::ApplyTransform(const Matrix& tform)
{
    bbox = Transform(bbox, tform);
    Quadric::ApplyTransform(tform);
}

Object3D *
Ellipsoid::Dup() const
{
    return new Ellipsoid(a, b, c, surf);
}

void
Ellipsoid::Describe(int ind) const
{
    indent(ind);
    cout << "Ellipsoid: (" << a << ", " << b << ", " << c << ")\n";
}

AxisAlignedBox
Ellipsoid::BBox() const
{
    return bbox;
}

