// ===================================================================
// span.cpp
//	MergeIntersection, MergeDifference and MergeUnion functions
//	take two SpanLists and merge them according to a CSG
//	operation.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"

SpanList *
MergeIntersection(SpanList *A, SpanList *B)
{
    SpanList *ret;
    Span *workingA, *workingB;

    if (! A || ! B) {
	if (A)
	    delete A;
	if (B)
	    delete B;
	return 0;
    }

    ret = new SpanList;
    workingA = workingB = 0;
    while (A->NumSpans() && B->NumSpans()) {
	if (! workingA)
	    workingA = new Span(A->ExtractMinSpan());
	if (! workingB)
	    workingB = new Span(B->ExtractMinSpan());
	if (Overlap(*workingA, *workingB)) {
	    Span addme = *workingA;

	    if (workingB->tmin > workingA->tmin) {
		addme.tmin = workingB->tmin;
		addme.omin = workingB->omin;
	    }
	    if (workingB->tmax < workingA->tmax) {
	        addme.tmax = workingB->tmax;
		addme.omax = workingB->omax;
	    }
	    ret->AddSpan(addme);
	    // Eliminate the span with the smaller max parameter
	    if (workingA->tmax < workingB->tmax) {
		delete workingA;
		workingA = 0;
	    }
	    else {
		delete workingB;
		workingB = 0;
	    }
	}
	else {	    // Spans are disjoint, eliminate the one with min parameter
	  if (workingA->tmin < workingB->tmin) {
	      delete workingA;
	      workingA = 0;
	  }
	  else {
	      delete workingB;
	      workingB = 0;
	  }
	}
    }
    if (workingA)
	delete workingA;
    if (workingB)
	delete workingB;
    delete A;
    delete B;
    if (ret->NumSpans())
	return ret;
    else {
	delete ret;
	return 0;
    }
}

SpanList *
MergeUnion (SpanList *A, SpanList *B)
{
    SpanList *ret;
    Span *workingA, *workingB, *addme;

    if (! A)
      return B;
    if (! B)
      return A;

    ret = new SpanList;
    workingA = workingB = addme = 0;
    while(A->NumSpans() || B->NumSpans()) {
	if (! workingA)
	    workingA = new Span(A->ExtractMinSpan());
	if (! workingB)
	    workingB = new Span(B->ExtractMinSpan());
	if (addme) {	    // Compare against addme if there
	    if (Overlap(*workingA, *addme)) {	      // A overlaps addme?
		if (workingA->tmin < addme->tmin) {
		    addme->tmin = workingA->tmin;
		    addme->omin = workingA->omin;
		}
		if (workingA->tmax > addme->tmax) {
		    addme->tmax = workingA->tmax;
		    addme->omax = workingA->omax;
		}
		delete workingA;
		workingA = 0;
	    }
	    else if (Overlap(*workingB, *addme)) {    // B overlaps addme?
		if (workingB->tmin < addme->tmin) {
		    addme->tmin = workingB->tmin;
		    addme->omin = workingB->omin;
		}
		if (workingB->tmax > addme->tmax) {
		    addme->tmax = workingB->tmax;
		    addme->omax = workingB->omax;
		}
		delete workingB;
		workingB = 0;
	    }
	    else {		      // Neither overlaps addme, so addme
		ret->AddSpan(*addme); // must be disjoint.
		delete addme;
		addme = 0;
	    }
	}
	else {	    // No addme to compare to, compare A and B
	    if (Overlap(*workingA, *workingB)) {
		addme = workingA;
		if (workingB->tmin < addme->tmin) {
		    addme->tmin = workingB->tmin;
		    addme->omin = workingB->omin;
		}
		if (workingB->tmax > addme->tmax) {
		    addme->tmax = workingB->tmax;
		    addme->omax = workingB->omax;
		}
		workingA = 0;
		delete workingB;
		workingB = 0;
	    }
	    else {	      // Disjoint, add the minimum
		if (workingA && workingB) {	// Compare if both there
		    if (workingA->tmin < workingB->tmin) {
			ret->AddSpan(*workingA);
			delete workingA;
			workingA = 0;
		    }
		    else {
			ret->AddSpan(*workingB);
			delete workingB;
			workingB = 0;
		    }
		}
		else {				// One SpanList is exhausted
		    if (workingA) {
			ret->AddSpan(*workingA);
			delete workingA;
			workingA = 0;
		    }
		    else {
			ret->AddSpan(*workingB);
			delete workingB;
			workingB = 0;
		    }
		}
	    }
	}
    }
    delete A;
    delete B;
    if (addme) {
	ret->AddSpan(*addme);
	delete addme;
    }
    return ret;
}

SpanList *
MergeDifference (SpanList *A, SpanList *B)
{
    SpanList *ret;
    Span *workingA, *workingB;

    if (! A) {
	if (B)
	    delete B;
	return 0;
    }

    if (! B)
	return A;

    ret = new SpanList;
    workingA = workingB = 0;
    while (A->NumSpans() && B->NumSpans()) {
	if (! workingA)
	    workingA = new Span(A->ExtractMinSpan());
	if (! workingB)
	    workingB = new Span(B->ExtractMinSpan());

	if (Overlap(*workingA, *workingB)) {

	    int switchon = (workingA->tmin < workingB->tmin) << 1 |
			   (workingA->tmax < workingB->tmax);

	    switch (switchon) {
	      case 0:
		  /* Have to clip A with B, add A to the SpanList, and delete A */
		  workingA->tmin = workingB->tmax;
		  workingA->omin = workingB->omax;
		  delete workingB;
		  workingB = 0;
		  break;
	      case 1: /* workingA encompasses workingB */
		  delete workingA;
		  workingA = 0;
		  break;
	      case 2: /* workingB encompasses workingA */
		  ret->AddSpan(Span(	workingA->tmin, workingA->omin,
		  			workingB->tmin, workingB->omin));
//		  ret->AddSpan(Span(workingA->min, workingB->min));
		  workingA->tmin = workingB->tmax;
		  workingA->omin = workingB->omax;
		  delete workingB;
		  workingB = 0;
		  break;
	      case 3:
		  workingA->tmax = workingB->tmin;
		  workingA->omax = workingB->omin;
		  delete workingA;
		  workingA = 0;
		  break;
	  }
	}
	else {	    // Spans are disjoint, eliminate the one with min parameter
	    if (workingA->tmin < workingB->tmin) {
		ret->AddSpan(*workingA);
		delete workingA;
		workingA = 0;
	    }
	    else {
		delete workingB;
		workingB = 0;
	    }
	}
    }

    while (A->NumSpans())
	ret->AddSpan(A->ExtractMinSpan());
    delete A;
    delete B;

    if (workingA) {
	ret->AddSpan(*workingA);
	delete workingA;
    }

    if (workingB)
	delete workingB;

    if (ret->NumSpans())
	return ret;
    else {
	delete ret;
	return 0;
    }
}
