// ==============================================================
// surface.cpp
//	Implementation of different shading models.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ==============================================================

#include "oort.h"
#include "world.h"

void
HallSurface::PreMulTransform(const Matrix& tform)
{
    // Bump maps can't get transforms applied to them yet.
    //  if (NewNormal)
    //    NewNormal->PreMulTransform(tform);
    if (Ambient)
	Ambient->PreMulTransform(tform);
    if (Diffuse)
	Diffuse->PreMulTransform(tform);
    if (Specular)
	Specular->PreMulTransform(tform);
    if (Transmit)
	Transmit->PreMulTransform(tform);
    if (Reflect)
	Reflect->PreMulTransform(tform);
}

void
HallSurface::ApplyTransform(const Matrix& tform)
{
    // Bump maps can't get transforms applied to them yet.
    //  if (NewNormal)
    //    NewNormal->ApplyTransform(tform);
    if (Ambient)
	Ambient->ApplyTransform(tform);
    if (Diffuse)
	Diffuse->ApplyTransform(tform);
    if (Specular)
	Specular->ApplyTransform(tform);
    if (Transmit)
	Transmit->ApplyTransform(tform);
    if (Reflect)
	Reflect->ApplyTransform(tform);
}

Surface *
HallSurface::Dup() const
{
    HallSurface *ret = new HallSurface;
    if (NewNormal)
	ret->SetBumpMap(NewNormal->Dup());
    if (Ambient)
	ret->SetAmbient(Ambient->Dup());
    if (Diffuse)
	ret->SetDiffuse(Diffuse->Dup());
    if (Specular)
	ret->SetSpecular(Specular->Dup(), specnum);
    if (Transmit)
	ret->SetTransmit(Transmit->Dup(), index);
    if (Reflect)
	ret->SetReflect(Reflect->Dup());
    return ret;
}

int
HallSurface::Transparent()
{
    return Transmit != 0;
}

void
HallSurface::SetBumpMap(BumpMap *_bumpmap)
{
    NewNormal = _bumpmap;
}

void
HallSurface::SetAmbient(Texture *_ambient)
{
    Ambient = _ambient;
}

void
HallSurface::SetDiffuse(Texture *_diffuse)
{
    Diffuse = _diffuse;
}

void
HallSurface::SetSpecular(Texture *_specular, float _specnum)
{
    Specular = _specular;
    specnum = _specnum;
}

void
HallSurface::SetTransmit(Texture *_transmit, float _index)
{
    Transmit = _transmit;
    index = _index;
}

void
HallSurface::SetReflect(Texture *_reflect)
{
    Reflect = _reflect;
}

void
HallSurface::BeginShadeLight(Vector3D& loc)
{
    diffuse = (Diffuse) ? Diffuse->GetColor(loc) : RGBColor(0);
    specular = (Specular) ? Specular->GetColor(loc) : RGBColor(0);
}

RGBColor
HallSurface::ShadeLight(ShadingInfo& shade, Vector3D& dir)
{
    float diffuseatt = DotProd(shade.normal, dir);
    RGBColor ret = (diffuseatt >= 0) ? (RGBColor) (diffuseatt * diffuse) : RGBColor(0);
    if (Specular) {
	float specularatt = DotProd(shade.reflected, dir);
	if (specularatt >= 0)
	    ret += specular * pow(specularatt, specnum);
    }
    return ret;
}

RGBColor
HallSurface::ComputeColor(ShadingInfo& shade, int depth)
{
    if (NewNormal)
	NewNormal->PerturbNormal(shade);

    RGBColor AmtLight = RGBColor(0);

    if (Ambient)
	AmtLight += Ambient->GetColor(shade.p) * shade.world->GetAmbientLight();

    if (Diffuse || Specular) {
	BeginShadeLight(shade.p);
	AmtLight += shade.world->DiffuseLight(shade, this);
    }

    if (shade.world->HitDepthLimit(depth))
	return AmtLight;

    if (Reflect) {
	Ray ReflectedRay(shade.p, shade.ray->ReflectRay(shade.normal));
	Statistics::ReflectedRays++;
	AmtLight += Reflect->GetColor(shade.p) *
		    shade.world->FindColor(ReflectedRay, depth+1);
    }

    if (Transmit) {
	Vector3D normal = shade.normal;
	float gamma, dot, sqrterm;

	Statistics::TransmittedRays++;

    	dot = DotProd(shade.normal, -shade.incident);
    	if (dot > 0)
    	    gamma = 1 / index;
    	else {
    	    gamma = index;
    	    dot = -dot;
    	    normal = -normal;
    	}
    	sqrterm = 1.0 - gamma*gamma*(1.0 - dot*dot);

    	// Check for total internal reflection.  
	// Do nothing if it applies.
    	if (sqrterm > 0) {
    	    sqrterm = gamma*dot - sqrt(sqrterm);
    	    Ray TransmittedRay = Ray(shade.p,
				     normal*sqrterm + shade.incident*gamma);
    	    AmtLight += Transmitted(shade.p) *
			shade.world->FindColor(TransmittedRay, depth+1);
    	}
    }
    return AmtLight;
}

RGBColor
HallSurface::Transmitted(Vector3D& Intersection)
{
    return Transmit->GetColor(Intersection);
}

Surface *
DistributedHall::Dup() const
{
    DistributedHall *ret = new DistributedHall;

    if (NewNormal)
	ret->SetBumpMap(NewNormal->Dup());
    if (Ambient)
	ret->SetAmbient(Ambient->Dup());
    if (Diffuse)
	ret->SetDiffuse(Diffuse->Dup());
    if (Specular)
	ret->SetSpecular(Specular->Dup(), specnum);
    if (Transmit)
	ret->SetTransmit(Transmit->Dup(), index);
    if (Reflect)
	ret->SetReflect(Reflect->Dup());
    ret->ReflectParms = ReflectParms;
    ret->TransParms = TransParms;
    return ret;
}

void
DistributedHall::SetReflectParms(const DistribParms& dist)
{
    ReflectParms = dist;
}

void
DistributedHall::SetTransmitParms(const DistribParms& dist)
{
    TransParms = dist;
}

Vector3D
DistributedHall::NewSample(const DistribParms& d, const Vector3D *samples, int n)
{
    float ang = RandomValue(0, 2*M_PI);
    float rad = RandomValue(0, d.GetAngle());

    Vector3D ret = RotationZMatrix(ang) * (sin(rad) * Vector3D(1, 0, 0));
    if (! n)
	return ret;// + Vector3D(0, 0, cos(rad));
    float mindist = MinDist(ret, samples, n);
    ret += Vector3D(0, 0, cos(rad));
    for (int i = 0; i < d.GetCandidates(); i++) {
	ang = RandomValue(0, 2*M_PI);
	rad = RandomValue(0, d.GetAngle());

	Vector3D candidate = RotationZMatrix(ang) * (sin(rad) * Vector3D(1, 0, 0));
	float dist = MinDist(candidate, samples, n);
	if (dist > mindist) {
	    ret = candidate;// + Vector3D(0, 0, cos(rad));
	    mindist = dist;
	}
    }
    return ret;
}

RGBColor
DistributedHall::SampleSpace(ShadingInfo& shade,
			     Vector3D& dir,
			     const DistribParms& dparms,
			     int depth,
			     long *IncMe)
{
    RGBColor Imin(MAXFLOAT), Imax(-MAXFLOAT), Intens(0, 0, 0);
    int n = 0;

    // Compute a transformation matrix that transforms a ray pointing
    // down the +Z axis so it points along dir.
    Vector3D z = dir;
    Vector3D y = Normalize(Vector3D(0, 1, 0) - dir * DotProd(Vector3D(0, 1, 0), dir));
    if (Magnitude(y) < Limits::Small) {
	cout << "Small y\n";	// npw-I've never seen this happen.
				// I just want to know when it does.
    }
    Vector3D x = CrossProd(y, z);
    Matrix xform = Transpose(GenRotation(x, y, z));

    Vector3D *pts = new Vector3D[dparms.GetMaxSamples()];

    for (int i = 0; i < dparms.GetMaxSamples(); i++) {
	pts[n] = NewSample(dparms, pts, n);
	n += 1;

	float mag = Magnitude(pts[i]);
	Vector3D shootat = pts[i] + Vector3D(0, 0, 1-mag*mag);
	shootat = xform * shootat;

	*IncMe += 1;
	RGBColor clr = shade.world->FindColor(Ray(shade.p, shootat), depth + 1);
	Minimize(Imin, clr);
	Maximize(Imax, clr);
	Intens += clr;

	if (i > dparms.GetMinSamples()) {
	    if (i >= dparms.GetMaxSamples())
	    	break;
	    if (! MoreRays(Imin, Imax, dparms.GetThreshold()))
		break;
	}
    }
    delete[] pts;
    return Intens / n;
}

RGBColor
DistributedHall::ComputeColor(ShadingInfo& shade, int depth)	
{
    if (NewNormal)
	NewNormal->PerturbNormal(shade);

    RGBColor AmtLight = RGBColor(0);

    if (Ambient)
	AmtLight += Ambient->GetColor(shade.p) * 
		    shade.world->GetAmbientLight();

    if (Diffuse || Specular) {
	BeginShadeLight(shade.p);
	AmtLight += shade.world->DiffuseLight(shade, this);
    }

    if (shade.world->HitDepthLimit(depth))
	return AmtLight;

    if (Reflect) {
	AmtLight += Reflect->GetColor(shade.p) *
		    SampleSpace(shade,
				shade.ray->ReflectRay(shade.normal),
				ReflectParms, 
				depth, 
				&Statistics::ReflectedRays);
    }


    if (Transmit) {
	Vector3D normal = shade.normal;
    	float gamma, dot, sqrterm;
    	dot = DotProd(shade.normal, -shade.incident);
    	if (dot > 0)
    	    gamma = 1 / index;
    	else {
    	    gamma = index;
    	    dot = -dot;
    	    normal = -normal;
    	}
    	sqrterm = 1.0 - gamma*gamma*(1.0 - dot*dot);

    	// Check for total internal reflection.  
	// Do nothing if it applies.
	if (sqrterm > 0) {
	    sqrterm = gamma*dot - sqrt(sqrterm);
	    Vector3D transmitteddir = Normalize(shade.normal*sqrterm + 
						shade.incident*gamma);
	    AmtLight += Transmitted(shade.p) *
	    		SampleSpace(shade,
				    transmitteddir,
				    TransParms,
				    depth,
				    &Statistics::TransmittedRays);
	}
    }
    return AmtLight;
}
