// ===================================================================
// objs.cpp
//	Implementation file for object-manufacturing functions.
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "utils.h"

Aggregate *
MakeBox(const Vector3D& min, const Vector3D& max, Surface *surf)
{
    Aggregate *ret = new Aggregate;
    Vector3D sides[4];

    sides[3] = Vector3D(min.x, min.y, min.z);
    sides[2] = Vector3D(min.x, max.y, min.z);
    sides[1] = Vector3D(min.x, max.y, max.z);
    sides[0] = Vector3D(min.x, min.y, max.z);
    ret->AddObject(Polygon(4, sides, surf));

    sides[0] = Vector3D(max.x, min.y, min.z);
    sides[1] = Vector3D(max.x, max.y, min.z);
    sides[2] = Vector3D(max.x, max.y, max.z);
    sides[3] = Vector3D(max.x, min.y, max.z);
    ret->AddObject(Polygon(4, sides, surf));

    sides[3] = Vector3D(min.x, min.y, min.z);
    sides[2] = Vector3D(max.x, min.y, min.z);
    sides[1] = Vector3D(max.x, max.y, min.z);
    sides[0] = Vector3D(min.x, max.y, min.z);
    ret->AddObject(Polygon(4, sides, surf));

    sides[0] = Vector3D(min.x, min.y, max.z);
    sides[1] = Vector3D(max.x, min.y, max.z);
    sides[2] = Vector3D(max.x, max.y, max.z);
    sides[3] = Vector3D(min.x, max.y, max.z);
    ret->AddObject(Polygon(4, sides, surf));

    sides[0] = Vector3D(min.x, max.y, min.z);
    sides[1] = Vector3D(min.x, max.y, max.z);
    sides[2] = Vector3D(max.x, max.y, max.z);
    sides[3] = Vector3D(max.x, max.y, min.z);
    ret->AddObject(Polygon(4, sides, surf));

    sides[3] = Vector3D(min.x, min.y, min.z);
    sides[2] = Vector3D(min.x, min.y, max.z);
    sides[1] = Vector3D(max.x, min.y, max.z);
    sides[0] = Vector3D(max.x, min.y, min.z);
    ret->AddObject(Polygon(4, sides, surf));

    return ret;
}

Object3D *
MakeCone(float a, float c, Surface *surf)
{
    return new Quadric(1/(a*a), -1, 1/(c*c), 0, 0, 0, 0, 0, 0, 0, surf); 
//    return new Algebraic(Quadric(1/(a*a), -1, 1/(c*c), 0, 0, 0, 0, 0, 0, 0, surf)); 
//    return new AlgQuadric(1/(a*a), -1, 1/(c*c), 0, 0, 0, 0, 0, 0, 0, surf); 
}

Object3D *
MakeCylinder(float a, float c, Surface *surf)
{
    return new Quadric(1/(a*a), 0, 1/(c*c), 0, 0, 0, 0, 0, 0, -1, surf);
//    return new Algebraic(Quadric(1/(a*a), 0, 1/(c*c), 0, 0, 0, 0, 0, 0, -1, surf));
//    return new AlgQuadric(1/(a*a), 0, 1/(c*c), 0, 0, 0, 0, 0, 0, -1, surf);
}

Object3D *
MakeHyperboloid(float a, float b, float c, Surface *surf)
{
    return new Quadric(1/(a*a), -1/(b*b), 1/(c*c), 0, 0, 0, 0, 0, 0, -1, surf);
//    return new Algebraic(Quadric(1/(a*a), -1/(b*b), 1/(c*c), 0, 0, 0, 0, 0, 0, -1, surf));
//    return new AlgQuadric(1/(a*a), -1/(b*b), 1/(c*c), 0, 0, 0, 0, 0, 0, -1, surf);
}

Object3D *
MakeParaboloid(float a, float c, float f, Surface *surf)
{
    return new Quadric(1/(a*a), 0, 1/(c*c), 0, 0, 0, 0, -4 * f, 0, 0, surf);
//    return new Algebraic(Quadric(1/(a*a), 0, 1/(c*c), 0, 0, 0, 0, -4 * f, 0, 0, surf));
//    return new AlgQuadric(1/(a*a), 0, 1/(c*c), 0, 0, 0, 0, -4 * f, 0, 0, surf);
}

Algebraic *
MakeCubicCusp(Surface *surf)
{
    Algebraic *ret = new Algebraic;

    ret->AddTerm(1, 0, 0, 3);	// z^3 +
    ret->AddTerm(1, 1, 0, 1);	// xz +
    ret->AddTerm(1, 0, 1, 0);	// y

    return ret;
}

Algebraic *
MakeSteiner(Surface *surf)
{
    Algebraic *ret = new Algebraic(surf);

    ret->AddTerm(1, 2, 2, 0);	// x^2y^2 +
    ret->AddTerm(1, 0, 2, 2);	// y^2z^2 +
    ret->AddTerm(1, 2, 0, 2);	// x^2z^2 +
    ret->AddTerm(1, 1, 1, 1);	// xyz

    return ret;
}

Algebraic *
MakeKummer(Surface *surf)
{
    Algebraic *ret = new Algebraic(surf);

    ret->AddTerm(1, 4, 0, 0);	// (x^4 + y^4 + z^4 + 1) -
    ret->AddTerm(1, 0, 4, 0);
    ret->AddTerm(1, 0, 0, 4);
    ret->AddTerm(1, 0, 0, 0);
    ret->AddTerm(-1, 2, 0, 0);	// (x^2 + y^2 + z^2 + y^2z^2 + z^2x^2 + x^2y^2)
    ret->AddTerm(-1, 0, 2, 0);
    ret->AddTerm(-1, 0, 0, 2);
    ret->AddTerm(-1, 0, 2, 2);
    ret->AddTerm(-1, 2, 0, 2);
    ret->AddTerm(-1, 2, 2, 0);

    return ret;
}

Algebraic *
MakeQuarticCylinder(Surface *surf)
{
    Algebraic *ret = new Algebraic(surf);

    ret->AddTerm(1, 2, 2, 0);		// x^2y^2 +
    ret->AddTerm(0.01, 2, 0, 0);	// 0.01x^2 +
    ret->AddTerm(1, 0, 2, 2);		// y^2z^2 +
    ret->AddTerm(0.01, 0, 0, 2);	// 0.01z^2 +
    ret->AddTerm(-0.01, 0, 0, 0);	// -0.01

    return ret;
}

Algebraic *
MakeFolium(Surface *surf)
{
    Algebraic *ret = new Algebraic(surf);

    ret->AddTerm(2, 2, 0, 0);	// 2x^2 +
    ret->AddTerm(-3, 1, 2, 0);	// -3xy^2 +
    ret->AddTerm(-3, 1, 0, 2);	// -3xz^2 +
    ret->AddTerm(1, 0, 2, 0);	// y^2 +
    ret->AddTerm(1, 0, 0, 2);	// z^2

    return ret;
}

Algebraic *
MakePiriform(Surface *surf)
{
    Algebraic *ret = new Algebraic(surf);

    ret->AddTerm(4, 4, 0, 0);	// 4x^4 +
    ret->AddTerm(-4, 3, 0, 0);	// -4x^3 +
    ret->AddTerm(1, 0, 2, 0);	// y^2 +
    ret->AddTerm(1, 0, 0, 2);	// z^2

    return ret;
}

Algebraic *
MakeLemniscate(Surface *surf)
{
    Algebraic *ret = new Algebraic(surf);

    ret->AddTerm(1, 4, 0, 0);	// x^4 +
    ret->AddTerm(-1, 2, 0, 0);	// -x^2 +
    ret->AddTerm(1, 0, 2, 0);	// y^2 +
    ret->AddTerm(1, 0, 0, 2);	// z^2

    return ret;
}

