// ===================================================================
// stetra.cpp
//	Implementation file for STetrahedron class.
//	The code for this file is derived from tetra.c in Eric
//	Haines's public domain Standard Procedural Database (SPD).
//	For more information on the SPD, see his article "A Proposal 
//	For Standard Graphics Environments" in the November 1987 
//	issue of IEEE Computer Graphics & Applications.
//
//
//	     The Object-Oriented Ray Tracer (OORT)
//            Copyright (C) 1993 by Nicholas Wilt.
//
// This software product may be freely copied and distributed in
// unmodified form but may not be sold.  A nominal distribution
// fee may be charged for media and handling by freeware and
// shareware distributors.  The software product may not be
// included in whole or in part into any commercial package
// without the express written consent of the author.
// 
// This software product is provided as is without warranty of
// any kind, express or implied, including but not limited to
// the implied warranties of merchantability and fitness for a
// particular purpose.  The author assumes no liability for any
// alleged or actual damages arising from the use of this
// software.  The author is under no obligation to provide 
// service, corrections or upgrades to the software.
//
// ------------------------------------------------------------
//
// Please contact me with questions, comments, suggestions or
// other input about OORT.  My Compuserve account number is
// [75210,2455] (Internet sites can reach me at 
// 75210.2455@compuserve.com).
//					--Nicholas Wilt
// ===================================================================

#include "oort.h"
#include "utils.h"

STetrahedron::STetrahedron(int depth, float mag, Surface *surf)
{
    CreateTetra(depth, Vector3D(0), mag, surf);
}

// Create tetrahedrons recursively
void 
STetrahedron::CreateTetra(int depth, const Vector3D& center, float mag, Surface *surf)
{
    int	num_face, num_vert, swap, vert_ord[3];
    int x_dir, y_dir, z_dir;
    Vector3D dir;
    Vector3D face_pt[3], obj_pt[4];
    Vector3D sub_center;

    if ( depth <= 1 ) {
	/* Output tetrahedron */

	/* find opposite corners of a cube which form a tetrahedron */
	for ( num_vert = 0, x_dir = -1 ; x_dir <= 1 ; x_dir += 2 ) {
	    for ( y_dir = -1 ; y_dir <= 1 ; y_dir += 2 ) {
		for ( z_dir = -1 ; z_dir <= 1 ; z_dir += 2 ) {
		    if ( x_dir*y_dir*z_dir == 1 ) {
			Vector3D dir(x_dir, y_dir, z_dir);
			obj_pt[num_vert] = center + dir * mag;
			++num_vert ;
		    }
		}
	    }
	}

	/* find faces and output */
	for ( num_face = 0 ; num_face < 4 ; ++num_face ) {
	    /* output order:
	     *   face 0:  points 0 1 2
	     *   face 1:  points 3 2 1
	     *   face 2:  points 2 3 0
	     *   face 3:  points 1 0 3
	     */
	    for ( num_vert = 0 ; num_vert < 3 ; ++num_vert ) {
		vert_ord[num_vert] = (num_face + num_vert) % 4 ;
	    }
	    if ( num_face%2 == 1 ) {
		swap = vert_ord[0] ;
		vert_ord[0] = vert_ord[2] ;
		vert_ord[2] = swap ;
	    }

	    for ( num_vert = 0 ; num_vert < 3 ; ++num_vert )
		face_pt[num_vert] = obj_pt[vert_ord[num_vert]];
	    AddObject(new Polygon(3, face_pt, surf));
	}
    }
    else {
	/* Create sub-tetrahedra */

	/* find opposite corners of a cube to form sub-tetrahedra */
	for ( x_dir = -1 ; x_dir <= 1 ; x_dir += 2 ) {
	    for ( y_dir = -1 ; y_dir <= 1 ; y_dir += 2 ) {
		for ( z_dir = -1 ; z_dir <= 1 ; z_dir += 2 ) {
		    if ( x_dir*y_dir*z_dir == 1 ) {
			Vector3D dir(x_dir, y_dir, z_dir);
			Vector3D subcenter = center + dir * mag / 2.0;
			CreateTetra(depth - 1, subcenter, mag / 2, surf);
		    }
		}
	    }
	}
    }
}


