// lsearch.cpp: locally sorted search for median-cut quantizer.
// Copyright (C) 1991 by Nicholas Wilt.  All rights reserved.

#include <iostream.h>
#include <stdlib.h>
#include <mem.h>
#include "lsearch.h"

#define USEASM

extern int numboxes;

LocalSearch::LocalSearch(Color *NewClrs, int NumClrs)
{
  int i, j, k;

  clrs = NewClrs;
  numclrs = NumClrs;
  list = new Color***[32];
  cnts = new unsigned char[32768];
  for (i = 0; i < 32; i++)
    list[i] = new Color**[32];
  for (i = 0; i < 32; i++)
    for (j = 0; j < 32; j++) {
      list[i][j] = new Color*[32];
      for (k = 0; k < 32; k++)
	list[i][j][k] = 0;
    }
}

extern "C" Color *SearchArray(Color *array, Color& query, int num);
extern "C" long Dist(Color& a, Color& b);
extern "C" int CheckCorner(Color& min, Color& query, long r);

#ifndef USEASM
Color *
SearchArray(Color *array, Color& query, int num)
{
  Color *ret = array;
  long min = Dist(*array++, query);
  for (num--; num--; array++) {
    long temp = Dist(*array, query);
    if (temp < min) {
      min = temp;
      ret = array;
    }
  }
  return ret;
}
#endif

#ifdef USEASM
extern "C" int IsNearCell(Color& cellcent, Color& query, long r);
#else

int IsNearCell(Color& cellcent, Color& query, long r)
{
  Color min, max;
  int i, j, k;

  min.clr.r = cellcent.clr.r - 4;
  min.clr.g = cellcent.clr.g - 4;
  min.clr.b = cellcent.clr.b - 4;
  for (i = 0; i <= 8; i += 8)
    for (j = 0; j <= 8; j += 8)
      for (k = 0; k <= 8; k += 8) {
	Color corner((int) min.clr.r + i,
		     (int) min.clr.g + j,
		     (int) min.clr.b + k);
	if (Dist(corner, query) < r)
	  return 1;
      }
  return 0;
}
#endif

void
LocalSearch::FindColorList(Color& query)
{
  Color localclrs[256];
  Color cellcent;
  long ndist;
  Color *A;
  long r;
  int i, j, k;
  Color *sc;
  int clrcnt;

  cellcent.clr.r = (unsigned char) (((int) query.clr.r >> 3) * 8 + 4);
  cellcent.clr.g = (unsigned char) (((int) query.clr.g >> 3) * 8 + 4);
  cellcent.clr.b = (unsigned char) (((int) query.clr.b >> 3) * 8 + 4);
  A = SearchArray(clrs, cellcent, numclrs);
  ndist = 0L;
  for (i = -4; i <= 4; i += 8)
    for (j = -4; j <= 4; j += 8)
      for (k = -4; k <= 4; k += 8) {
	Color corner((int) cellcent.clr.r + i,
		     (int) cellcent.clr.g + j,
		     (int) cellcent.clr.b + k);
	if (Dist(corner, *A) > ndist)
	  ndist = Dist(corner, *A);
      }
  // ndist now contains largest distance from A to a cell corner
  sc = localclrs;
  clrcnt = 0;
  for (i = 0; i < numclrs; i++) {
    if (IsNearCell(cellcent, clrs[i], ndist)) {
      *sc++ = clrs[i];
      clrcnt++;
    }
  }
  i = query.clr.r >> 3;
  j = query.clr.g >> 3;
  k = query.clr.b >> 3;
  list[i][j][k] = new Color[clrcnt];
  if (! list[i][j][k]) {
    cerr << "Out of memory\n";
    exit(2);
  }
  memcpy(list[i][j][k], localclrs, clrcnt * sizeof(Color));
  cnts[(((i << 5)+j) << 5)+k] = (unsigned char) clrcnt;
}

Color *
LocalSearch::FindNearest(Color& query)
{
  int i = query.clr.r >> 3;
  int j = query.clr.g >> 3;
  int k = query.clr.b >> 3;

  if (! list[i][j][k])
    FindColorList(query);
  return SearchArray(list[i][j][k], query, cnts[(((i << 5) + j) << 5) + k]);
}

/*
long Dist(Color& a, Color& b)
{
  int i;
  long subt = 0L;

  for (i = 0; i < 3; i++) {
    int temp = a.clrs[i] - b.clrs[i];
    subt += (long) temp * temp;
  }
  return subt;
}
*/
