/* display.c */

#include <windows.h>
#include <stdlib.h>
#include "display.h"
#include "ascii.h"

#define MIN(a,b) ((a<=b)?(a):(b))

/* private variables */
static HWND hTheWnd = 0;
static int CharHeight;       /* char height */
static int CharWidth;        /* (fixed) char width */
static int TheRow = 0;       /* current row */
static int TheCol = 0;       /* current col */
static int TopRow = 0;       /* top row in invalid rectangle */
static int LeftCol= 0;       /* left col in invalid rectangle */
static int RightCol = 0;     /* right col in invalid rectangle */
static char Buffer[NROWS][NCOLS];  /* display buffer */
static char *RowPtr[NROWS];  /* array of row pointers */
static TEXTMETRIC tm;        /* text metric structure */

/*** PRIVATE functions ***/

static void DoTheScroll(void)
{int Row;
 int Col;
 char *Ptr;
 RECT rect;
 /* scroll display buffer */
 TheRow = NROWS-1;
 Ptr = RowPtr[0];
 for(Row=0;Row<NROWS-1;Row++) RowPtr[Row] = RowPtr[Row+1];
 RowPtr[NROWS-1] = Ptr;
 for(Col=0;Col<NCOLS;Col++) *Ptr++ = ' ';
 /* scroll the display */
 ScrollWindow(hTheWnd,0,0-CharHeight,NULL,NULL);
 /* invalidate last row */
 rect.left = 0;
 rect.top  = CharHeight * (NROWS-2);
 rect.right  = CharWidth * (RightCol+1);
 rect.bottom = CharHeight * (NROWS-1);
 InvalidateRect(hTheWnd,&rect,TRUE);
 /* reset boundary */
 TopRow = TheRow;
 LeftCol = TheCol;
 RightCol = TheCol;
} /* end DoTheScroll */

/*** PUBLIC functions ***/

int FAR PASCAL GetCol(void)
{return(TheCol);
}

int FAR PASCAL GetRow(void)
{return(TheRow);
}

void FAR PASCAL DisplayInit(HWND hWnd)
{int Col;
 int Row;
 HDC hDC;
 hTheWnd = hWnd;
 hDC = GetDC(hWnd);
 SelectObject(hDC, GetStockObject(OEM_FIXED_FONT) );
 GetTextMetrics(hDC,&tm);
 CharHeight = tm.tmHeight + tm.tmExternalLeading;
 CharWidth = tm.tmMaxCharWidth;
 ReleaseDC(hWnd,hDC);
 /* initialize screen buffer */
 for(Row=0;Row<NROWS;Row++)
   {for(Col=0;Col<NCOLS;Col++) Buffer[Row][Col] = ' ';
    RowPtr[Row] = &Buffer[Row][0];
   }
 TheRow = 0;
 TheCol = 0;
} /* end DisplayInit */


void FAR PASCAL DisplayText(char *String)
{int i;
 int Length;
 char TheChar;
 RECT rect;
 /* begin */
 TopRow = TheRow;
 LeftCol = TheCol;
 RightCol = TheCol;
 Length = lstrlen(String);
 for(i=0;i<Length;i++)
   {TheChar = *String++;
    switch(TheChar)
     {case BS:
        if(TheCol>0)
          {*(RowPtr[TheRow]+TheCol) = ' ';
           TheCol--;
          }
        break;
      case CR:
        TheCol = 0;
        LeftCol = 0;
        break;
      case LF:
        /* next line */
        if(++TheRow>=NROWS) DoTheScroll();
        break;
      default:
        /* put char into display buffer */
        *(RowPtr[TheRow]+TheCol) = (char)TheChar;
        /* increment 'cursor' */
        if(++TheCol>=NCOLS)
          {/* next line */
           TheCol = 0;
           LeftCol = 0;
           if(++TheRow>=NROWS) DoTheScroll();
          }
        else RightCol++;
        break;
     } /* end switch */
   } /* end for */
 /* compute invalid rectangle */
 if((TopRow!=TheRow)||(LeftCol!=TheCol)||(RightCol!=TheCol))
   {rect.left = CharWidth * LeftCol;
    rect.top  = CharHeight * TopRow;
    rect.right  = CharWidth * (RightCol+1);
    rect.bottom = CharHeight * (TheRow+1);
    InvalidateRect(hTheWnd,&rect,TRUE);
   }
}  /* end DisplayText */


void FAR PASCAL DisplayPaint(HDC hDC,PAINTSTRUCT *ps)
{int Row;
 int FirstRow;
 int FirstCol;
 int NbrRows;
 int NbrCols;
 int ColWidth;
 int X;
 int Y;
 RECT rect;
 /* compute row & col stuff */
 FirstRow = ps->rcPaint.top  / CharHeight;
 FirstCol = ps->rcPaint.left / CharWidth;
 NbrRows = (ps->rcPaint.bottom - ps->rcPaint.top)  / CharHeight;
 ColWidth = ps->rcPaint.right  - ps->rcPaint.left;
 NbrCols = MIN(NCOLS,(1+ColWidth) / CharWidth);
 X = ps->rcPaint.left;
 /* consider each row */
 for(Row=FirstRow;Row<FirstRow+NbrRows;Row++)
   {/* paint part of row */
    if((Row>=0)&&(Row<NROWS))
      {/* good row number */
       Y = CharHeight*Row;
       /* compute bounding rectangle */
       rect.left = X;
       rect.top  = Y;
       rect.right  = X + ColWidth;
       rect.bottom = Y + CharHeight;
       /* paint it */
       SetBkMode(hDC,OPAQUE);
       ExtTextOut(hDC,X,Y,ETO_OPAQUE|ETO_CLIPPED,&rect,
          RowPtr[Row]+FirstCol,NbrCols,NULL);
      }
   }
} /* end DisplayPaint */
