/*************************************************************************

    File:     display.cpp
    Contents: Code for tDisplay class, including low level graphics driver
              and 3D projection and plot routines
              This file contains the first half of the code.
    Author:   Mike Lyons with modifications by Sheldon Young

    Project: 3D Lab, Virtual Reality Walk-Through
    Class:   CNC CSC 216, Spring, 1994
    Prof:    George Kaweesi


    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Sheldon Young           (604)963-7067
    PO Box 2931             FidoNet:  Sheldon Young 1:359/197.1
    Prince George, BC       Internet: sheldon.young@datex.com
    V2N 4T7
    Canada

    Mike Lyons              (604)964-0725
    7833 Piedmont Crescent  FidoNet: Mike Lyons 1:359/262.1
    Prince George, BC       Internet: mike.lyons@datex.com
    V2N 3K8
    Canada

*************************************************************************/

#include <iostream.h>
#include <stdlib.h>
#include <mem.h>
#include <alloc.h>
#include <assert.h>
#include <dos.h>

// Included needed definitions
#include "display.h"
#include "texture.h"

/* -----------------------------------------------------------------------
    Constants
----------------------------------------------------------------------- */

// Pointer to the video hardware buffer
unsigned char far * const VideoHardware = (unsigned char far *)0xA0000000L;

/* -----------------------------------------------------------------------
    Constructors and destructors
----------------------------------------------------------------------- */

tDisplay::tDisplay(void)
// Overview: tDisplay constructor.
// Duties:   Sets display to MCGA (320x200x256) mode and
//           allocates memory for hidden video buffer.
// Errors:   Message to cerr then a call to exit().
// Future:   Support for C++ exception throwing
{
    // Try to set up MCGA graphics mode 320 across, 200 down, 256 colors
    _AX = 0x0013;           // See mode 13h
    geninterrupt(0x10);     // Call BIOS SetVideoMode
    if (!_AX)
        {
        cerr << "\n\nSorry; this program requies a VGA or MCGA\n";
        cerr << "compatible graphics adaptor.\n\n";
        exit(EXIT_FAILURE);
        }

    // Try to allocate video buffer
    // Even though an MCGA screen only requires 64000 bytes,
    // we allocate a full 64k segment to make the program more rebust.
    // A wild pointer offset (using large memory model) can only wreak
    // destruction within it's own segment in real mode under DOS.
    VideoBuff = (unsigned char far *)malloc((size_t)65535L);
    if (VideoBuff == NULL)
        {
        cerr << "\n\nUnable to allocate memory for video buffer.\n\n";
        exit(EXIT_FAILURE);
        }

    // Set VideoBase to the hidden buffer so all graphics operations will
    // draw on the hidden buffer instead of directly to video memory.
    VideoBase = VideoBuff;

    // Set color information
    SetPaletteTo(trippyPal);
    BackgroundColor = Black;
    ForegroundColor = White;

    // Setup default text parameters
    CurrentFont = balloonFont;
    TextColor  = 0;                  // No color translation by default
    AutoScroll = False;
    TextSprite = True;
    SetTextBorder(False);

    // Move the text cursor to the top of the screen
    // Keep in mind that text cursor indicates BOTTOM-LEFT of next character
    // (although descenders can extend below this)
    TextX = 0;
    TextY = CurrentFont.BaseHeight;

    // Set display mode to overhead and clear the screen for good measure
    SetMode(Overhead);
    ClearScreen();
}


tDisplay::~tDisplay(void)
// Overview: Destructor for tDisplay
// Effect:   Restores screen to text mode and deallocates display buffer.
{
    // Restore the screen to text mode
    _AX = 0x0003;           // Set mode 03h (80x25 color text)
    geninterrupt(0x10);     // Call BIOS SetVideoMode

    // Deallocate video buffer
    free(VideoBuff);
}

/* -----------------------------------------------------------------------
    Trapezoids
----------------------------------------------------------------------- */

void tDisplay::DrawSolidColorTrapezoid(unsigned int LeftX,
                                       unsigned int LeftTop,
                                       unsigned int LeftBot,
                                       unsigned int RightX,
                                       unsigned int RightTop,
                                       unsigned int RightBot,
                                       const tColor Color)
// Overview: Draws a trapezoid at the specified coordinates.
//           Trapezoid will have it's left and right sides parallel with
//           the left and right edges of the screen.  Trapezoid will be
//           color solid with the color given.  Note that it won't work
//           if any of the coordinates are backwards.
// Inputs:   X-coordinate of left edge of trapezoid.
//           Y coordinate of top of left edge.
//           Y coordinate of bottom of left edge.
//           X coordinate of right edge of trapezoid.
//           Y coordinate of top of right edge.
//           Y coordinate of bottom of right edge.
//           Color to fill with.
// Effects:  Trapezoid filled on the VideoBase.
{
    unsigned int X, Y;          // Position
    unsigned int YTop, YBottom; // Coordinate of top/bottom of line
    int ErrorTop, ErrorBottom;  // Error trackers for Bresnehan's Algorithm
    int DX, DTop, DBottom;      // Changes for Bresnehan
    unsigned char *VideoPtr;    // Pointer to spot in video RAM

    // Initalize Bresnehan values
    DX       = RightX - LeftX;
    DTop     = RightTop - LeftTop;
    DBottom  = RightBot - LeftBot;
    ErrorTop = ErrorBottom = -(DX/2);

    // Break on special cases
    if (DX == 0)
        return;

    // Go from top to bottom for each column
    YTop = LeftTop;
    YBottom = LeftBot;
    for (X = LeftX ; X <= RightX ; X++)
        {

        // Fill column
        VideoPtr = VideoBase + (SCREEN_SIZE_X * YTop) + X;
        for (Y = YTop ; Y <= YBottom ; Y++)
            {
            VideoPtr += SCREEN_SIZE_X;
            *VideoPtr = (char)Color;
            }

        // Find new top and bottom using Bresnehan's Algorithm
        ErrorTop += abs(DTop);
        while(ErrorTop > 0)
            {
            YTop += sgn(DTop);
            ErrorTop -= DX;
            }
        ErrorBottom += abs(DBottom);
        while(ErrorBottom > 0)
            {
            YBottom += sgn(DBottom);
            ErrorBottom -= DX;
            }
        }
}

// Overview: Draws a trapezoid at the specified coordinates.
//           Trapezoid will have it's left and right sides parallel with
//           the left and right edges of the screen.  Trapezoid will be
//           mapped with the given texture.
// Inputs:   X-coordinate of left edge of trapezoid
//           Y coordinate of top of left edge
//           Y coordinate of bottom of left edge
//           X coordinate of right edge of trapezoid
//           Y coordinate of top of right edge
//           Y coordinate of bottom of right edge
//           Texture control structure of texture to fill with
// Effects:  Trapezoid filled on the VideoBase.
void tDisplay::DrawTrapezoid(unsigned int LeftX,
                             unsigned int LeftTop,
                             unsigned int LeftBot,
                             unsigned int RightX,
                             unsigned int RightTop,
                             unsigned int RightBot,
                             const struct texture &Texture)
{
    // If a solid color use an optimized routine
    if(Texture.UseTexture == False)
        {
        DrawSolidColorTrapezoid(LeftX, LeftTop, LeftBot,
                                RightX, RightTop, RightBot, Texture.Color);
        return;
        }

    // We don't used fixed-sized her
    int e1,e2;          // Error trackers for Bresnehan's Algorithm
    int sx,sy1,sy2;     // Signs of the delta values
    unsigned int dx, dy1, dy2, y1, y2, ddx, ddy, dy;
    unsigned int bx, by,
                    bxrs=Texture.TextBmp.X_Size,      // Real sizes
                    byrs=Texture.TextBmp.Y_Size,
                    bxs=bxrs-1, bys=byrs-1;

    // Put this code in if necessary to always draw from outside in
    //      dy1=abs(LeftBot-LeftTop);
    //      dy2=abs(RightBot-RightTop);
    //      if (dy1<dy2) {
    //        swap(LeftX, RightX);
    //        swap(LeftTop, RightTop);
    //        swap(LeftBot, RightBot);
    //      }

    // Parse delta values into sign and absolute value components
    sx  = (signed)RightX   - (signed)LeftX;
    sy1 = (signed)RightTop - (signed)LeftTop;
    sy2 = (signed)RightBot - (signed)LeftBot;
    dx  = abs(sx);  sx  = sgn(sx);
    dy1 = abs(sy1); sy1 = sgn(sy1);
    dy2 = abs(sy2); sy2 = sgn(sy2);

    // Break on special cases
    if (dx == 0)
        return;

    if (Texture.FixedSize)
        {
        // Using size multiples method for texture scaling
        }

    else
        {
        // Using tiling constants method for texture scaling
        bys=(short)((long)byrs*Texture.Texture_Y_Tiling)/256;
        bxs=(short)((long)bxrs*Texture.Texture_X_Tiling)/256;
        // Effectively make it into new bitmap out of the proper amount
        // of smaller tiles
        }

    // Initalize
    ddx = 0;
    y1  = LeftTop;
    y2  = LeftBot;
    e1  = e2 =-(dx/2);

    unsigned char far *DestPtr;
    for (ddx = 0 ; ddx < dx ; ddx++)
        {
        // Map one column
        if (Texture.FixedSize)
            {
            // Using size multiples method for texture scaling
            }
        else
            {
            // Using tiling constants method for texture scaling

            bx = ((short)((long)bxs*ddx)/ dx+Texture.Texture_X_Offset) % bxrs;  // Compute current bitmap X coordinate
            dy = abs(y2-y1);    // Compute current screen dy
                                            // Note that screen dx is fixed for a trapezoid
            DestPtr=VideoBase+(SCREEN_SIZE_X*y1) + (LeftX + ddx); // Find current screen dest point
            for(ddy=0; ddy <= dy; ddy++)
                {
                by = ((short)((long)bys*ddy)/dy+Texture.Texture_Y_Offset) % byrs; // Compute current bitmap Y coordinate
                *DestPtr  = *(Texture.TextBmp.Bits + (byrs * by) + bx); // Copy the pixel
                DestPtr  += SCREEN_SIZE_X;  // Down one scanline
                }
            }

        // Use Bresnehan's Algorithm to find position of next column
        e1 += dy1;
        while(e1>0)
            {
            y1+=sy1;
            e1-=dx;
            }
        e2 += dy2;
        while(e2>0)
            {
            y2+=sy2;
            e2-=dx;
            }
        }
}

void tDisplay::DrawTextureTrapezoid(unsigned int LeftX,
                                    unsigned int LeftTop,
                                    unsigned int LeftBot,
                                    unsigned int RightX,
                                    unsigned int RightTop,
                                    unsigned int RightBot,
                                    const tTexture TextureNumber)
// Overview: This function performs identically to DrawTrapezoid,
//           except it takes a tTexture for the input texture
//           instead of a texture control structure reference.
//           This allows it to be used by functions outside the
//           tDisplay class.
// Inputs:  Screen coordinates of the trapezoid and a texture.
// Effects: Trapezoid drawn on the screen (if it's not transparent).
{
    DrawTrapezoid(LeftX, LeftTop, LeftBot, RightX, RightTop, RightBot,
                ConvertTexture(TextureNumber));
}

void tDisplay::FillScreen(tColor color)
// Overview: Fills the MCGA screen with the specified color.
// Inputs:   The color to fill the screen with
// Effect:   Screen filled with color
{
    memset(VideoBase, color, (size_t)SCREEN_SIZE_X * SCREEN_SIZE_Y);
}


void tDisplay::DrawLine(unsigned int x1, unsigned int y1,
                        unsigned int x2, unsigned int y2,
                        tColor color)
// Overview: Draws a straight line connecting the two points.  The line
//           must have valid coordinates.
// Inputs:  Two sets of screen coordinates.
//          A color to plot in.
// Effects: Line drawn between two points in the specified color.
{

    // Make sure coordinates are okay during debugging
    assert(x1 < SCREEN_SIZE_X);
    assert(y1 < SCREEN_SIZE_Y);
    assert(x2 < SCREEN_SIZE_X);
    assert(y2 < SCREEN_SIZE_Y);

    int e;                  // Error track for Bresnehan's algorithm
    unsigned int x=x1,y=y1; // x and y trackers
    unsigned char far *Dest=VideoBase+(320*(unsigned int)y)+ (unsigned int)x; // Pointer to pixel
    unsigned char far *Done=VideoBase+(320*(unsigned int)y2)+(unsigned int)x2; // Pointer to last

    // Delta values for Bresnehan
    int sx = (signed)x2 - (signed)x1;
    int sy = (signed)y2 - (signed)y1;
    unsigned int dx = abs(sx); sx=sgn(sx);
    unsigned int dy = abs(sy); sy=sgn(sy);

/*  // Clip (not)
    if (x2>319) x2=319;
    if (y2>199) y2=199;
    if (x1>319) x1=319;
    if (y1>199) y1=199;
*/

    // handle special case lines so we dont barf
    if (y1 == y2)           // Horizontal line
        {
        for (x = x1 ; x != x2 ; x += sx)
            {
            *Dest = (unsigned char)color;
            Dest  = Dest + sx;
            }
        return;
        }
    if (x1 == x2)
        {                   // Vertical line
        for(y = y1 ; y != y2 ; y += sy)
            {
            *Dest = (unsigned char)color;
            Dest  = Dest + (SCREEN_SIZE_X * sy);
            }
        return;
        }

    // For moving up and down we need 320 times the sign

    sy *= 320;
    if (dx > dy)
        {
        e = -(dx/2);  // Initialize error for dx>dy
        do
            {
            *Dest = (unsigned char)color; // Plot pixel at x,y
            Dest = Dest + sx;             // Make sure Dest tracks x
            e += dy;                      // Add to the Y error we've accululated
            while (e > 0)
                {                           // If it's time to move y...
                Dest = Dest + sy;           // Make sure Dest tracks y
                *Dest=(unsigned char)color; // Plot pixel at x,y
                e -= dx;                    // Subtract from accumulated Y error
                }
            }
        while(Dest != Done);  // Do until we arrive at destination
        }
    else
        {
        e=-(dy/2);  // Initialize error for dy>=dx
        do
            {
            *Dest = (unsigned char)color;   // Plot pixel at x,y
            Dest = Dest + sy;               // Make sure Dest tracks y
            e += dx;                        // Add to the X error we've accululated
            while(e > 0)                    // If it's time to move y...
                {
                Dest = Dest + sx;             // Make sure Dest tracks x
                *Dest = (unsigned char)color; // Plot pixel at x,y
                e -= dy;                      // Subtract from accumulated X error
                }
            }
        while(Dest != Done);  // Do until we arrive at destination
        }
}


void tDisplay::DissolveScreenToColor(tColor theColor,
                                    unsigned short int x,
                                    unsigned short int y,
                                    unsigned short int pixelNum)
// Overview: Dissolves the hardware screen to the specified color
// Inputs: The color to dissolve the screen to.
//     X and Y, paramaters to seed the linear congruential
//     randomized set generator
//     pixelNum, a constant that controls how quickly the
//     screen is dissolved
// Effects: Screen dissolved to the specified color
// Notes:
//  If x, y or pixelNum are specified as zero, default values will be
//  used that take about 3 seconds to dissolve the screen and dissolve
//  it in a suitable random-looking pattern.
//
// pixelNum:  This value is ANDed with an internal counter each time
//        a pixel is plotted.
//        The algorithm stops to wait for a VBL whenever the pixel
//        number being plotted AND pixelNum=0.
//        I.E. the default value of 0x3FF causes the routine
//        to stop and wait after every 3FFth pixel is plotted.
// X and Y: These control the "pattern" that the pixels are dissolved in.
//      Some values work, some do not.
//      The default values of 501 and 32767 produce a nice, random
//      dissolve.
{
    unsigned short int z=(unsigned short)64000L;
    unsigned long int b=0;
    unsigned char far *PlotPlace=VideoHardware;
    x=(!x)?501:x;
    y=(!y)?32767:y;
    pixelNum=(!pixelNum)?0x3FF:pixelNum;
    asm {
        push ds
        mov cx,64000
        mov si,WORD PTR b
}
DSNPix:
;
asm {
        les di,PlotPlace;
        add di,si
        mov ax,theColor
        stosb
        mov ax,cx
        and ax,pixelNum
        cmp ax,0
        jne SNoWay;
        mov dx,0x3da
}
DSWait:
;
asm {
        in al,dx;
        and al,8
        cmp al,8
        jne DSWait;
}
SNoWay:
;
asm {
        mov ax,si;
        mov bx,x
        mul bx
        clc
        add ax,y
        adc dx,0
        mov bx,z
        div bx
        mov si,dx
        loop DSNPix;
        pop ds
    }
}


void tDisplay::GetPaletteTo(unsigned char far *thePal)
// Overview: Copies the hardware VGA palette to the supplied buffer.
// Input: A far pointer to the buffer
// Returns: Buffer filled with RGB triples
// Notes: Buffer MUST be at least 768 bytes long.
{
    asm {
        les dx,thePal
        mov ax,0x1017
        mov bx,0
        mov cx,256
        int 0x10
        }
}

void tDisplay::SetPaletteTo(unsigned char far *thePal)
// Overview: Sets the VGA palette to the supplied palette.
// Input: A far pointer to a list of 256 UCHAR RGB triples.
//    (Byte) Red intensity from 0 to 63
//    (Byte) Green intensity from 0 to 63
//    (Byte) Blue intensity from 0 to 63
//    ... Repeat 256 times
// Returns: Nothing
// Effects: Palette changed to specified palette.
// Notes: Buffer should contain sufficient, and in-range, palette values.
//    Behavior is unpredictable if numbers outside the range 0 to 63
//    are specified.
{
    asm {
        les dx,thePal
        mov ax,0x1012
        mov bx,0
        mov cx,256
        int 0x10
        }
}


tBool tDisplay::PlotBitmap(struct bitmap &bm,
                             unsigned int x,
                             unsigned int y)
// Overview: Plots the specified bitmap at the specified screen location
// Inputs: A bitmap to blot
//     The screen coordinates of the upper-left corner
// Returns: TRUE if successful
//      FALSE if bitmap could not be plotted
{
    unsigned int x1, y1, x2, y2; // Coordinates of screen area to plot
    unsigned int XSkip=0;          // X Skip Factor (if bitmap goes off screen)
    unsigned int YSkip;      // Value to add to SourcePtr to position it
                                 // at the beginning of the next scanline.
    int ly, lx;
    unsigned char far *SourcePtr=bm.Bits;
    unsigned char far *DestPtr=VideoBase;

    x1=x; y1=y;

    // Bitmap too small to plot
    if ((bm.Y_Size<1) || (bm.X_Size<1))
        return(False);

    x2 = x + bm.X_Size - 1;
    y2 = y + bm.Y_Size - 1;

    if (y1>199) return(False); // Cant plot off the bottom of the screen!
    if (y2>199) y2=199; // Clip to bottom row of screen

    if (x1>319) return(False); // Cant plot off the right edge of the screen!
    if (x2>319) {
    XSkip=x2-319; // We need to skip one data point in the X direction
                    // for every one we clip off the final image
    x2=319;
    }

    DestPtr+=(320*y1)+x1;

    YSkip=319-(y2-y1);  // Y Skip value

    for(ly=y1; ly<=y2; ly++)
        {
        for(lx=x1; lx<=x2; lx++)
            {
            *DestPtr = *SourcePtr;
            SourcePtr++; DestPtr++;  // This'll run awesome on Pentium...
                            // Faster to use memcpy()
                            // faster still to use inline ASM (less pointer
                                                                // arithmetic)
            }
        SourcePtr += XSkip;  // Skip any bits we didn't plot
        DestPtr += YSkip;      // Move to X1 on next scanline
        }

    return(True);
}


void tDisplay::DrawRect(unsigned int x1, unsigned int y1,
                        unsigned int x2, unsigned int y2,
                        tTexture Border, tTexture Fill,
                        tRectMethod Method)
// Overview: Draws a rectangle on the screen.
//       The rectangle can have any texture for a border
//       and fill value, including any color, transparent,
//       or any texture.
//       If a texture is used you have the choice of specifying
//       ScaleRect or TileRect for Method;
//       if TileRect is used, the bitmap for the appropriate
//       texture will be kept it's natural size, but will be tiled
//       or cropped as necessary to fill the rectangle.
//       If ScaleRect is used, the bitmap for the appropriate
//       texture will be scaled to fit the rectangle size.
//
//       Border supports transparent or color tTextures.
//
//       Fill supports transparent, color, or textured tTextures.
//
// Inputs: The screen coordinates of the rectangle
//     The tTextures to fill the border and inside of the rect with
//     The tRectMethod (if a texture was specified for the inside)
// Effects: Rectangle plotted on the VideoBase
{
    unsigned int x, y, dx, dy;
    unsigned char far *Scan;
    unsigned char far *Dest;
    unsigned int bx, by;
    unsigned int bdx, bdy;
    unsigned char far *BitPtr;
    unsigned char far *BitScan;

    if (x2<x1) swapInt(x1,x2); // Swap if x2 is left of x1
    if (y2<y1) swapInt(y1,y2); // Swap if y2 is above y1

    dx=x2-x1+1; // Imagine this as width of line in pixels
    dy=y2-y1+1;
    Scan=VideoBase+(320*y1)+x1;

    if (between(Fill,1,255)) {
    // Fill is with solid color
    for(y=y1;y<=y2;y++) {
        memset(Scan,Fill,dy);
        Scan+=320;
    }
    } else if (Fill>255) {
    // Fill with texture

    // First get the texture number

    struct texture &text=ConvertTexture(Fill);
    bdx=text.TextBmp.X_Size;
    bdy=text.TextBmp.Y_Size;

    if (bdx>0) { // If theres a bitmap to plot...
        if (Method==ScaleRect) {
        // Scale bitmap to fit rectangle
        Dest=Scan;
        for(y=0;y<=dy;y++) {
            by=(short)((long)y*bdy)/dy;
            BitPtr=text.TextBmp.Bits+(by*bdy);
            for(x=0;x<=dx;x++) {
            bx=(short)((long)x*bdx)/dx;
            BitScan=BitPtr+bx;
            (*(Dest))=(*(BitScan));
            Dest++;
            }
            Dest=(Scan+=320);
        }
        } else {
        // Tile bitmap within rectangle
        Dest=Scan;
        for(y=0;y<=dy;y++) {
            by=y % bdy;
            BitPtr=text.TextBmp.Bits+(by*bdy);
            for(x=0;x<=dx;x++) {
            bx=x % bdx;
            BitScan=BitPtr+bx;
            (*(Dest))=(*(BitScan));
            Dest++;
            }
            Dest=(Scan+=320);
        }
        }
    }
    }

    // Of course, if fill was transparent we didn't want to do anything
    // anyway, and so we didn't!

    // The border can be either transparent or a solid color
    if (between(Border,1,255)) {
    Dest=VideoBase+(320*y2)+x1;
    memset(Dest,(unsigned char)Border,(size_t)dx);
    Dest=VideoBase+(320*y1)+x1;
    memset(Dest,(unsigned char)Border,(size_t)dx);
    for(y=0;y<=dy;y++) {
        (*(Dest))=(unsigned char)(Border);
        (*(Dest+dx-1))=(unsigned char)(Border);
        Dest+=320;
    }
    }

}


struct texture &tDisplay::ConvertTexture(tTexture TextureNumber)
// Overview: Used to convert a tTexture number, such as that supplied by
//       external functions, into a texture control structure reference
//       required by DrawTrapezoid and other tDisplay member functions.
// Inputs: The tTexture to convert
// Returns: A reference to the corresponding texture control structure.
{
    switch(TextureNumber) {
    case stone1:   return(stone1Tx);
    case brick1:   return(brick1Tx);
    case starry:   return(starryTx);
    case marblue1: return(marblue1Tx);
    case brick2:   return(brick2Tx);
    case wood1:    return(wood1Tx);
    case wood2:    return(wood2Tx);
    case transparent:
        colorTx.Color=Black;
        return(colorTx);
    default:
        colorTx.Color=(tColor)(TextureNumber % 256);
        return(colorTx);
    }
}


void tDisplay::SetDoubleBuffer(tBool dbl)
// Overview: Used to turn double buffering on and off.
//       When double buffering is on, all graphics operations
//       are performed on the hidden display page.
//       The screen is not displayed until you call Update()
//       When double buffering is off, all graphics operations
//       draw directly onto the screen.
// Inputs: True if double buffering should be enabled
//     False if double buffering should be disabled
{
    if (dbl)
        VideoBase=VideoBuff;
    else
        VideoBase=VideoHardware;
}


void tDisplay::Update(void)
// Overview: Copies the contents of the hidden display buffer
//           onto the screen.
{
    CopyPage(VideoBuff, VideoHardware);
}


void tDisplay::ScreenScrollUp(unsigned int NumLines)
// Overview: Scrolls the VideoBase upward the specified number of
//           scanlines
// Inputs: The number of scanlines to scroll up
{
    unsigned char far *DestLine;
    unsigned char far *SourceLine;
    unsigned int y;

    if (NumLines==0) return; // Nothing to scroll

    if (NumLines<200) {    // If we aren't clearing the entire screen...
    DestLine=VideoBase;      // Copy from lines below to lines above
    SourceLine=VideoBase+(320*NumLines);
    for(y=0;y<(200-NumLines);y++) {
        memcpy(DestLine,SourceLine,(size_t)320); // Copy a line
        SourceLine+=320;
        DestLine+=320;
    }
    }

    // Clear the stuff we have scrolled up
    memset(DestLine,BackgroundColor,(size_t)(320*NumLines));
}

void tDisplay::GotoXY(unsigned int x, unsigned int y)
// Overview: Sets the graphic cursor location.
// Inputs: The screen coordinates to set the cursor to
// Notes: Input will be clipped to acceptible values
{
    // Ensure that the next character plotted will not extend off the
    // screen
    if (y < CurrentFont.BaseHeight)
        y = CurrentFont.BaseHeight;
    if (y > 199 - (CurrentFont.MaxHeight - CurrentFont.BaseHeight))
        y= 199 - (CurrentFont.MaxHeight - CurrentFont.BaseHeight);
    if (x > 318)
        x = 318;
    TextX = x;
    TextY = y;
}


void tDisplay::GetCursorLocation(unsigned int &x, unsigned int &y)
// Overview: Stores the current cursor location in the supplied
//       integer references.
{
    x=TextX; y=TextY;
}


void tDisplay::CarriageReturn(void)
// Overview: Performs a carriage return and linefeed.
//       Screen will be scrolled if necessary if we are allowed to
// Effects: Text cursor moved to beginning of next line.
{
    TextX=0; TextY+=CurrentFont.MaxHeight+1;
    if (TextY>199-(CurrentFont.MaxHeight-CurrentFont.BaseHeight)) {
    // Screen needs to scroll
    if (AutoScroll) {
        // We're allowed to scroll automatically, so go for it
        ScreenScrollUp(CurrentFont.MaxHeight);
        TextY-=CurrentFont.MaxHeight;
    } else {
        // Cant scroll, so let the cursor wrap around bottom line
        TextY=199-(CurrentFont.MaxHeight-CurrentFont.BaseHeight);
    }
    }
}


void tDisplay::OutChar(unsigned char c)
// Overview: Outputs the specified character to the current cursor
//       location and in the current font
// Inputs: The character to output
// Effects: Character drawn on VideoBase
//      Cursor position updated
{
    // If theres no room to plot this character on this line...
    if ((CurrentFont.Ch[c].XSize+TextX)>319) {
    // Carriage return and linefeed
    CarriageReturn();
    }
    if (c=='\n') { // If carriage return character..
    CarriageReturn(); // Perform it
    } else if (c==32) {        // If it's a space just move over
    TextX+=CurrentFont.AvgWidth/2;
    } else if (CurrentFont.Ch[c].Bytes!=NULL) {  // But if theres a bitmap...
    // If textborder is true we do funky stuff first
    if (TextBorder) {
        DrawMonoChar(TextX-1,TextY-1,Black,c,CurrentFont);
        DrawMonoChar(TextX+1,TextY-1,Black,c,CurrentFont);
        DrawMonoChar(TextX-1,TextY+1,Black,c,CurrentFont);
        DrawMonoChar(TextX+1,TextY+1,Black,c,CurrentFont);
    }
    DrawChar(TextX, TextY, c, CurrentFont);   // draw it...
    TextX+=CurrentFont.Ch[c].XSize+1;   // and move over.
    }
}


void tDisplay::DrawChar(unsigned int x,
                        unsigned int y,
                        unsigned char c,
                        tColorFont &font)
// Overview: Draws a character on the VideoBase at the specified
//       coordinates in the specified font
// Inputs: The coordinates to plot
//     The character to plot there
//     A reference to the font to use
// Effects: Character drawn on VideoBase
{
    unsigned char far *Dest;
    unsigned char far *Source=font.Ch[c].Bytes;
    unsigned char w=font.Ch[c].XSize,
                h=font.Ch[c].YSize;
    unsigned int sy, ey;

    if ((h==0)||(w==0)) return; // Can't draw a nonexistant character

    sy=(y-font.BaseHeight+1); // Starting Y
    ey=sy+h-1;        // Ending Y

    Dest=VideoBase+(320*sy)+x;

    if (TextSprite) { // If text plotting is in "sprite" mode...
    for(y=sy; y<=ey; y++) {
        for(x=1; x<=w; x++) {
        register unsigned char b=(*(Source));
        if (b) (*(Dest))=b+TextColor;
        Dest++;
        Source++;
        }
        Dest+=(320-w);
    }
    } else {
    // Text always plotted on it's own background (usually black)
    // and color translation ignored
    for(y=sy; y<=ey; y++) {
        memcpy(Dest,Source,(size_t)w); // Plot this line
        Dest+=320; // Move down one screen scanline
        Source+=w; // Move down to next bitmap line
    }
    }
}


void tDisplay::DrawMonoChar(unsigned int x,
                            unsigned int y,
                            unsigned char col,
                            unsigned char c,
                            tColorFont &font)
// Overview: Draws a character on the VideoBase at the specified
//       coordinates in the specified font such that any
//       non-transparent pixel is plotted in color col
// Inputs: The coordinates to plot
//     The color to plot it in
//     The character to plot there
//     A reference to the font to use
// Effects: Character drawn on VideoBase in the specified color
{
    unsigned char far *Dest;
    unsigned char far *Source=font.Ch[c].Bytes;
    unsigned char w=font.Ch[c].XSize,
                h=font.Ch[c].YSize;
    unsigned int sy, ey;

    if ((h==0)||(w==0)) return; // Can't draw a nonexistant character

    sy=(y-font.BaseHeight+1); // Starting Y
    ey=sy+h-1;        // Ending Y

    Dest=VideoBase+(320*sy)+x;

    // Monochar always plotted in sprite mode (for obvious reasons)
    for(y=sy; y<=ey; y++) {
    for(x=1; x<=w; x++) {
        register unsigned char b=(*(Source));
        if (b) (*(Dest))=col;
        Dest++;
        Source++;
    }
    Dest+=(320-w);
    }
}


void tDisplay::OutString(char *str)
// Overview: Outputs all characters in the string, updating cursor
//       position as necessary, until a NULL character.
// Inputs: The string to draw
{
/*  If we're going to play with pointers the FOR loop is probably nicer...
    char *s=str;
    while(*s)
        {
        OutChar(*(s));
        s++;
        }
*/
    for (char *s = str ; *s ; s++)
        OutChar(*s);  
}

unsigned int tDisplay::StrWidth(char *str)
// Overview: Finds the width of a string in pixels.
// Inputs: The string to examine
// Returns: Sum of the widths of it's component characters,
//      as well as the inter-character spacing distances.
{
    unsigned char c;
    tBool Nudged=False;  // If True, last character checked
                         // included a spacing distance
    int wid=0;
    while(*(str)) {
    c=(*(str));
    if (c==32) {
        wid+=CurrentFont.AvgWidth/2;
        Nudged=False;
    } else if (CurrentFont.Ch[c].XSize>0) {
        wid+=CurrentFont.Ch[c].XSize+1;
        Nudged=True;
    }
    str++;
    }
    if (Nudged) wid--;   // Spacing distance at the end of the string shouldn't
                     // count
    return(wid);
}


void tDisplay::CenterString(char *str)
// Overview: Centers the string horizontally
// Inputs: The string to center
{
    unsigned int wid;

    wid=StrWidth(str);
    GotoXY((unsigned int)(160-(wid/2)),YCursor());
    OutString(str);
}


void tDisplay::DissolveToVideoMemory(unsigned int x,
                                     unsigned int y,
                                     unsigned int pixelNum)
// Overview: Dissolves the contents of the hidden video buffer
//       on to the physical screen.
// Inputs: Same as DissolveScreenToColor, except the color argument
//     isn't required.
// Effects: Contents of hidden buffer copied to screen.
{
    unsigned short int z=(unsigned short)64000L;
    unsigned long int b=0;
    unsigned char far *PlotPlace=VideoHardware;
    unsigned char far *FromPlace=VideoBuff;
    x=(!x)?501:x;
    y=(!y)?32767:y;
    pixelNum=(!pixelNum)?0x3FF:pixelNum;
    asm {
        push ds
        mov cx, 64000
        mov si,WORD PTR b
        }
DSVMPix:
;
asm {
        mov bx,si
        les di,PlotPlace
        lds si,FromPlace
        add di,bx
        add si,bx
        movsb
        mov si,bx
        mov ax,cx
        and ax,pixelNum
        cmp ax,0
        jne DSVMNoWay
        mov dx,0x3da
}
DSVMWait:
;
asm {
        in al,dx;
        and al,8
        cmp al,8
        jne DSVMWait;
}
DSVMNoWay:
;
asm {
        mov ax,si;
        mov bx,x
        mul bx
        clc
        add ax,y
        adc dx,0
        mov bx,z
        div bx
        mov si,dx
        loop DSVMPix;
        pop ds
    }
}

void tDisplay::ToggleMode(void)
// Overview: Cycles through each available display mode.
// Effects: The display mode is changed, but the display is not updated.
{
    switch (DisplayMode)
        {
        case Overhead :
            SetMode(Clipped);
            break;
        case Clipped :
            SetMode(Wireframe);
            break;
        case Wireframe:
            SetMode(Solid);
            break;
        case Solid:
            SetMode(ThreeD);
            break;
        case ThreeD:
            SetMode(Overhead);
            break;
        default :
            assert("Bad DisplayMode!");
            break;
        }
}

/* -----------------------------------------------------------------------
    Overloaded stubs of functions.  Helps keep casts in one place.
----------------------------------------------------------------------- */

void tDisplay::DrawLine(tCoord x1, tCoord y1, tCoord x2, tCoord y2, tColor color)
{
  DrawLine((unsigned int)x1, (unsigned int)y1, (unsigned int)x2,
             (unsigned int)y2, color);
}

void tDisplay::DrawTextureTrapezoid(tCoord LeftX, tCoord LeftTop, tCoord LeftBot,
                                    tCoord RightX, tCoord RightTop, tCoord RightBot,
                                    const tTexture TextureNumber)
{
    DrawTextureTrapezoid((unsigned int) LeftX,      (unsigned int) LeftTop,
                         (unsigned int) LeftBot,    (unsigned int) RightX,
                         (unsigned int) RightTop,   (unsigned int) RightBot,
                         TextureNumber);
}


void tDisplay::DrawTrapezoid(tCoord LeftX,
                             tCoord LeftTop,
                             tCoord LeftBot,
                             tCoord RightX,
                             tCoord RightTop,
                             tCoord RightBot,
                             const struct texture &Texture)
{
    DrawTrapezoid((unsigned int)LeftX, (unsigned int)LeftTop,
                  (unsigned int)LeftBot, (unsigned int)RightX,
                  (unsigned int)RightTop, (unsigned int)RightBot,
                  Texture);
}

