/************************************************************************

    File:      display.h
    Contents: definition of tDisplay class
    Purpose:  Low level display driver
              3d projection and display routines
    Author:   Mike Lyons
    Overview: This class represents the program's display subsystem.
    Interface:  tDisplay::SetMode(tMode newmode)
               tDisplay::Display(tPlayer whichview)
    where tMode is defined in this file, and tPlayer is defined in basic.h

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Sheldon Young           (604)963-7067
    PO Box 2931             FidoNet:  Sheldon Young 1:359/197.1
    Prince George, BC       Internet: sheldon.young@datex.com
    V2N 4T7
    Canada

    Mike Lyons              (604)964-0725
    7833 Piedmont Crescent  FidoNet: Mike Lyons 1:359/262.1
    Prince George, BC       Internet: mike.lyons@datex.com
    V2N 3K8
    Canada

---------------------------------------------------------------------- */

#ifndef _DISPLAYHEADER
#define _DISPLAYHEADER

#include <stdlib.h>
#include <mem.h>

#include "player.h"

// Include our group's common declaration set
#include "basic.h"

// Include the font declarations
#include "font.h"

// Bring in balloonFont from balloon.cpp
extern tColorFont balloonFont;

// Bitmap control structure
struct bitmap {
  unsigned int X_Size;
  unsigned int Y_Size;
  unsigned char far *Bits;
};

enum tRectMethod {TileRect, ScaleRect};

// Texture control structure
// NOTE: Not to be confused with tTexture, which is the interface standard
//       for storing textures within the tWall class.
// This structure form is used internally by tDisplay
struct texture {
    tColor Color;           // Default color to use if no mapping enabled
    tBool UseTexture;       // TRUE=map a texture FALSE=solid color
    struct bitmap TextBmp;  // The bitmap to map with if mapping is enabled
    int Texture_X_Offset;   // X and Y texture offsets
    int Texture_Y_Offset;
    tBool FixedSize;        // TRUE=use X and Y Size multiples
                            // FALSE=use X and Y tiling constants

    int Texture_X_Multiple; // Controls "chunkiness" of mapped texture
    int Texture_Y_Multiple;

    // For Size Multiples method:
    //
    // A size multiple of 256 is the standard bitmap size.
    //
    // It will keep the textures from looking too chunky at close range
    // and too "spotty" (scaled down to the point of no detail remaining)
    // at far range.
    //
    // Using this method guarantees that the scaled pixels will always
    // appear to be the same size on the screen.  Texture alignment
    // becomes more difficult, however.

    int Texture_X_Tiling;   // Alternate approach: with this method of
    int Texture_Y_Tiling;   // texture scaling, the texture is forced to
                            // fit within the enclosing shape this many
                            // times.

    // For Tiling Constant method:
    //
    // Texture_X_Tiling controls the number of times the bitmap fits
    // across the enclosing shape.
    //
    // A value of 256=1; thus, if Texture_X_Tiling and Texture_Y_Tiling
    // are both set to 256, the texture will appear just once, exactly
    // scaled to fit within the enclosing shape.
    //
    // This method allows you to have fractional scaling values...
    // 128 means fit half the bitmap, 64 means 1/4, etc.
    //
    // This method makes it easy to have a texture (say a picture of a door)
    // exactly fill the enclosing wall.
    //
    // The disadvantage is that different size walls will make for
    // different apparant scaling resolutions.

    // Feel free to mix-n'-match texture scaling methods!
};

// Definition of tMode type
enum tMode {Overhead = 0, ThreeD, Wireframe, Solid, Clipped};

// Definition of tDisplay error return values
enum tDisplayResult {DISPLAY_OK=0, DISPLAY_BADMOD};

// Definition of tDisplay class
class tDisplay {

public:

  // Constructor: Sets video mode to graphics and allocates display memory
  tDisplay(void);

  // Destructor: Sets video mode to text and deallocates display memory
  ~tDisplay(void);

  // Fills the VideoBase with the specified color
  void FillScreen(tColor color);

  // Draws a line on the VideoBase
  void DrawLine(unsigned int x1, unsigned int y1,
                unsigned int x2, unsigned int y2,
                tColor color);
  void DrawLine(tCoord x1, tCoord y1,
                tCoord x2, tCoord y2,
                tColor color);

 // Clears the VideoBase to the current background color
  inline void ClearScreen(void) { FillScreen(BackgroundColor); }

    // Copies the a video page from the source to dest
    inline void CopyPage(unsigned char far *source, unsigned char far *dest)
        {
        memcpy(dest, source, (size_t)SCREEN_SIZE_X*SCREEN_SIZE_Y);
        }

    // Plots a point at VideoBase in the given color
    inline void PlotPoint(const unsigned int x, const unsigned int y, const tColor color)
        {
        (*(VideoBase + x + (SCREEN_SIZE_X*y))) = color;
        }

  // Dissolves the hardware screen to the specified color
  // (ignoring VideoBase and going directly to VideoHardware)
  void DissolveScreenToColor(tColor theColor,
                             unsigned short int x,
                             unsigned short int y,
                             unsigned short int pixelNum);

  // Sets the VGA hardware palette to the specified palette
  void SetPaletteTo(unsigned char far *thePal);

  // Retrieves the VGA hardware palette to the specified buffer
  void GetPaletteTo(unsigned char far *thePal);

  // Draws a frame on the VideoBase
  void DrawRect(unsigned int x1, unsigned int y1,
                unsigned int x2, unsigned int y2,
                tTexture Border, tTexture Fill,
                tRectMethod Method);

  // Draws a texture-mapped trapezoid on the VideoBase.
  // Accepts a tTexture for the texture value.
  void DrawTextureTrapezoid(unsigned int LeftX,
                            unsigned int LeftTop,
                            unsigned int LeftBot,
                            unsigned int RightX,
                            unsigned int RightTop,
                            unsigned int RightBot,
                            const tTexture TextureNumber);
  void DrawTextureTrapezoid(tCoord LeftX,
                            tCoord LeftTop,
                            tCoord LeftBot,
                            tCoord RightX,
                            tCoord RightTop,
                            tCoord RightBot,
                            const tTexture TextureNumber);

    // Draws a solid color trapezoid
    void DrawSolidColorTrapezoid(unsigned int LeftX,
                                 unsigned int LeftTop,
                                 unsigned int LeftBot,
                                 unsigned int RightX,
                                 unsigned int RightTop,
                                 unsigned int RightBot,
                                 const tColor Color);

  // Copies the contents of the hidden page onto the screen
  void Update(void);

  // Turns double buffering on and off (on by default)
  void SetDoubleBuffer(tBool dbl);

  // Draws a character on the VideoBase at the current location
  // and in the current font.  Current location is advanced.
  void OutChar(unsigned char c);

  // Calls OutChar() for every character in the AsciiZ string str
  void OutString(char *str);

  // Scrolls the VideoBase up one or more scanlines
  void ScreenScrollUp(unsigned int NumLines);

  // Performs a text carriage return
  void CarriageReturn(void);

  // Sets the text cursor location in pixel increments
  void GotoXY(unsigned int x, unsigned int y);

  // Returns the current text cursor location in pixels
  void GetCursorLocation(unsigned int &x, unsigned int &y);

  // Cursor X position
  inline unsigned int XCursor(void) { return(TextX); }

  // Cursor Y position
  inline unsigned int YCursor(void) { return(TextY); }

  // Set text color
  inline void SetTextColor(unsigned char c) { TextColor=c*16; }

  // Returns the width of a string in pixels
  unsigned int StrWidth(char *str);

  // Centers the string horizontally on the current line
  void CenterString(char *str);

  // Dissolves the contents of the video buffer onto the screen
  void DissolveToVideoMemory(unsigned int x,
                             unsigned int y,
                             unsigned int pixelNum);

  // Sets the AutoScroll variable
  inline void SetAutoScroll(tBool s) { AutoScroll=s; }

  // Retreives the AutoScroll setting
  inline tBool GetAutoScroll(void) { return(AutoScroll); }

  // Sets the current display mode
  inline void SetMode(tMode newmode) { DisplayMode=newmode; }

  // Toggle the current display mode
  void ToggleMode(void);

  // Displays the view in the current mode for the specified player
  void Display(tPlayer whichview);

  // Sets the TextBorder property
  inline void SetTextBorder(tBool b) { TextBorder=b; }

  // Clips coordinates of a line to the visible screen area
  tBool ClipToScreen(long &x1, long &y1,
                     long &x2, long &y2);

  // Plots a line that may contain out of bounds coordinates.
  tBool DrawClippedLine(long x1, long y1,
                        long x2, long y2,
                        tColor color);

  // Plots a bitmap on the VideoBase at the specified location.
  // Because it requires a reference to a bitmap control structure
  // DrawRect is generally more usefull.
  tBool PlotBitmap(struct bitmap &bm,
                   unsigned int x,
                   unsigned int y);


private:
  // Pointer to whatever video buffer we are drawing on
  unsigned char far *VideoBase;

  // Pointer to our allocated video buffer
  unsigned char far *VideoBuff;

  // Reference to the current font (set to balloonFont by default)
  struct tColorFont CurrentFont;

  // Whether or not to put a border around each character
  tBool TextBorder;

  // Location of text cursor
  unsigned int TextX, TextY;

  // Current text color offset (set to 0 by default)
  unsigned char TextColor;

  // Should holes in text be transparent(true) or black?
  // Set to true by default, but text plots faster if false
  // Color translation is ignored unless this is true
  tBool TextSprite;

  // Whether or not to scroll the display automatically when text
  // reaches the bottom of the screen (set to False by default)
  tBool AutoScroll;

  // Background and foreground colors so we know what colors to
  // use for certain operations (BG for clearing the screen and scrolling)
  tColor BackgroundColor, ForegroundColor;

  // DisplayMode can be either Overhead or ThreeD.  Set Overhead by default.
  tMode DisplayMode;

  // Draws a texture-mapped trapezoid on the VideoBase.
  // Requires a reference to a texture control structure for the
  // texture, rendering this function unusable to a function outside
  // the tDisplay class.  External functions should call
  // DrawTextureTrapezoid() instead.
  void DrawTrapezoid(unsigned int LeftX,
                     unsigned int LeftTop,
                     unsigned int LeftBot,
                     unsigned int RightX,
                     unsigned int RightTop,
                     unsigned int RightBot,
                     const struct texture &Texture);
  void DrawTrapezoid(tCoord LeftX,
                     tCoord LeftTop,
                     tCoord LeftBot,
                     tCoord RightX,
                     tCoord RightTop,
                     tCoord RightBot,
                     const struct texture &Texture);

  // Converts a tTexture into a Texture Control Structure reference.
  struct texture &ConvertTexture(tTexture TextureNumber);

  // Draws a character on the screen at the specified coordinates
  void DrawChar(unsigned int x,
                unsigned int y,
                unsigned char c,
                tColorFont &font);

  // Draws a character in the specified color
  void DrawMonoChar(unsigned int x,
                    unsigned int y,
                    unsigned char col,
                    unsigned char c,
                    tColorFont &font);

  // Draws the overhead map view for the given player on VideoBase
  void DrawOverheadView(tPlayer whichview);

  // Draws the 3d view for the given player on videobase
  void Draw3DView(tPlayer Player);

};


#endif
