/*************************************************************************

    File:     display2.cpp
    Contents: Code for tDisplay class, including low level graphics driver
              and 3D projection and plot routines
              This file contains the other half of the code.
    Author:   Mike Lyons with modifications by Sheldon Young

    Project: 3D Lab, Virtual Reality Walk-Through
    Class:   CNC CSC 216, Spring, 1994
    Prof:    George Kaweesi

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 1, or (at your option)
    any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

    Sheldon Young           (604)963-7067
    PO Box 2931             FidoNet:  Sheldon Young 1:359/197.1
    Prince George, BC       Internet: sheldon.young@datex.com
    V2N 4T7
    Canada

    Mike Lyons              (604)964-0725
    7833 Piedmont Crescent  FidoNet: Mike Lyons 1:359/262.1
    Prince George, BC       Internet: mike.lyons@datex.com
    V2N 3K8
    Canada

*************************************************************************/

#include <stdlib.h>
#include <assert.h>
#include <limits.h>
// Included needed definitions
#include "maze.h"
#include "grafdrvr.h"

/* -----------------------------------------------------------------------
    Globals
----------------------------------------------------------------------- */

// Define some constants for clipping routines
#define cRight  1
#define cXMid   2
#define cLeft   4
#define cBelow  8
#define cYMid  16
#define cAbove 32

// Hard coded distances
#define DistanceToScreen 100L
#define ClippingDistance 200L

#define XMapLow -8000
#define XMapHigh 8000
#define YMapLow 8000
#define YMapHigh -8000

// Need to access some internals of the maze.  A little messy, but
// not too horrible if we consider tDipslay a partner of tMaze.
extern tMaze Maze;

/* -----------------------------------------------------------------------
    Utility functions that aren't part of the tDisplay class.
----------------------------------------------------------------------- */

// Overview:  The compare function for qsort() during depth sorting.
// Inputs:    Pointers to two tWallSortStructs
// Returns:    0 if equal.
//            -1 if A belongs in front of B
//            +1 if B belongs in front of A
// Notes:     Used internally during the depth sort
int DistanceCompare(const struct tWallSortStruct *a,
                    const struct tWallSortStruct *b)
{
    tCoord Delta = (a->Distance - b->Distance);
    if (Delta > 0)
        return(-1);
    else if (Delta < 0)
        return(+1);
    return(0);
}

// Overview: Maps p, which is in the range [Low1, Hight1] into the
//           range [Low2, High2] instead.
// Inputs:   Point to map, it's native range and the desired new range.
// Returns:  The points position in the new range.
inline long MapPoint(long p, long Low1, long High1, long Low2, long High2)
{
    return(((p-Low1) * (long)(High2 - Low2))/(High1 - Low1)+Low2);
}


// Overview: Rotates point around the origin by (a) pseudodegrees.
// Inputs:   Point to rotate and rotation angle.
// Effect:   Stores new point in rx and ry.
inline void Rotate2D(const long x, const long y,
                     long &rx, long &ry,
                     tAngle a)
{
    tFract sr = Sine(a), cr = Cosine(a);
    rx = (x*cr - y*sr)/SHRT_MAX;
    ry = (x*sr + y*cr)/SHRT_MAX;
}

// Overview: Rotate world around player.
// Effects:  Translate vertices such that Player's position becomes the origin,
//           then rotate vertices such that Player's view angle becomes
//           zero degrees (pointing east).
// Inputs:   Player - Player to rotate around.
//           NumberVertices - Number of vertices to rotate.
//           Source - Source array to rotate.
//           Destination - Where to place the results.
void Rotate2DArray(const tPlayer &Player, const int NumberVertices,
                   const tPoint Source[], tPoint Destination[])
{
    int i;
    tFract sr = Sine(-Player.ViewAngle), cr = Cosine(-Player.ViewAngle);
    for(i = 0 ; i < NumberVertices ; i++)
        {
        Destination[i].x = ((Source[i].x - Player.Location.x)*cr -
                            (Source[i].y - Player.Location.y)*sr)/SHRT_MAX;
        Destination[i].y = ((Source[i].x - Player.Location.x)*sr +
                            (Source[i].y - Player.Location.y)*cr)/SHRT_MAX;
        }
}

inline long Sector(long x, long y, long x1, long y1, long x2, long y2)
// Overview: Returns the sector number of point x,y relative
//           to the box x1,y1 to x2,y2.
// Inputs:  The point to check and the box to check against.
// Returns: The sector number as follows:
//
//          cLeft + cAbove | cXMid + cAbove | cRight + cAbove
//          ---------------+----------------+-----------------
//          cLeft + cYMid  | cXMid + cYMid  | cRight + cYMid
//          ---------------+----------------+-----------------
//          cLeft + cBelow | cXMid + cBelow | cRight + cBelow
//
//          So to check if it is in the top right sector the statment
//          would be "if (Sector == (cRight + cAbove))".
{
    long Sector;

    if(x2<x1) {
        long foo=x2;
        x2=x1;
        x1=foo;
    }

    if(y2<y1) {
        long foo=y2;
        y2=y1;
        y1=foo;
    }

    // Set horizontal bit
    if (x < x1)
        Sector = cLeft;
    else if (x > x2)
        Sector = cRight;
    else
        Sector = cXMid;

    // Return with OR'd vertical bit
    if (y < y1)
        return(Sector | cAbove);
    else if (y>y2)
        return(Sector | cBelow);
    else
        return(Sector | cYMid);
}

tBool PointInRect(long x, long y, long rx1, long ry1, long rx2, long ry2)
{
    if ((x >= rx1) &&
        (x <= rx2) &&
        (y >= ry1) &&
        (y <= ry2))
        return(True);
    else
        return(False);
}

tBool tDisplay::ClipToScreen(long &x1, long &y1, long &x2, long &y2)
// Overview: Performs actual clipping on the input coordinates
//       to fit them all onto the visible screen area without
//       changing the line's shape
// Input: The line segment to clip, which may extend past the visible
//    screen in the x and y directions.
// Returns: No error return, as clipping is always successful.
//      Returns FALSE if line was completely off the screen
//      (x1,y1,x2,y2 unchanged)
//      Returns TRUE if line is visible
//      (x1,y1 to x2,y2 now describe the visible portion of the line)
// Notes: This algorithm sucks.  Can anyone think of a less ugly way?
//    Supports nine sectors a point can be in.
//    Contains rules for clipping each sector.
{
    // Sector numbers
    long Sector1 = Sector(x1,y1,0,0, SCREEN_SIZE_X-1, SCREEN_SIZE_Y-1);
    long Sector2 = Sector(x2,y2,0,0, SCREEN_SIZE_X-1, SCREEN_SIZE_Y-1);

    // Easy cases using sector numbers
    if (((Sector1 & cLeft)  && (Sector2 & cLeft))  ||             // Totally off
        ((Sector1 & cRight) && (Sector2 & cRight))   ||
        ((Sector1 & cAbove) && (Sector2 & cAbove))   ||
        ((Sector1 & cBelow) && (Sector2 & cBelow)))
        return(False);
    if ((Sector1 == (cXMid+cYMid)) && (Sector2 == (cXMid+cYMid))) // Totally on
        return(True);

    // Compute slope
    tCoord dx, dy;
    dx = x2 - x1;
    dy = y2 - y1;
    if (dx == 0) {                              // Vertical
        if(y1<0) y1=0;
        if(y2<0) y2=0;
        if(y1>199) y1=199;
        if(y2>199) y2=199;
        return(True);
        }
    if(dy==0) {                                 // Horizontal
        if(x1<0) x1=0;
        if(x2<0) x2=0;
        if(x1>319) x1=319;
        if(x2>319) x2=319;
        return(True);
    }

    // Line is nonvertical so we can find its intercept
    long tx1=x1, ty1=y1, tx2=x2, ty2=y2;
    long b = y1 - dy*x1/dx;
    if (Sector1 & cAbove)                        // Push Sector1 to top
        {
        tx1=0 - b*dx/dy;
        ty1=0;
        Sector1=Sector(tx1,ty1,0,0,319,199);
        }
    else if (Sector2 & cAbove)                  // Push Sector2 to top
        {
        tx2=0 - b*dx/dy;
        ty2=0;
        Sector2=Sector(tx2,ty2,0,0,319,199);
        }
    if ((Sector1 & cAbove) && (Sector2 & cAbove))
        return (False);

    if (Sector1&cBelow)                         // Push Sector1 to bottom
        {
        tx1=0-(b-199)*dx/dy;
        ty1=199;
        Sector1=Sector(tx1,ty1,0,0,319,199);
        }
    else if (Sector2&cBelow)                    // Push Sector2 to bottom
        {
        tx2=0-(b-199)*dx/dy;
        ty2=199;
        Sector2=Sector(tx2,ty2,0,0,319,199);
        }
    if ((Sector1 & cBelow) && (Sector2 & cBelow))
        return (False);

    if (Sector1&cLeft)                          // Push Sector1 to left
        {
        ty1=b;
        tx1=0;
        Sector1=Sector(tx1,ty1,0,0,319,199);
        }
    else if (Sector2&cLeft)                      // Push Sector2 to left
        {
        ty2=b;
        tx2=0;
        Sector2=Sector(tx2,ty2,0,0,319,199);
        }
    if ((Sector1 & cLeft) && (Sector2 & cLeft))
        return (False);

    if (Sector1&cRight)                          // Push Sector1 to right
        {
        ty1=b+dy*319/dx;
        tx1=319;
        Sector1=Sector(tx1,ty1,0,0,319,199);
        }
    else if (Sector2&cRight)                     // Push Sector2 to right
        {
        ty2=b+dy*319/dx;
        tx2=319;
        Sector2=Sector(tx2,ty2,0,0,319,199);
        }
    if ((Sector1 & cRight) && (Sector2 & cRight))
        return (False);

    if ((Sector1 == (cXMid+cYMid)) &&
        (Sector2==(cXMid+cYMid)))
        {
        x1=tx1; y1=ty1; x2=tx2; y2=ty2;
        return(True);  // We did it! Yay!
        }
    return(False);
}

// Overview: Draws the line in the specified color with clipping enabled.
// Inputs:   Line to draw and it's color.
// Returns:  TRUE if line was successfully drawn.
//           FALSE if line was completely off the screen.
tBool tDisplay::DrawClippedLine(long x1, long y1, long x2, long y2, tColor color)
{
    if (ClipToScreen(x1,y1,x2,y2))
        {
        DrawLine(x1,y1,x2,y2,color);
        return(True);
        }
    return(False);
}

// Overview: Draws the overhead map on the screen.
// Inputs: The player who's viewpoint will be indicated.
void tDisplay::DrawOverheadView(tPlayer whichview)
{
    const long ArrowSize=500;

    // Create the cone of view by drawing two long lines at 45 degrees
    long x1, y1, x2, y2, x3, y3, x4, y4;
    x1 = 0;
    y1 = 0;
    x2 = x3 = x1+10000;
    y2 = y3 = 0;
    Rotate2D(x2,y2,x2,y2,whichview.ViewAngle+8192);
    Rotate2D(x3,y3,x3,y3,whichview.ViewAngle-8192);
    x1 = MapPoint(x1 + whichview.Location.x, XMapLow, XMapHigh, 0, 319);
    y1 = MapPoint(y1 + whichview.Location.y, YMapLow, YMapHigh, 0, 199);
    x2 = MapPoint(x2 + whichview.Location.x, XMapLow, XMapHigh, 0, 319);
    y2 = MapPoint(y2 + whichview.Location.y, YMapLow, YMapHigh, 0, 199);
    x3 = MapPoint(x3 + whichview.Location.x, XMapLow, XMapHigh, 0, 319);
    y3 = MapPoint(y3 + whichview.Location.y, YMapLow, YMapHigh, 0, 199);
    DrawClippedLine(x1,y1,x2,y2,Black);
    DrawClippedLine(x1,y1,x3,y3,Black);

    // Draw walls on the map
    int i;
    for(i = 0 ; i < Maze.NumberWalls() ; i++)
        {
        x1 = MapPoint(Maze.Vertices[Maze.Walls[i].a].x, XMapLow, XMapHigh, 0, 319);
        y1 = MapPoint(Maze.Vertices[Maze.Walls[i].a].y, YMapLow, YMapHigh, 0, 199);
        x2 = MapPoint(Maze.Vertices[Maze.Walls[i].b].x, XMapLow, XMapHigh, 0, 319);
        y2 = MapPoint(Maze.Vertices[Maze.Walls[i].b].y, YMapLow, YMapHigh, 0, 199);
        DrawClippedLine(x1,y1,x2,y2,ConvertTexture(Maze.Walls[i].Texture).Color);
        }

    // Create arrow
    x1 = (-(ArrowSize/2));  // Base of arrow
    y1 = 0;
    x2 = -x1;               // Head of arrow
    y2 = 0;
    x3 = x2 - ArrowSize/4;  // Left half of arrow point
    y3 = y2 + ArrowSize/4;
    x4 = x3;                // Right half of arrow point
    y4 = -y3;

    // Rotate and draw arrow
    Rotate2D(x1, y1, x1, y1, whichview.ViewAngle);
    Rotate2D(x2, y2, x2, y2, whichview.ViewAngle);
    Rotate2D(x3, y3, x3, y3, whichview.ViewAngle);
    Rotate2D(x4, y4, x4, y4, whichview.ViewAngle);
    x1 = MapPoint(x1+whichview.Location.x, XMapLow, XMapHigh, 0, 319);
    y1 = MapPoint(y1+whichview.Location.y, YMapLow, YMapHigh, 0, 199);
    x2 = MapPoint(x2+whichview.Location.x, XMapLow, XMapHigh, 0, 319);
    y2 = MapPoint(y2+whichview.Location.y, YMapLow, YMapHigh, 0, 199);
    x3 = MapPoint(x3+whichview.Location.x, XMapLow, XMapHigh, 0, 319);
    y3 = MapPoint(y3+whichview.Location.y, YMapLow, YMapHigh, 0, 199);
    x4 = MapPoint(x4+whichview.Location.x, XMapLow, XMapHigh, 0, 319);
    y4 = MapPoint(y4+whichview.Location.y, YMapLow, YMapHigh, 0, 199);
    DrawClippedLine(x1, y1, x2, y2, whichview.Color);
    DrawClippedLine(x2, y2, x3, y3, whichview.Color);
    DrawClippedLine(x2, y2, x4, y4, whichview.Color);
}

void tDisplay::Display(tPlayer whichview)
// Overview: Generates a view for the specified player in the
//           specified mode.
// Inputs:   The player whos viewpoint we wish to use
// Notes:    This routine does not update video memory automatically.
//           You must manually call Update() when you wish to display
//           the view, if double buffering is enabled.
//
//    This is so that you can put additional information (like a
//    mouse pointer, or a status bar, superimposed map, etc.) on the
//    screen if you wish.
{
    switch(DisplayMode)
        {
        case Overhead:
            DrawOverheadView(whichview);
            break;
        case ThreeD:
        case Wireframe:
        case Clipped:
        case Solid:
            Draw3DView(whichview);
            break;
        default:
            assert("Bad tDisplayMode!");
            break;
        }
}

// Overview: Renders a three-D view of the area
// Inputs:   The player whos view will be generated
// Effects:  View drawn in the memory pointed to by VideoBase.
void tDisplay::Draw3DView(tPlayer Player)
{

    // Rotate world around the player
    Rotate2DArray(Player, Maze.NumberVertices(), Maze.Vertices, Maze.ViewVert);

    // Store distances to origion from midpoint and initial drawing order.
    int i;
    for(i = 0; i < Maze.NumberWalls(); i++)
        {
        Maze.WallSort[i].Order = i;
        Maze.WallSort[i].Distance = DistanceToPoint(
                        ((Maze.ViewVert[Maze.Walls[i].a].x+
                          Maze.ViewVert[Maze.Walls[i].b].x)/2),
                        ((Maze.ViewVert[Maze.Walls[i].a].y+
                          Maze.ViewVert[Maze.Walls[i].b].y)/2));
        }

     // Sort it using good old QuickSort
    qsort((void *)Maze.WallSort, Maze.NumberWalls(),
          sizeof(Maze.WallSort[0]),
          (int(*)(const void *, const void *))DistanceCompare);

    // Draw clipped view
    tCoord x1, y1, x2, y2, x3, y3, x4, y4;
    if (DisplayMode == Clipped)
        {
        // Indicate translated, rotated walls on a 2d map
        for(i = 0; i < Maze.NumberWalls() ; i++)
            {
            x1  = MapPoint(Maze.ViewVert[Maze.Walls[i].a].x,XMapLow,XMapHigh,0,319);
            y1  = MapPoint(Maze.ViewVert[Maze.Walls[i].a].y,YMapLow,YMapHigh,0,199);
            x2  = MapPoint(Maze.ViewVert[Maze.Walls[i].b].x,XMapLow,XMapHigh,0,319);
            y2  = MapPoint(Maze.ViewVert[Maze.Walls[i].b].y,YMapLow,YMapHigh,0,199);
            DrawClippedLine(x1, y1, x2, y2, Black);
            }
        // Indicate the origin and cone of view on a 2D map underneath
        x1 = 0;
        y1 = 0;
        x2 = x3 = (x1 + 8000);              // Make two very long lines
        y2 = y3 = 0;
        Rotate2D(x2, y2, x2, y2, 8192);     // Swing lines 45 degrees
        Rotate2D(x3, y3, x3, y3, -8192);
        x1=MapPoint(x1, XMapLow, XMapHigh, 0, 319);
        y1=MapPoint(y1, YMapLow, YMapHigh, 0, 199);
        x2=MapPoint(x2, XMapLow, XMapHigh, 0, 319);
        y2=MapPoint(y2, YMapLow, YMapHigh, 0, 199);
        x3=MapPoint(x3, XMapLow, XMapHigh, 0, 319);
        y3=MapPoint(y3, YMapLow, YMapHigh, 0, 199);
        DrawClippedLine(x1,y1,x2,y2,Grey);
        DrawClippedLine(x1,y1,x3,y3,Grey);
        }

    // Draw walls
    tCoord vh = Player.ViewHeight + Player.BounceHeight; // View point height
    tColor Color;                            // Coloring of the current wall
    struct texture tex;                      // Texture control structure
    long LeftX,LeftTop,LeftBot,RightX,RightTop,RightBot;

    // Now run through draw list and draw walls in order
    int DrawIndex;
    for(DrawIndex = 0 ; DrawIndex < Maze.NumberWalls() ; DrawIndex++)
        {
        // Get index of next wall to draw
        i     = Maze.WallSort[DrawIndex].Order;
        Color = ConvertTexture(Maze.Walls[i].Texture).Color;
        tex   = ConvertTexture(Maze.Walls[i].Texture);

        // If solid color mode force texture not to be used
        if (DisplayMode == Solid)
            tex.UseTexture = False;

        x1  = Maze.ViewVert[Maze.Walls[i].a].x;
        y1  = Maze.ViewVert[Maze.Walls[i].a].y;
        x2  = Maze.ViewVert[Maze.Walls[i].b].x;
        y2  = Maze.ViewVert[Maze.Walls[i].b].y;
        if (DisplayMode == Clipped)
            {
            // Indicate translated, rotated walls on a 2d map
            x3  = MapPoint(x1,XMapLow,XMapHigh,0,319);
            y3  = MapPoint(y1,YMapLow,YMapHigh,0,199);
            x4  = MapPoint(x2,XMapLow,XMapHigh,0,319);
            y4  = MapPoint(y2,YMapLow,YMapHigh,0,199);
            //DrawClippedLine(x3,y3,x4,y4,Black);
            }

        // Cone of view bounded by y=x, y=-x, and y=ClippingDistance

        // If wall is completely behind us skip the whole plot thing
        if ((x1 < ClippingDistance) && (x2 < ClippingDistance))
            continue;

        // We'll need to describe this line segment in standard form
        // ax + by + c = 0
        // We also need the differentials for good measure
        tCoord dx = (x2-x1);
        tCoord dy = (y2-y1);
        tCoord a  = -dy;
        tCoord b  = dx;
        tCoord c  = (-dx*y2)+(dy*x2);

        // First clip against the line y=ClippingDistance
        if((x1<ClippingDistance)||(x2<ClippingDistance)) {
            // One of the points is behind us
            tCoord y=(-a*ClippingDistance-c)/b;
            if(x1<ClippingDistance)
                {
                x1=ClippingDistance;
                y1=y;
                }
            else
                {
                x2=ClippingDistance;
                y2=y;
                }
        }

        // Prevent division by zero during cone clipping
        // kludge but it works
        if(labs(a)==labs(b)) a++;

        // Now clip lines against y=x and y=-x (90 degree field of view)
        if(y1>x1)
         {
            y1=-c/(a+b);
            x1=y1;
        }

        if(y2>x2) {
            y2=-c/(a+b);
            x2=y2;
        }

        if(y1<(-x1)) {
            y1=c/(a-b);
            x1=-y1;
        }

        if(y2<(-x2)) {
            y2=c/(a-b);
            x2=-y2;
        }

        // Now clip against the line y=ClippingDistance again
        // since it may have moved during cone clipping

        if((x1<ClippingDistance)&&(x2<ClippingDistance)) continue;
        if((x1<ClippingDistance)||(x2<ClippingDistance)) {
            // One of the points is behind us
            tCoord y=(-a*ClippingDistance-c)/b;
            if(x1<ClippingDistance) {
                x1=ClippingDistance;
                y1=y;
            } else {
                x2=ClippingDistance;
                y2=y;
            }
        }

        // Desperation moves
        if(x1<ClippingDistance) x1=ClippingDistance;
        if(x2<ClippingDistance) x2=ClippingDistance;

        if (DisplayMode == Clipped) {
            // Remap points that may have changed due to clipping
            x3  = MapPoint(x1,XMapLow,XMapHigh,0,319);
            y3  = MapPoint(y1,YMapLow,YMapHigh,0,199);
            x4  = MapPoint(x2,XMapLow,XMapHigh,0,319);
            y4  = MapPoint(y2,YMapLow,YMapHigh,0,199);
        }

        // Project to get sides of trapezoid in proper perspective
        LeftX    = DistanceToScreen*y1/x1;
        LeftTop  = DistanceToScreen*(Maze.Walls[i].Height-vh)/x1;
        LeftBot  = DistanceToScreen*(-vh)/x1;
        RightX   = DistanceToScreen*y2/x2;
        RightTop = DistanceToScreen*(Maze.Walls[i].Height-vh)/x2;
        RightBot = DistanceToScreen*(-vh)/x2;

        // Convert to screen coordinates
        LeftX    = SCREEN_SIZE_X/2 - LeftX*3/2;
        RightX   = SCREEN_SIZE_X/2 - RightX*3/2;
        LeftTop  = SCREEN_SIZE_Y/2 - LeftTop*3;
        LeftBot  = SCREEN_SIZE_Y/2 - LeftBot*3;
        RightTop = SCREEN_SIZE_Y/2 - RightTop*3;
        RightBot = SCREEN_SIZE_Y/2 - RightBot*3;

         // Skip this wall if it's normal is facing away from us
         // (Don't bother if we are in wireframe mode)
        if ((RightX <= LeftX)&&(DisplayMode!=Wireframe)) continue;

        if (DisplayMode == Clipped)
            {
            // Indicate on 2d map that this wall has passed
            // visibility tests and is now being plotted
            DrawClippedLine(x3,y3,x4,y4, Color);
            }
        else
            {
            ClipToScreen(LeftX,  LeftTop,  LeftX,  LeftBot);
            ClipToScreen(RightX, RightTop, RightX, RightBot);
            ClipToScreen(LeftX,  LeftTop,  RightX, RightTop);
            ClipToScreen(LeftX,  LeftBot,  RightX, RightBot);
            if (DisplayMode == Wireframe)
                {
                DrawLine(LeftX,  LeftTop,  LeftX,  LeftBot,  Color);
                DrawLine(RightX, RightTop, RightX, RightBot, Color);
                DrawLine(LeftX,  LeftTop,  RightX, RightTop, Color);
                DrawLine(LeftX,  LeftBot,  RightX, RightBot, Color);
                }
            else
                {
                DrawTrapezoid(LeftX,LeftTop,LeftBot,RightX,RightTop,RightBot,tex);
                }
            }
    }  // End of for(i=0;i<NUMWALLS)
}

