//===========================================================================
// Project  : GREYMIX
// Module   : greymix.c
// Purpose  : DEMO: palette tweening, image mixing, scroll fades
// Notice   : 
// Author   : David Bollinger
// Created  : 09/20/94
// Revised  : 09/23/94
// Notes    : compile small memory model
//---------------------------------------------------------------------------

//===========
// directives
//-----------
#pragma inline
asm .386

//=========
// includes
//---------
#include <bios.h>
#include <mem.h>
#include <stdio.h>
#include <stdlib.h>

//========
// defines
//--------
#define drIN                   0     // slide image into viewspace
#define drOUT                  1     // slide image out of viewspace

#define NUM_MODES              12
#define mdL                    0     // left
#define mdR                    1     // right
#define mdT                    2     // top
#define mdB                    3     // bottom
#define mdTL                   4     // top left
#define mdTR                   5     // top right
#define mdBL                   6     // bottom left
#define mdBR                   7     // bottom right
#define mdH                    8     // center horizontal line
#define mdV                    9     // center vertical line
#define mdHV                   10    // center
#define mdLH                   11    // left & center horz line

//===================
// typedefs & structs
//-------------------
typedef unsigned char uchar;
typedef unsigned int uint;

//===========
// prototypes
//-----------
int  LoadFile(char *filename, uchar *buffer, uint len);
void SetMode(int mode);
void Blit64(int x1, int y1, uchar *image);
void Blit64Plus64(int x1, int y1, uchar *image);
void Copy64(uchar *dest, uchar *source);
void Slide64(uchar *mover, uchar *stayer, int dir, int mode);
void Mix64(uchar *source, uchar *dest,
           int srcx1, int srcy1, int destx1, int desty1,
           int width, int height, int sweight, int dweight);
void TweenPalette(uchar *startpal, uchar *endpal, int steps);
void SetPalette(uchar *palette, int start, int count);
void WaitVBI(void);
void MakeYTable(uint *table);
void MakeGreyPalette(uchar *palette);
void Usage(char *msg);

// =======
// globals - malloc or localize if you wish
//--------
uchar defaultpal[768], greypal[768];
uchar image1[4096], image2[4096], work[4096];
uint  ytable[200];

//============================================================================
main()
   {
   int mode;

   // load stuff
   if (LoadFile("greymix.pal", defaultpal, 768))
      Usage("Error reading GREYMIX.PAL\r\n");
   if (LoadFile("greymix1.img", image1, 4096))
      Usage("Error reading GREYMIX.PAL\r\n");
   if (LoadFile("greymix2.img", image2, 4096))
      Usage("Error reading GREYMIX2.IMG\r\n");

   // init video
   SetMode(19);
   SetPalette(defaultpal, 0, 256);
   MakeYTable(ytable);
   MakeGreyPalette(greypal);

   // display starting and reference images
   Blit64(128, 16, image1);
   Blit64Plus64(16, 119, image1);
   Blit64Plus64(239, 119, image2);

   // demo loop
   while(!bioskey(1))
      {
      // bring on image 2
      if (!bioskey(1))
         {
         TweenPalette(defaultpal, greypal, 64);
         mode = random(NUM_MODES);
         Slide64(image2, image1, drIN, mode);
         }

      // remove image 1
      if (!bioskey(1))
         {
         mode = random(NUM_MODES);
         Slide64(image1, image2, drOUT, mode);
         TweenPalette(greypal, defaultpal, 64);
         }

      // bring on image 1
      if (!bioskey(1))
         {
         TweenPalette(defaultpal, greypal, 64);
         mode = random(NUM_MODES);
         Slide64(image1, image2, drIN, mode);
         }

      // remove image 2
      if (!bioskey(1))
         {
         mode = random(NUM_MODES);
         Slide64(image2, image1, drOUT, mode);
         TweenPalette(greypal, defaultpal, 64);
         }
      }

   bioskey(0);
   SetMode(3);
   puts("GREYMIX by David Bollinger\r\n");
   return 0;
   }

//============================================================================
int LoadFile(char *filename, uchar *buffer, uint len)
   {
   FILE *file;

   if ((file=fopen(filename,"rb")) != 0)
      if (fread(buffer, len, 1, file) == 1)
         if (fclose(file) == 0)
            return 0;
   return 1;
   }

//============================================================================
void SetMode(int mode)
   {
   asm   xor   ah, ah
   asm   mov   al, byte ptr mode
   asm   int   10h
   if (mode==3) // clear text screen
      {
      asm   mov   ax, 0600h
      asm   mov   bh, 31
      asm   mov   cx, 0
      asm   mov   dx, 184fh
      asm   int   10h
      }
   }

//============================================================================
// dumb, brute force blit
//----------------------------------------------------------------------------
void Blit64(int x1, int y1, uchar *image)
   {
   asm   mov   si, word ptr image            // source pointer
   asm   mov   ax, 0xa000                    // dest pointer...
   asm   mov   es, ax
   asm   mov   bx, word ptr y1
   asm   shl   bx, 1                         // index into word values
   asm   mov   di, word ptr ytable[bx]
   asm   add   di, word ptr x1

   asm   mov   bx, 320-64                    // offset to next row
   asm   mov   dx, 64                        // 64 rows
RowLoop:
   asm   mov   cx, 64/4                      // 16 dwords
   asm   rep   movsd
   asm   add   di, bx                        // point to next screen row
   asm   dec   dx
   asm   jnz   RowLoop
   }

//============================================================================
// because of palette layout (2nd 64 = 1st 64) this displays an image which
// does not fade to grey, for reference - kind of an afterthought
//----------------------------------------------------------------------------
void Blit64Plus64(int x1, int y1, uchar *image)
   {
   uchar *wptr=work;
   register int i;

   for (i=4096; i>0; i--)
      *wptr++ = (*image++) + 64;

   Blit64(x1, y1, work);
   }

//============================================================================
// this is intended to be faster than memcpy(), haven't profiled it though
//----------------------------------------------------------------------------
void Copy64(uchar *dest, uchar *source)
   {
   // memcpy(dest, source, 4096);
   asm   push  ds
   asm   pop   es
   asm   mov   si, word ptr source
   asm   mov   di, word ptr dest
   asm   mov   cx, 4096/4                    // 1024 dwords
   asm   rep   movsd
   }

//============================================================================
// Front end to Mix64() to produce scroll fades
//----------------------------------------------------------------------------
void Slide64(uchar *mover, uchar *stayer, int dir, int mode)
   {
   int i, i2, istart, iend, iinc;

   if (dir == drIN)
      {
      istart = 0;
      iend   = 65;
      iinc   = 1;
      }
   else
      {
      istart = 64;
      iend   = -1;
      iinc   = -1;
      }

   for (i=istart; i!=iend; i+=iinc)
      {
      i2 = i>>1;
      Copy64(work, stayer);
      switch(mode)
         {
         case mdL  : Mix64(mover, work, 64-i, 0, 0, 0, i, 64, i2, 64-i2); break;
         case mdR  : Mix64(mover, work, 0, 0, 64-i, 0, i, 64, i2, 64-i2); break;
         case mdT  : Mix64(mover, work, 0, 64-i, 0, 0, 64, i, i2, 64-i2); break;
         case mdB  : Mix64(mover, work, 0, 0, 0, 64-i, 64, i, i2, 64-i2); break;
         case mdTL : Mix64(mover, work, 64-i, 64-i, 0, 0, i, i, i2, 64-i2); break;
         case mdTR : Mix64(mover, work, 0, 64-i, 64-i, 0, i, i, i2, 64-i2); break;
         case mdBL : Mix64(mover, work, 64-i, 0, 0, 64-i, i, i, i2, 64-i2); break;
         case mdBR : Mix64(mover, work, 0, 0, 64-i, 64-i, i, i, i2, 64-i2); break;
         case mdH  : Mix64(mover, work, 0, 0, 0, 32-i2, 64, i, i2, 64-i2); break;
         case mdV  : Mix64(mover, work, 0, 0, 32-i2, 0, i, 64, i2, 64-i2); break;
         case mdHV : Mix64(mover, work, 0, 0, 32-i2, 32-i2, i, i, i2, 64-i2); break;
         //
         // add your own:  combine from above routines
         // for example :  mdLH = mdL + mdH (plus a srcy1 change for fun)
         //                (scrunches to horz line while going left)
         //                (also show fixed weight mixing)
         //
         case mdLH : Mix64(mover, work, 64-i, 32-i/2, 0, 32-i/2, i, i, 32, 32); break;
         }
      WaitVBI();
      Blit64(128, 16, work);
      }
   }

//============================================================================
// adds source image to dest image
//
// srcx1, srcy1   - where to start reading from source image
// destx1, desty1 - write to start writing in dest image
// width, height  - how many to read/write from/to both images
// sweight        - relative brightness of source image
// dweight        - relative brightness of dest image
//
// sweight + dweight should = 64
// see commented lines marked VTW for variable total weight
//
// Note: no error checking here, be sure to pass reasonable values
//----------------------------------------------------------------------------
void Mix64(uchar *source, uchar *dest,
           int srcx1, int srcy1, int destx1, int desty1,
           int width, int height, int sweight, int dweight)
   {
   int x, y, nextrow;
//VTW   int totweight;

//VTW   totweight = sweight + dweight;
   nextrow = 64-width;
   source += (srcy1 * 64) + srcx1;
   dest += (desty1 * 64) + destx1;

   // special case equal weight mixing for speed
   if (sweight == dweight)
      {
      for (y=height; y>0; y--, source+=nextrow, dest+=nextrow)
         for (x=width; x>0; x--, source++, dest++)
            *dest = (*source + *dest) >> 1;
      }
   else
      {
      for (y=height; y>0; y--, source+=nextrow, dest+=nextrow)
         for (x=width; x>0; x--, source++, dest++)
            *dest = ((*source*sweight) + (*dest*dweight)) >> 6;
//VTW         *dest = ((*source*sweight) + (*dest*dweight)) / totweight
      }
   }

//============================================================================
// slowly fade between two palettes
// this could easily be made more generic by adding <start> and <count>
// values for the palette index, instead of hard-coded 0-63
//
// If fading to grey: color palette needs to be sorted in luma order,
//                    otherwise you will get unpleasing results
//                    (bright colors won't equal bright greys)
//----------------------------------------------------------------------------
void TweenPalette(uchar *startpal, uchar *endpal, int steps)
   {
   uchar workpal[64*3], *sptr, *eptr, *wptr;
   int step, palidx, diff;

   for (step=1; step<=steps; step++)
      {
      sptr = startpal;
      eptr = endpal;
      wptr = workpal;
      for (palidx=64; palidx>0; palidx--)
         {
         // red
         diff = (*eptr++) - *sptr;
         *wptr++ = (*sptr++) + ((diff*step)/steps);

         // grn
         diff = (*eptr++) - *sptr;
         *wptr++ = (*sptr++) + ((diff*step)/steps);

         // blu
         diff = (*eptr++) - *sptr;
         *wptr++ = (*sptr++) + ((diff*step)/steps);
         }
      WaitVBI();
      SetPalette(workpal, 0, 64);
      }
   }

//============================================================================
void SetPalette(uchar *palette, int start, int count)
   {
   asm   push   si
   asm   cli
   asm   mov   dx, 0x03DA
   asm   mov   si, word ptr palette
   asm   mov   dx, 0x03C8
   asm   mov   al, byte ptr start
   asm   out   dx, al   
   asm   inc   dx   
   asm   mov   cx, word ptr count
DACLoop:
   asm   mov   al, [si]
   asm   out   dx, al
   asm   inc   si
   asm   mov   al, [si]
   asm   out   dx, al
   asm   inc   si
   asm   mov   al, [si]
   asm   out   dx, al
   asm   inc   si
   asm   loop  DACLoop
   asm   sti
   asm   pop   si
   }

//============================================================================
void WaitVBI()
   {
   asm mov dx, 0x3da
VBIWait1:
   asm   in   al, dx
   asm   test al, 0x8
   asm   jnz  VBIWait1
VBIWait2:
   asm   in   al, dx
   asm   test al, 0x8
   asm   jz   VBIWait2
   }

//============================================================================
void MakeYTable(uint *table)
   {
   register uint i;

   for (i=0; i<200; i++)
      *table++ = i * 320u;
   }

//============================================================================
void MakeGreyPalette(uchar *palette)
   {
   register int i;

   for (i=0; i<64; i++)
      {
      *palette++ = i;
      *palette++ = i;
      *palette++ = i;
      }
   }

//============================================================================
void Usage(char *msg)
   {
   puts(msg);
   exit(1);
   }

//============================================================================
