// File:         BigeMDI.cpp
//
// Created:      Tue Apr 05 09:26:29 1994
//
// Description:  Implementation of MDI child window classes
//
// Regenerated:  Tue Apr 05 09:26:29 1994
// Skeleton by:  Winpro/3, a product from Xian Corporation
//               bc4owl2.skl version 1.00 developed by:
//                 Terry Richards
//                 Terry Richards Software
//                 (908) 545-6533

#include "Beditcli.h"

//Mask used to check most significant bit

#ifdef WIN32
const int MSB=0x80000000;
#else
const int MSB=0x8000;
#endif

DEFINE_RESPONSE_TABLE1(TEditClient,TWindow)
    EV_COMMAND (CM_FILESAVE,CM_FILESAVEMethod),
    EV_COMMAND (CM_FILESAVEAS,CM_FILESAVEASMethod),
    EV_COMMAND (CM_FILECLOSE,CM_FILECLOSEMethod),
    EV_COMMAND (CM_EDITUNDO,CM_EDITUNDOMethod),
    EV_COMMAND (CM_EDITCUT,CM_EDITCUTMethod),
    EV_COMMAND (CM_EDITCOPY,CM_EDITCOPYMethod),
    EV_COMMAND (CM_EDITPASTE,CM_EDITPASTEMethod),
    EV_COMMAND (CM_EDITDELETE,CM_EDITDELETEMethod),
    EV_COMMAND (CM_EDITCLEAR,CM_EDITCLEARMethod),
    EV_COMMAND (CM_EDITFIND,CM_EDITFINDMethod),
    EV_COMMAND (CM_EDITFINDNEXT,CM_EDITFINDNEXTMethod),
    EV_COMMAND (CM_EDITREPLACE,CM_EDITREPLACEMethod),
    EV_COMMAND_ENABLE (CM_FILESAVE, CM_FILESAVEEnable),
    EV_COMMAND_ENABLE (CM_EDITUNDO,CM_EDITUNDOEnable),
    EV_COMMAND_ENABLE (CM_EDITCUT,CM_EDITCUTCOPYEnable),
    EV_COMMAND_ENABLE (CM_EDITCOPY,CM_EDITCUTCOPYEnable),
    EV_COMMAND_ENABLE (CM_EDITPASTE,CM_EDITPASTEEnable),
    EV_COMMAND_ENABLE (CM_EDITDELETE,CM_EDITCUTCOPYEnable),
    EV_COMMAND_ENABLE (CM_EDITFIND,CM_EDITFINDEnable),
    EV_COMMAND_ENABLE (CM_EDITFINDNEXT,CM_EDITFINDNEXTEnable),
    EV_COMMAND_ENABLE (CM_EDITREPLACE,CM_EDITREPLACEEnable),
    EV_REGISTERED(FINDMSGSTRING,WMFindMsg),
    EV_WM_SETFOCUS,
    EV_WM_KILLFOCUS,
    EV_WM_LBUTTONDOWN,
    EV_WM_LBUTTONDBLCLK,
    EV_WM_MOUSEMOVE,
    EV_WM_LBUTTONUP,
    EV_WM_VSCROLL,
    EV_WM_HSCROLL,
    EV_WM_KEYDOWN,
    EV_WM_CHAR,
END_RESPONSE_TABLE;

#define ARRAY_INCREMENT 1000

TEditClient :: TEditClient(TMDIChild* pParent,char * FileName):
                   TWindow(pParent),
                   pLines(10 * ARRAY_INCREMENT,0,ARRAY_INCREMENT),
                   FindData(0,81),MenuDescr(MENU_CHILD,1,2,0,0,0,0),
                   CaretPos(&pLines),HiStart(&pLines),HiEnd(&pLines){

TWindowDC     DC(GetDesktopWindow()); //Our HWindow isn't valid yet
TEXTMETRIC    tm;

    SearchDialog = NULL;
    FindData.Flags = FR_DOWN | FR_MATCHCASE;

    //Get some metrics

    SetFont(DC);
    DC.GetTextMetrics(tm);
    LineHeight = tm.tmHeight + tm.tmExternalLeading;
    CharAveWidth = tm.tmAveCharWidth;
    CaretWidth = GetSystemMetrics(SM_CXBORDER) * 2;

    MouseDown = FALSE;
    OverType = FALSE;
    IsDirty = FALSE;
    SetCursor(NULL,IDC_IBEAM);
    SetTabs();

    //Make a scroller. Our files are too big for AutoOrg to work
    //in 16-bit mode so we turn it off.

    Attr.Style |= (WS_VSCROLL | WS_HSCROLL);
    Scroller = new TScroller(this,CharAveWidth,LineHeight,100,100);
    Scroller->AutoOrg = FALSE;

    pLines.OwnsElements(TRUE);

    if (FileName){

        strcpy(File,FileName);
        LoadFile();

    }

    else{

        strcpy(File,"");
        pLines.Add(new string()); //One empty string
        pLines[0]->skip_whitespace(FALSE);

    }

}

TEditClient :: ~TEditClient(){

    //The lines array owns the elements so this
    //will delete all the strings too.
    pLines.Flush();

    ClearUndoStack();

}

void TEditClient :: ClearUndoStack(){

    while (!UndoStack.IsEmpty())
        delete UndoStack.Pop();

}

void TEditClient :: SetupWindow(){

TFrameWindow* TheFrame;

    TheFrame = TYPESAFE_DOWNCAST(Parent, TFrameWindow);
    TheFrame->SetMenuDescr(MenuDescr);

    if (File[0])
        TheFrame->SetCaption(File);
    else
        TheFrame->SetCaption("Untitled");

}

char far * TEditClient :: GetClassName(){

    return "TEDITCLIENT";

}

void TEditClient :: Paint(TDC& PaintDC,BOOL /*erase*/,TRect& /*rc*/){

int        i,First,Last;
TRect      cr=GetClientRect();
TPoint     p,Workp;
TSize      Size;
TBufferPos HStart(&pLines),HEnd(&pLines);
int        DrawMode=0; //Normal drawing
BOOL       NeedHilite=FALSE;
string     WorkString;

    HideCaret();

    WorkString.skip_whitespace(FALSE);

    //If the user is dragging backwards HiStart & HiEnd can be
    //reversed until dragging is completed.

    if (HiStart > HiEnd){

        HStart = HiEnd;
        HEnd   = HiStart;

    }

    else{

        HStart = HiStart;
        HEnd   = HiEnd;

    }

    SetFont(PaintDC);

    //Set the drawing range to the window vertical size.

    //Text files are generally much "taller" than they are "wide"
    //so we don't worry about limiting the x range - normal clipping
    //will take care of it and the resulting code is *much* simpler.

    //You could probably get this even tighter by limiting the range
    //to the invalidated rectangle but painting is pretty fast already.

    p.x = -(int)Scroller->XPos * CharAveWidth;
    First = (int)Scroller->YPos;
    Last = First + (cr.Height()/LineHeight) + 1;

    if (Last >= pLines.GetItemsInContainer())
        Last = pLines.GetItemsInContainer() - 1;

    //Set the initial font colors depending on where we are in the hilite

    if (HStart != HEnd)
        NeedHilite = TRUE;

    if (NeedHilite && First > HStart.GetLine() && First <= HEnd.GetLine())
        DrawMode = 1; //Hilite drawing

    //See this routine for a description of the DrawMode parameter
    SetColor(PaintDC,DrawMode);

    //Loop through all the lines that need painting and paint them.

    for (i=First;i<=Last;i++){

        p.y = (i-First)*LineHeight;
        Workp = p;

        //Figure out if a change of hilite occurs anywhere on this line

        if (NeedHilite && i == HStart.GetLine()){

            //Hilite starts somewhere on this line. This block also
            //handles the case where the hilite starts and ends on
            //the same line.

            //There are 3 sub-strings to draw - before the hilite,
            //the hilite itself & after the hilite. Either the first
            //or last can be zero-length.

            //First the part before

            if (HStart.GetCol() != 0){

                WorkString = pLines[i]->substr(0,HStart.GetCol());
                PaintDC.TabbedTextOut(Workp,WorkString.c_str(),WorkString.length(),NumTabs,TabArray,p.x,Size);
                Workp.x += Size.cx;

            }

            //Now the hilite

            DrawMode = 1;
            SetColor(PaintDC,DrawMode);

            if (HStart.GetLine() == HEnd.GetLine())
                WorkString = pLines[i]->substr(HStart.GetCol(),HEnd.GetCol() - HStart.GetCol());
            else
                WorkString = pLines[i]->substr(HStart.GetCol());

            PaintDC.TabbedTextOut(Workp,WorkString.c_str(),WorkString.length(),NumTabs,TabArray,p.x,Size);
            Workp.x += Size.cx;

            //Finally, the part after the hilite

            if (HStart.GetLine() == HEnd.GetLine()){

                DrawMode = 0;
                SetColor(PaintDC,DrawMode);
                WorkString = pLines[i]->substr(HEnd.GetCol());
                PaintDC.TabbedTextOut(Workp,WorkString.c_str(),WorkString.length(),NumTabs,TabArray,p.x,Size);
                Workp.x += Size.cx;

            }

        }

        else if (NeedHilite && i == HEnd.GetLine()){

            //Hilite ends somewhere on this line
            //In this case there are only two sub-strings to worry
            //about - the hilite part & the part after it. Only the
            //second part can be zero-length.

            //The hilite part

            WorkString = pLines[i]->substr(0,HEnd.GetCol());
            PaintDC.TabbedTextOut(Workp,WorkString.c_str(),WorkString.length(),NumTabs,TabArray,p.x,Size);
            Workp.x += Size.cx;

            //The part after

            DrawMode = 0;
            SetColor(PaintDC,DrawMode);
            WorkString = pLines[i]->substr(HEnd.GetCol());
            PaintDC.TabbedTextOut(Workp,WorkString.c_str(),WorkString.length(),NumTabs,TabArray,p.x,Size);
            Workp.x += Size.cx;

        }

        else{

            //Continue with the status quo and draw a full line

            PaintDC.TabbedTextOut(p,pLines[i]->c_str(),pLines[i]->length(),NumTabs,TabArray,p.x,Size);
            Workp.x = Size.cx;

        }

        if (DrawMode)
            SetColor(PaintDC,0);

        PaintDC.TabbedTextOut(Workp," ",1,NumTabs,TabArray,p.x,Size);

        if (DrawMode)
            SetColor(PaintDC,DrawMode);

    }

    ShowCaret();

}

void TEditClient :: ScrollToCaret(){

TRect  Client;
TPoint Caret;
int    dx=0,dy=0;

    HideCaret();
    Caret = BufferToClient(CaretPos);
    GetClientRect(Client);

    //Make a bit of a margin

    Client.bottom -= 2 * LineHeight;
    Client.right  -= 2 * CharAveWidth;

    if (!Client.Contains(Caret)){

        if (Caret.x > Client.right)
            dx = Caret.x - Client.right;

        else if (Caret.x < Client.left)
            dx = Caret.x - Client.left;

        if (Caret.y > Client.bottom)
            dy = Caret.y - Client.bottom;

        else if (Caret.y < Client.top)
            dy = Caret.y - Client.top;

        Scroller->ScrollBy(dx/CharAveWidth,dy/LineHeight);

        Caret = BufferToClient(CaretPos);

    }

    SetCaretPos(Caret);
    ShowCaret();

}

BOOL TEditClient :: CanClose(){

int    rc;
char * Caption=File;
char   NoTitle[]="Untitled";

    if (!IsDirty)
        return TRUE;

    if (!File[0])
        Caption = NoTitle;

    rc = MessageBox("This file has changed. Do you want to save the changes?",Caption,MB_YESNOCANCEL);

    if (rc == IDNO)
        return TRUE;

    if (rc == IDCANCEL)
        return FALSE;

    if (File[0])
        CM_FILESAVEMethod();
    else
        CM_FILESAVEASMethod();

    return TRUE;

}

// menu selections & accelerator keys

void TEditClient :: CM_FILESAVEMethod(){

ofstream out(File);
int      i;

    //This test isn't strictly needed as the command is disabled
    //if this is a new file but a derived class might call it...

    if (!File[0]){

        CM_FILESAVEASMethod();
        return;

    }

    //NOTE: This will always write a CR at the end of every line.
    //      If the original file didn't have one at the end of the
    //      last line then it will grow by two extra bytes (CR & LF)
    //      the first time it is saved. Oh Well.

    for (i=0;i<pLines.GetItemsInContainer();i++)
        out << *pLines[i] << '\n';

    IsDirty = FALSE;
    ClearUndoStack();

}

void TEditClient :: CM_FILESAVEASMethod(){

TFileOpenDialog::TData FilenameData;
char                   Caption[80]="";
char                   szFile[MAXPATH+1]="";
char                   szFilter[256]="";
char                   szCustomFilter[256]="";
char                   szInitialDir[MAXPATH+1]="";
char                   szDefExt[3]="";
char                   Sep;
int                    i,Len;
TFrameWindow*          TheFrame;

    TheFrame = TYPESAFE_DOWNCAST(Parent, TFrameWindow);

    FilenameData.FileName     = szFile;
    FilenameData.Filter       = szFilter;
    FilenameData.CustomFilter = szCustomFilter;
    FilenameData.InitialDir   = szInitialDir;
    FilenameData.DefExt       = szDefExt;

    strcpy(szFilter,"All Files (*.*)|*.*|");
    FilenameData.FilterIndex = 1;  // Initially select the first filter

    Len = strlen(szFilter);
    Sep = szFilter[Len-1];

    for (i=0;i < Len;i++)
        if (szFilter[i] == Sep)
            szFilter[i] = 0;

    FilenameData.Flags = 0;

    FilenameData.Flags |= OFN_ALLOWMULTISELECT;
    FilenameData.Flags |= OFN_PATHMUSTEXIST;
    FilenameData.Flags |= OFN_OVERWRITEPROMPT;

    if (TFileSaveDialog(this,FilenameData,0,Caption).Execute() == IDOK) {

        // User selected a file
        strcpy(File,szFile);
        TheFrame->SetCaption(File);
        CM_FILESAVEMethod();

    }

    else{

        if (FilenameData.Error) {

            char msg[50];
            wsprintf(msg,"GetSaveFileName returned Error #%ld",FilenameData.Error);
            MessageBox(msg,"WARNING",MB_OK | MB_ICONSTOP);

        }

    }

}

void TEditClient :: CM_FILECLOSEMethod(){

    CloseWindow();

}

void TEditClient :: CM_EDITUNDOMethod(){

JournalEntry * pJE;
BOOL           ContinueLoop;

    //Because we disable the Undo command when there is nothing
    //to undo, this next check is not really needed. However,
    //this is a protected member function and may be called by a
    //derived class which may not be as careful.

    if (UndoStack.IsEmpty())
        return;

    //There may be a linked set of changes to undo.
    //E.G. pasting text over a hilited block causes a
    //delete and an insert and we should undo them
    //both in one action. Also, because of memory
    //limitations on an array in 16-bit mode, deletes
    //are logged in 32K "chunks" that are linked.

    do{

        pJE = UndoStack.Pop();
        pJE->Undo(this);
        ContinueLoop = pJE->IsLinked;
        delete pJE;

    } while (ContinueLoop);

    //If all changes have been undone then the
    //file is back to it's original condition

    if (UndoStack.IsEmpty())
        IsDirty = FALSE;

}

void TEditClient :: CM_EDITCUTMethod(){

    //Because we disable the cut/copy commands when there is nothing
    //to cut/copy, this next check is not really needed. However,
    //this is a protected member function and may be called by a
    //derived class which may not be as careful.

    if (HiStart == HiEnd)
        return;

    if (!DoCopy())
        return;

    DoDelete();
    ScrollToCaret();

}

void TEditClient :: CM_EDITCOPYMethod(){

    //Because we disable the cut/copy commands when there is nothing
    //to cut/copy, this next check is not really needed. However,
    //this is a protected member function and may be called by a
    //derived class which may not be as careful.

    if (HiStart == HiEnd)
        return;

    DoCopy();

}

void TEditClient :: CM_EDITPASTEMethod(){

HANDLE  hMemory;
char *  AString;
char *  ClipText;
int     i,j;

    //Because we disable the paste command when there is nothing
    //to paste, this next check is not really needed. However,
    //this is a protected member function and may be called by a
    //derived class which may not be as careful.

	if (!IsClipboardFormatAvailable(CF_TEXT))
		return;

    //First open the clipboard which gives us ownership.

    //You would think a TClipBoard object would be useful here
    //but I'll be damned if I can figure out how to construct
    //one as all the useful constructors seem to be private or
    //protected. If you can figure it out please write your
    //findings down, fold it until it is all corners and mail
    //it to Borlands' documentation department with a suitable
    //suggestion.

	if (!::OpenClipboard(HWindow)){

		MessageBox("Can't lock clipboard","Clipboard Error",MB_ICONEXCLAMATION | MB_OK);
		return;

    }

	//Get the clipboard data and convert to a pointer

	hMemory = GetClipboardData(CF_TEXT);

	if (!hMemory){

    	CloseClipboard();
		MessageBox("Can't get clipboard data","Clipboard Error",MB_ICONEXCLAMATION | MB_OK);
		return;

    }

	ClipText = (char *)GlobalLock(hMemory);
	AString = new char[strlen(ClipText)+1];

    for (i=0,j=0;ClipText[i];i++)
        if (ClipText[i] != '\r')
            AString[j++] = ClipText[i];

    AString[j] = 0;

	DoInsert(AString);
	GlobalUnlock(hMemory);
    delete[] AString;
    CloseClipboard();

    ScrollToCaret();

}

void TEditClient :: CM_EDITDELETEMethod(){

    if (HiStart == HiEnd)
        return;

    DoDelete();

    ScrollToCaret();

}

void TEditClient :: CM_EDITCLEARMethod(){

    //Select everything

    HiStart.Home(TRUE);
    HiEnd.End(TRUE);
    CaretPos = HiStart;

    //And pretend the user pressed delete

    DoDelete();

    ScrollToCaret();

}

void TEditClient :: CM_EDITFINDMethod(){

    // Show modeless Find Text common dialog

    // It could already exist.  Don't start a second one.

    if (SearchDialog && !ReplaceDialog){

        if (! SearchDialog->IsWindowVisible())
            SearchDialog->ShowWindow(SW_SHOWNORMAL);

        if (! SearchDialog->IsWindowEnabled())
            SearchDialog->EnableWindow(TRUE);

        SearchDialog->SetFocus();

        return;

    }

    if (SearchDialog)
        SearchDialog->CloseWindow();

    // Doesn't already exist.  Start one up.

    SearchDialog = new TFindDialog(this,FindData);
    SearchDialog->Create();
    ReplaceDialog = FALSE;

}

void TEditClient :: CM_EDITFINDNEXTMethod(){

    SearchDialog->UpdateData();
    FindData.Flags |= FR_FINDNEXT;
    DoSearch();

}

void TEditClient :: CM_EDITREPLACEMethod(){

    // Show modeless Replace Text common dialog

    // It could already exist.  Don't start a second one.

    if (SearchDialog && ReplaceDialog){

        if (! SearchDialog->IsWindowVisible())
            SearchDialog->ShowWindow(SW_SHOWNORMAL);

        if (! SearchDialog->IsWindowEnabled())
            SearchDialog->EnableWindow(TRUE);

        SearchDialog->SetFocus();

        return;

    }

    if (SearchDialog)
        SearchDialog->CloseWindow();

    // Doesn't already exist.  Start one up.

    SearchDialog = new TReplaceDialog(this,FindData);
    SearchDialog->Create();
    ReplaceDialog = TRUE;

}

//Command enablers

void TEditClient :: CM_FILESAVEEnable(TCommandEnabler& commandHandler){

    commandHandler.Enable(IsDirty && File[0]);

}

void TEditClient :: CM_EDITUNDOEnable(TCommandEnabler& commandHandler){

    commandHandler.Enable(!UndoStack.IsEmpty());

}

void TEditClient :: CM_EDITCUTCOPYEnable(TCommandEnabler& commandHandler){

    commandHandler.Enable(HiEnd != HiStart);

}

void TEditClient :: CM_EDITPASTEEnable(TCommandEnabler& commandHandler){

    commandHandler.Enable(IsClipboardFormatAvailable(CF_TEXT));

}

void TEditClient :: CM_EDITFINDEnable(TCommandEnabler& commandHandler){

    commandHandler.Enable(SearchDialog == NULL || ReplaceDialog);

}

void TEditClient :: CM_EDITFINDNEXTEnable(TCommandEnabler& commandHandler){

    commandHandler.Enable(SearchDialog != NULL);

}

void TEditClient :: CM_EDITREPLACEEnable(TCommandEnabler& commandHandler){

    commandHandler.Enable(SearchDialog == NULL || !ReplaceDialog);

}

LRESULT TEditClient :: WMFindMsg(WPARAM /*wp*/,LPARAM lp){

    //This is a registered message

    SearchDialog->UpdateData(lp);
    DoSearch();
    return 0;

}

void TEditClient :: DoSearch(){

BOOL Down=FALSE,MatchCase=FALSE,WholeWord=FALSE;

    if (FindData.Flags & FR_DIALOGTERM){

        // The dialog box is closing
        SearchDialog = NULL;
        return;

    }

    if (FindData.Flags & FR_DOWN)
        Down = TRUE;

    if (FindData.Flags & FR_MATCHCASE)
        MatchCase = TRUE;

    if (FindData.Flags & FR_WHOLEWORD)
        WholeWord = TRUE;

    if (FindData.Flags & FR_FINDNEXT){

        DoFind(Down,MatchCase,WholeWord,FindData.FindWhat);

        return;

    }

    if (FindData.Flags & FR_REPLACE){

        DoReplace(FindData.ReplaceWith);

        return;

    }

    if (FindData.Flags & FR_REPLACEALL){

        CaretPos.Home(TRUE);
        HiStart = CaretPos;
        HiEnd = CaretPos;

        while(DoFind(Down,MatchCase,WholeWord,FindData.FindWhat))
            DoReplace(FindData.ReplaceWith);

    }

}

void TEditClient :: EvSetFocus(HWND /*hw*/){

//This function is also called when the insert mode is toggled.
//You can overload this to change the caret shapes, set an insert
//mode indicator in a status bar, etc.

int Width=CaretWidth;
int Height=LineHeight;

    if (OverType){

        Width *= 3;
        Height /= 2;

    }

    //If you do overload this function you MUST do this stuff:

    CreateCaret(FALSE,Width,Height);
    SetCaretPos(BufferToClient(CaretPos));
    ShowCaret();

}

void TEditClient :: EvKillFocus(HWND /*hw*/){

    DestroyCaret();

}

void TEditClient :: EvLButtonDown(UINT modKeys, TPoint& point){

BOOL NeedRedraw=FALSE;

    //We don't need to capture the mouse - TScroller does it.

    if (HiEnd != HiStart)
        NeedRedraw = TRUE;

    MouseDown = TRUE;
    HiStart   = ClientToBuffer(point);
    HiEnd     = HiStart;

    if (NeedRedraw){

        Invalidate();
        UpdateWindow(); //Do it right now...

    }

    //This message needs to get to TScroller
    TWindow :: EvLButtonDown(modKeys,point);

}

void TEditClient :: EvLButtonDblClk(UINT /*modKeys*/, TPoint& /*point*/){

string BreakString;

    //Double click selects a whole word

    BreakString.skip_whitespace(FALSE);
    SetBreakChars(BreakString);

    HiStart = CaretPos;
    HiStart.PrevBreak(BreakString);
    HiEnd   = CaretPos;
    HiEnd.NextBreak(BreakString);

    CaretPos = HiEnd;
    SetCaretPos(BufferToClient(CaretPos));

    Invalidate(FALSE);
    MouseDown = FALSE;

}

void TEditClient :: EvMouseMove(UINT modKeys, TPoint& point){

    if (!MouseDown)
        return;

    HiEnd = ClientToBuffer(point);
    Invalidate(FALSE);
    UpdateWindow();

    //TScroller needs this one too
    TWindow :: EvMouseMove(modKeys,point);

}

void TEditClient :: EvLButtonUp(UINT modKeys, TPoint& point){

TBufferPos Temp(&pLines);

    if (!MouseDown)
        return;

    MouseDown = FALSE;
    InvalidateToEnd(CaretPos);
    CaretPos  = ClientToBuffer(point);
    HiEnd     = CaretPos;

    SetCaretPos(BufferToClient(CaretPos));

    if (HiStart > HiEnd){

        Temp    = HiEnd;
        HiEnd   = HiStart;
        HiStart = Temp;

    }

    //TScroller needs this one too
    TWindow :: EvLButtonUp(modKeys,point);

}

void TEditClient :: EvHScroll(UINT scrollCode, UINT thumbPos, HWND hWndCtl){

    HideCaret();
    TWindow :: EvHScroll(scrollCode,thumbPos,hWndCtl);
    SetCaretPos(BufferToClient(CaretPos));
    ShowCaret();

}

void TEditClient :: EvVScroll(UINT scrollCode, UINT thumbPos, HWND hWndCtl){

    HideCaret();
    TWindow :: EvVScroll(scrollCode,thumbPos,hWndCtl);
    SetCaretPos(BufferToClient(CaretPos));
    ShowCaret();

}

void TEditClient :: EvKeyDown(UINT key, UINT repeatCount, UINT /*flags*/){

TRect cr;
int   ScreenHeight;
BOOL  Shift,Ctrl;
BOOL  MoveStart;
BOOL  HiliteChanged=FALSE;
BOOL  WasCursorKey=TRUE;

    //The flags parameter is not reliable - it has the same value
    //for Shift & Ctrl-Shift. So we don't use it. Instead we look
    //at the Most Significant Bit of GetAsyncKeyState()

    Shift = GetAsyncKeyState(VK_SHIFT) & MSB;
    Ctrl  = GetAsyncKeyState(VK_CONTROL) & MSB;

    cr = GetClientRect();
    ScreenHeight = cr.Height() / LineHeight;

    //If shift is pressed then the cursor keys extend the selection.
    //But which end do we extend? The end the cursor is sitting on.

    if (CaretPos == HiEnd)
        MoveStart = FALSE;
    else
        MoveStart = TRUE;

	switch (key) {

        //Handle the cursor positioning keys here

		case VK_RIGHT:

            CaretPos += repeatCount;
            break;

		case VK_LEFT:

            CaretPos -= repeatCount;
            break;

		case VK_DOWN:

            CaretPos.LineChange(repeatCount);
            break;

		case VK_UP:

            CaretPos.LineChange(-repeatCount);
            break;

		case VK_NEXT: //Page Down

            CaretPos.LineChange(repeatCount * ScreenHeight);
			break;

		case VK_PRIOR:	//Page up

            CaretPos.LineChange(-repeatCount * ScreenHeight);
			break;

		case VK_HOME:

            CaretPos.Home(Ctrl);
			break;

		case VK_END:

            CaretPos.End(Ctrl);
			break;

        //We also need to handle delete & backspace here. Note that
        //we don't use the repeat count - it's a little safer that
        //way as Windows can stack up a *lot* of them if it was busy
        //elsewhere for a while.

		case VK_BACK:

            if (CaretPos.GetLine() == 0 && CaretPos.GetCol() == 0)
                return;

            --CaretPos;
            //Drop Through

		case VK_DELETE:

            DoDelete();
            WasCursorKey = FALSE;
			break;

        //Also, we toggle the insert mode bit here

		case VK_INSERT:

            OverType = !OverType;
            WasCursorKey = FALSE;
            EvKillFocus(HWindow);
            EvSetFocus(HWindow);
			break;

        //Anything else is a character key - let windows have it
        //back and we will catch it in EvChar.

		default:

            DefaultProcessing();
            return; //NOTE - return not break!

    }

    ScrollToCaret();

    if (WasCursorKey){

        if (Shift){

            if (MoveStart)
                HiStart = CaretPos;
            else
                HiEnd = CaretPos;

            HiliteChanged = TRUE;

        }

        else{

            if (HiStart != HiEnd)
                HiliteChanged = TRUE;

            HiEnd = CaretPos;
            HiStart = HiEnd;

        }

        if (HiliteChanged)
            Invalidate(FALSE);

    }

    UpdateWindow();

}

void TEditClient :: EvChar(UINT key, UINT repeatCount, UINT /*flags*/){

int i;

    //We don't deal with console bells etc.

	if (key <= VK_ESCAPE && key != VK_TAB && key != VK_RETURN)
        return;

    if (key == VK_RETURN)
        key = '\n';

    for (i=0;i<repeatCount;i++)
        DoInsert((char)key);

    ScrollToCaret();
    UpdateWindow();

}

TPoint TEditClient :: BufferToClient(TBufferPos pt){

TPoint    xy;
TSize     sz;
TClientDC DC(HWindow);
string    TestString;

    //This routine converts a TBufferPos position (line & col)
    //into a physical screen position.

    TestString.skip_whitespace(FALSE);

    xy.y = pt.GetLine() - (int)Scroller->YPos;
    xy.y *= LineHeight;

    if (pt.GetCol() == 0)
        xy.x = 0;

    else{

        TestString = pLines[pt.GetLine()]->substr(0,pt.GetCol());
        SetFont(DC);
        sz = DC.GetTabbedTextExtent(TestString.c_str(),TestString.length(),NumTabs,TabArray);
        xy.x = sz.cx;

    }

    xy.x -= (int)Scroller->XPos * CharAveWidth;

    return xy;

}

TBufferPos TEditClient :: ClientToBuffer(TPoint pt){

TBufferPos bp(&pLines);
TSize      sz;
TClientDC  DC(HWindow);

    //This routine converts a physical screen postion into
    //a TBufferPos (line & col) object.

    pt.y /= LineHeight;
    pt.y += (int)Scroller->YPos;
    bp.SetLine(pt.y);

    if (bp.GetLine() >= pLines.GetItemsInContainer()){

        bp.SetLine(pLines.GetItemsInContainer() - 1);
        bp.SetCol(pLines[bp.GetLine()]->length());

        return bp;

    }

    pt.x += (int)Scroller->XPos * CharAveWidth;
    SetFont(DC);

    for (bp.SetCol(0);bp.GetCol() < pLines[bp.GetLine()]->length();bp.SetCol(bp.GetCol()+1)){

        sz = DC.GetTabbedTextExtent(pLines[bp.GetLine()]->c_str(),bp.GetCol(),NumTabs,TabArray);

        if (sz.cx + (CharAveWidth / 2) >= pt.x)
            break;

    }

    return bp;

}

void TEditClient :: LoadFile(){

ifstream     in(File);
string *     pLine;
unsigned int Limit;
int          Num;

    Limit = MAXINT/2;
    Limit /= ARRAY_INCREMENT;
    Limit--;
    Limit *= ARRAY_INCREMENT;

    while (!in.eof()){

        pLine = new string;
        pLine->skip_whitespace(FALSE);
        pLine->max_waste(4);
        pLine->resize_increment(16);
        pLine->read_line(in); //Automatically strips of Carriage Return
        pLines.Add(pLine);

        if (pLines.GetItemsInContainer() >= Limit){

            MessageBox("Too many lines. Part file loaded.");
            break;

        }

        if (in.bad()){

            MessageBox("File I/O Error. Part file loaded.");
            break;

        }


    }

    //If the last line was terminated with a CR then we get an
    //extra null line - one for the CR and one for the EOF.

    Num = pLines.GetItemsInContainer() - 1;

    if (Num && pLines[Num]->length() == 0)
        pLines.Destroy(Num);

    Scroller->SetRange(100,Num);

}

#define WORK_SIZE 32759

void TEditClient :: DoDelete(BOOL Log){

int         i,Start,End;
TBufferPos  Cursor(&pLines);
char *      Deleted;
BOOL        Linked=FALSE;

    if (HiEnd != HiStart)
        CaretPos = HiStart;

    if (Log){

        i = 0;
        Deleted = new char[WORK_SIZE+1];

        if (HiEnd != HiStart){

            for(Cursor=HiStart;HiEnd > Cursor;++Cursor){

                if (Cursor.AtEnd())
                    Deleted[i++] = '\n';
                else
                    Deleted[i++] = (*pLines[Cursor.GetLine()])[Cursor.GetCol()];

                if (i == WORK_SIZE){

                    Deleted[i] = 0;
                    UndoStack.push(new JEDelete(Deleted,&pLines,CaretPos,Linked));
                    Linked = TRUE;
                    i = 0;

                }

            }

        }

        else{

            if (CaretPos.AtEnd())
                Deleted[i++] = '\n';
            else
                Deleted[i++] = (*pLines[CaretPos.GetLine()])[CaretPos.GetCol()];

        }

        Deleted[i] = 0;
        UndoStack.push(new JEDelete(Deleted,&pLines,CaretPos,Linked));
        delete Deleted;

    }

//There are two possibilities - either we are deleting a single
//character or a hilited block. We can tell by looking at the
//HiStart & HiEnd pointers.

    //Single Character

    if (HiStart == HiEnd){

        DeleteOneChar();
        return;

    }

    //Hilite block. Is it within a single line?

    if (HiStart.GetLine() == HiEnd.GetLine()){

        Start    = HiStart.GetCol();
        End      = HiEnd.GetCol();
        HiEnd    = HiStart;
        CaretPos = HiStart;

        for (i=Start;i<End;i++)
            DeleteOneChar();

        return;

    }

    //If we get this far it is a multi-line block. We delete it in
    //3 stages. The end of the 1st line, the complete lines, and
    //the start of the last line. For simpler coding we delete the
    //part lines first (The HiEnd buffer pointer will not be valid
    //after deleting the full lines).

    //The 1st line. We leave the CR on it for now.

    CaretPos = HiStart;
    Start = CaretPos.GetCol();
    End = pLines[CaretPos.GetLine()]->length();

    for (i=Start;i<End;i++)
        DeleteOneChar();

    //The last line;

    CaretPos = HiEnd;
    CaretPos.Home(FALSE);

    Start = 0;
    End = HiEnd.GetCol();

    for (i=Start;i<End;i++)
        DeleteOneChar();

    CaretPos = HiStart;

    //The stuff inbetween - note that we destroy them backwards
    //so that the index is still valid. EG If we want to destroy
    //Line[2] & Line[3] and destroyed [2] first then [3] would
    //become [2]...

    Start = HiStart.GetLine() + 1;
    End = HiEnd.GetLine()-1;

    for (i=End;i>=Start;i--)
        pLines.Destroy(i);

    //We left the CR on the first line. Now the intermediate lines
    //are gone we can go back for it.

    CaretPos.End(FALSE);
    DeleteOneChar();

    HiEnd = HiStart;
    CaretPos = HiEnd;

}

void TEditClient :: DeleteOneChar(){

int Line = CaretPos.GetLine();
int Col  = CaretPos.GetCol();

    //is it an embedded char or the CR?

    if (CaretPos.AtEnd()){ //CR

        //Don't delete the very last one

        if (Line >= pLines.GetItemsInContainer()-1)
            return;

        pLines[Line]->append(*pLines[Line+1]);
        pLines.Destroy(Line+1);
        Scroller->SetRange(100,pLines.GetItemsInContainer()-1);
        Invalidate();

    }

    else{ //Embedded

        InvalidateToEnd(CaretPos);
        pLines[Line]->remove(Col,1);

    }

    IsDirty = TRUE;

}

void TEditClient :: DoInsert(char Key,BOOL Log){

BOOL Linked=FALSE;

string * NewString;

    //If there is Hilite text this char will overwrite it.
    //Also, if we are in overtype mode we should delete a char.

    if (HiEnd != HiStart|| (OverType && !CaretPos.AtEnd())){

        DoDelete(Log);
        Linked = TRUE;

    }

    if (Log)
        UndoStack.push(new JEInsert(1,&pLines,CaretPos,Linked));

    IsDirty = TRUE;

    //If we are inserting a CR we have to split the string.
    //Do the easy one first.

    if (Key != '\n'){

        pLines[CaretPos.GetLine()]->insert(CaretPos.GetCol(),Key);
        InvalidateToEnd(CaretPos);
        ++CaretPos;

        return;

    }

    //If we get here we are splitting the string. First make a new
    //string from the second half.

    NewString = new string(pLines[CaretPos.GetLine()]->substr(CaretPos.GetCol()));
    NewString->skip_whitespace(FALSE);

    //Now delete those chars from the line

    pLines[CaretPos.GetLine()]->remove(CaretPos.GetCol());

    //Finally put the new string into the pLines array

    pLines.AddAt(NewString,CaretPos.GetLine()+1);
    Scroller->SetRange(100,pLines.GetItemsInContainer()-1);

    ++CaretPos;
    Invalidate(TRUE);

}

void TEditClient :: DoInsert(char * KeyString,BOOL Log){

BOOL        Linked = FALSE;
UINT        i,iSave,Len=0;
char *      Work;
string *    NewString;

    Len = strlen(KeyString);

    //If there is Hilite text this char will overwrite it.
    //Also, if we are in overtype mode we should delete a char.

    if (HiEnd != HiStart|| (OverType && !CaretPos.AtEnd())){

        DoDelete(Log);
        Linked = TRUE;

    }

    if (Log)
        UndoStack.push(new JEInsert(Len,&pLines,CaretPos,Linked));

    //The text to be inserted can be treated as 3 parts - the start
    //is the text leading up to (and including) the first CR. We
    //insert this one character at a time as it involves splitting
    //a string and we can use the existing logic. After the first
    //CR has been inserted we know we are at the start of a line
    //and can insert the middle part in whole-line chunks. Finally,
    //we insert the text after the last CR by returning to the single
    //character method. This puts all the tricky logic in the single
    //character method while still maintaining adequate performance
    //on large inserts.

    //Insert the first part

    for (i=0;i < Len;i++){

        DoInsert(KeyString[i],FALSE);

        if (KeyString[i] == '\n')
            break;

    }

    //Insert the middle part.

    while (i < Len){

        i++; //Skip over the current CR

        Work = KeyString + i;
        iSave = i;

        while (KeyString[i] && KeyString[i] != '\n')
            i++;

        if (KeyString[i] != '\n'){

            i = iSave;
            break;

        }

        KeyString[i] = 0;

        //Work now points to a NULL terminated line - insert it

        NewString = new string(Work);
        NewString->skip_whitespace(FALSE);
        pLines.AddAt(NewString,CaretPos.GetLine());
        CaretPos.LineChange(+1);

    }

    //Insert the end part.

    for (;i < Len;i++)
        DoInsert(KeyString[i],FALSE);

    Scroller->SetRange(100,pLines.GetItemsInContainer()-1);

}

BOOL TEditClient :: DoCopy(){

string TheText,CRLF;
int    SLine = HiStart.GetLine();
int    SCol  = HiStart.GetCol();
int    ELine = HiEnd.GetLine();
int    ECol  = HiEnd.GetCol();
int    i;
UINT   Limit=MAXINT/2;
char * TheBlock;
HANDLE hGlobalMemory;

    //You would think a TClipBoard object would be useful here
    //but I'll be damned if I can figure out how to construct
    //one as all the useful constructors seem to be private or
    //protected. If you can figure it out please write your
    //findings down, fold it until it is all corners and mail
    //it to Borlands' documentation department with a suitable
    //suggestion.

    TheText.skip_whitespace(FALSE);
    CRLF.skip_whitespace(FALSE);
    CRLF += "\r\n";

    //Is it a single line?

    if (SLine == ELine)
        TheText = pLines[SLine]->substr(SCol,ECol - SCol);

    //Or a block?

    else{

        TheText = pLines[SLine]->substr(SCol);
        TheText.append(CRLF);

        for (i=SLine+1;i<ELine;i++){

            TheText.append(*(pLines[i]));
            TheText.append(CRLF);

            if (TheText.length() >= Limit){

                MessageBox("The selected text is too large for the clipboard","Space Error",MB_OK | MB_ICONSTOP);
                return FALSE;

            }

        }

        TheText.append(pLines[ELine]->substr(0,ECol));

    }

	hGlobalMemory = GlobalAlloc(GHND,TheText.length()+1); //NULL terminated
	TheBlock = (char *)GlobalLock(hGlobalMemory);
    strcpy(TheBlock,TheText.c_str());
	GlobalUnlock(hGlobalMemory);

	::OpenClipboard(HWindow);
	::EmptyClipboard();
	::SetClipboardData(CF_TEXT,hGlobalMemory);
	::CloseClipboard();

    return TRUE;

}

BOOL TEditClient :: DoFind(BOOL Down,BOOL MatchCase,BOOL WholeWord,char * FindWhat){

string     BreakString;
TBufferPos Temp(&pLines);

    BreakString.skip_whitespace(FALSE);
    SetBreakChars(BreakString);

    if (Down){

        Temp = CaretPos;
        ++CaretPos;

        if (CaretPos == Temp) //EOF
            return FALSE;

        if (CaretPos.FindNext(MatchCase,WholeWord,BreakString,FindWhat)){

            HiStart = CaretPos;
            HiEnd   = CaretPos;
            HiEnd  += strlen(FindWhat);

            ScrollToCaret();
            Invalidate(FALSE);

            return TRUE;

        }

        else{

            --CaretPos;

            HiStart = CaretPos;
            HiEnd   = CaretPos;
            MessageBeep(-1);

            return FALSE;

        }

    }

    else{ //Up

        if (CaretPos.GetCol() == 0 && CaretPos.GetLine() == 0)
            return FALSE;

        --CaretPos;

        if (CaretPos.FindPrev(MatchCase,WholeWord,BreakString,FindWhat)){

            HiStart = CaretPos;
            HiEnd   = CaretPos;
            HiEnd  += strlen(FindWhat);

            ScrollToCaret();
            Invalidate(FALSE);

            return TRUE;

        }

        else{

            ++CaretPos;
            HiStart = CaretPos;
            HiEnd   = CaretPos;
            MessageBeep(-1);

            return FALSE;

        }

    }

}

void TEditClient :: DoReplace(char * ReplaceWith){

    //DoFind sets the hilite. If the user changed it we shouldn't
    //do the replace. Strictly speaking we should disable the replace
    //button when the user moves the cursor or deletes the hilite text
    //rather than relying on this half-assed method. That's a future
    //enhancement as this method catches most cases.

    if (HiEnd == HiStart)
        return;

    DoInsert(ReplaceWith);

}

void TEditClient :: InvalidateToEnd(TBufferPos& Start){

TBufferPos End(&pLines);
TPoint     PStart,PEnd;
TRect*     rc;

    End   = Start;
    End.End(FALSE);

    PStart = BufferToClient(Start);
    PEnd   = BufferToClient(End);
    PEnd.y += LineHeight;

    rc = new TRect(PStart,PEnd);
    InvalidateRect(*rc);
    delete rc;

}

void TEditClient :: Open(char * FileName){

//This routine is used to replace the current file with a new
//file. It will be most useful in an SDI situation. In MDI you
//will normally just open a new window.

//NOTE: This routine does not care if the existing file is "dirty"
//      You can check this yourself by calling CanClose().

TFrameWindow* TheFrame;

    pLines.Flush();
    ClearUndoStack();
    MouseDown = FALSE;
    IsDirty = FALSE;

    if (FileName){

        strcpy(File,FileName);
        LoadFile();

    }

    else{

        strcpy(File,"");
        pLines.Add(new string()); //One empty string
        pLines[0]->skip_whitespace(FALSE);

    }

    TheFrame = TYPESAFE_DOWNCAST(Parent, TFrameWindow);

    if (File[0])
        TheFrame->SetCaption(File);
    else
        TheFrame->SetCaption("Untitled");

    CaretPos.Home(TRUE);
    ScrollToCaret();
    Invalidate();

}

void TEditClient :: SetFont(TDC& /*DC*/){

    //Place Holder function, you can overload this.
    //Something like this would be good:

    //TFont FixedFont("Courier New", -12 * DC.GetDeviceCaps(LOGPIXELSY) / 72);
    //DC.SelectObject(FixedFont);

}

void TEditClient :: SetColor(TDC& DC,int Mode){

    //Mode tells us what we need to set the text colors for.
    //The base class uses:
    //
    //   0 - Normal text
    //   1 - Hilite text.

    //You can overload this and you can add new modes such as
    //found text, syntax hiliting, modified text, etc.

    switch (Mode){

        case 0:

            DC.SetTextColor(GetSysColor(COLOR_WINDOWTEXT));
            DC.SetBkColor(GetSysColor(COLOR_WINDOW));

            break;

        case 1:

            DC.SetTextColor(GetSysColor(COLOR_HIGHLIGHTTEXT));
            DC.SetBkColor(GetSysColor(COLOR_HIGHLIGHT));

            break;

    }

}

void TEditClient :: SetTabs(){

    //You can overload this.

    TabArray[0] = 4 * CharAveWidth;
    NumTabs = 1;

}

void TEditClient :: SetBreakChars(string& BreakString){

//These chars are what defines a break. the first three are double
//quote, back-slash, and tab. There is no need to specify a carriage
//return in this string - it is implied. You can overload this function.

    BreakString += "\"\\\t +-=/~!(){}[]<>'?.,;: ";

}

//*****************************************************************//
//The TBufferPos class. Basically a smart encapsulation of line &  //
//column. It knows when it is off the end of a line etc.           //
//*****************************************************************//

TBufferPos::TBufferPos(StringArray* ABuffer){

    Line       = 0;
    Col        = 0;
    Buffer     = ABuffer;
    LastUpDown = FALSE;

}

int TBufferPos::GetCol(){

    //When we are moving up & down Col may be off the end of the
    //string. This prevents the cursor drifting left as it passes
    //over shorter lines. However, this could cause problems to the
    //main program if it trys to read off the end of a string. So
    //we provide this cool accessor function.

    if (!LastUpDown)
        return Col;

    if (Col <= (*Buffer)[Line]->length())
        return Col;

    return (*Buffer)[Line]->length();

}

int TBufferPos::GetLine(){

    //Just for symetry, we provide this cool accessor function too.
    //It lets us make Line protected so nobody can fool with it.

    return Line;

}

void TBufferPos::LineChange(int Delta){

    Line += Delta;

    if (Line < 0)
        Line = 0;

    if (Line >= Buffer->GetItemsInContainer())
        Line = Buffer->GetItemsInContainer() - 1;

    LastUpDown = TRUE;

}

void TBufferPos::Home(BOOL Ctrl){

    if (Ctrl)
        Line = 0;

    Col = 0;

}

void TBufferPos::End(BOOL Ctrl){

    if (Ctrl)
        Line = Buffer->GetItemsInContainer() - 1;

    Col = (*Buffer)[Line]->length();

}

BOOL TBufferPos::AtEnd(){


    if (Col == (*Buffer)[Line]->length())
        return TRUE;

    return FALSE;

}

//NOTE: The documentation for find_first_of & find_last_of is
//      completely wrong. From a lot of playing with the debugger
//      it seems this is the correct way to use them. I wouldn't
//      be suprised if there's a gotcha or two hiding in there
//      though...

void TBufferPos::NextBreak(string BreakString){

    //Find the first break char AFTER Col
    Col = (*Buffer)[Line]->find_first_of(BreakString,Col);

    if (Col == -1) //Not found - set to end of string
        Col = (*Buffer)[Line]->length();

}

void TBufferPos::PrevBreak(string BreakString){

    //Find the first break char BEFORE Col
    Col = (*Buffer)[Line]->find_last_of(BreakString,Col);

    //find_last_of returns -1 for not found which just so happens
    //to work right for us as we really want the char after the break
    //or zero if no break found.

    if (Col != (*Buffer)[Line]->length())
        Col++;

}

BOOL TBufferPos::FindNext(BOOL MatchCase,BOOL WholeWord,string BreakString,char *FindWhat){

int  i,Limit,FoundAt;

    Limit = Buffer->GetItemsInContainer();
    FoundAt = Col;

    for (i=Line;i<Limit;i++){

        (*Buffer)[i]->set_case_sensitive(MatchCase);

        while (TRUE){

            FoundAt = (*Buffer)[i]->find(FindWhat,FoundAt);

            if (FoundAt == (int)NPOS)
                break; //Drop out of inner loop & check next line

            if (!WholeWord || IsWholeWord(i,FoundAt,strlen(FindWhat),BreakString)){

                Line = i;
                Col = FoundAt;

                return TRUE;

            }

            FoundAt++;

        }

        FoundAt = 0;

    }

    //Not found at all

    return FALSE;

}

BOOL TBufferPos::FindPrev(BOOL MatchCase,BOOL WholeWord,string BreakString,char *FindWhat){

int  i,FoundAt;

    FoundAt = Col+1;

    for (i=Line;i>=0;i--){

        (*Buffer)[i]->set_case_sensitive(MatchCase);

        while (TRUE){

            FoundAt = (*Buffer)[i]->rfind(FindWhat,FoundAt);

            if (FoundAt == (int)NPOS)
                break; //Drop out of inner loop & check next line

            if (!WholeWord || IsWholeWord(i,FoundAt,strlen(FindWhat),BreakString)){

                Line = i;
                Col = FoundAt;

                return TRUE;

            }

            FoundAt--;

        }

        if (i)
            FoundAt = (*Buffer)[i-1]->length();

    }

    //Not found at all

    return FALSE;

}

BOOL TBufferPos::IsWholeWord(int Line,int Col,int Length,string BreakString){

char WorkChar;
int  WorkPos;

    if (Col != 0){

        WorkPos = Col - 1;
        WorkChar = (*Buffer)[Line]->get_at(WorkPos);

        if (BreakString.find(WorkChar) == NPOS)
            return FALSE;

    }

    WorkPos = Col + Length;

    if (WorkPos < (*Buffer)[Line]->length()){

        WorkChar = (*Buffer)[Line]->get_at(WorkPos);

        if (BreakString.find(WorkChar) == NPOS)
            return FALSE;

     }

     return TRUE;

}

BOOL TBufferPos::operator == (TBufferPos ABuf){

    if (ABuf.Line == Line && ABuf.Col == Col)
        return TRUE;

    return FALSE;

}

BOOL TBufferPos::operator != (TBufferPos ABuf){

    return !operator == (ABuf);

}

void TBufferPos::operator = (TBufferPos ABuf){

    Line       = ABuf.Line;
    Col        = ABuf.Col;
    LastUpDown = ABuf.LastUpDown;

}

BOOL TBufferPos::operator > (TBufferPos ABuf){

    if (Line > ABuf.Line)
        return TRUE;

    if (Line < ABuf.Line)
        return FALSE;

    //Lines are equal - check column

    if (Col > ABuf.Col)
        return TRUE;

    return FALSE;

}

void TBufferPos::operator ++(){

    if (LastUpDown)
        FixCol();

    Col++;
    Normalize();

}

void TBufferPos::operator +=(int Delta){

    if (LastUpDown)
        FixCol();

    Col += Delta;
    Normalize();

}

void TBufferPos::operator --(){

    if (LastUpDown)
        FixCol();

    Col--;
    Normalize();

}

void TBufferPos::operator -=(int Delta){

    if (LastUpDown)
        FixCol();

    Col -= Delta;
    Normalize();

}

void TBufferPos::Normalize(){

    while (Col < 0){

        if (Line == 0){

            Col = 0;
            return;

        }

        Line--;
        Col += (*Buffer)[Line]->length()+1; //1 is for the CR

    }

    while (Col > (*Buffer)[Line]->length()){

        if (Line == Buffer->GetItemsInContainer() - 1){

            Col = (*Buffer)[Line]->length();
            return;

        }

        Col -= (*Buffer)[Line]->length()+1;  //1 is for the CR
        Line++;

    }

}

void TBufferPos::FixCol(){

    if (Col > (*Buffer)[Line]->length())
        Col = (*Buffer)[Line]->length();

    LastUpDown = FALSE;

}

//**************************************************************//
//The Journal classes. Changes are logged as one of the derived //
//classes which know how to undo themselves when called upon to //
//do so.                                                        //
//**************************************************************//

JournalEntry::JournalEntry(StringArray* pBuffer,TBufferPos Location,BOOL Linked):
                Where(pBuffer){

    Where = Location;
    Addr = this; //As good a way to uniquely identify a JE as any.
    IsLinked = Linked;

}

BOOL JournalEntry::operator == (JournalEntry Other){

    return (Addr == Other.Addr);

}

void JournalEntry::Undo(TEditClient* /*Parent*/){

    //This routine should never be called.

    //It should be a pure virtual function but the
    //template classes don't seem to like that idea.

}

JEInsert::JEInsert(int HowMany,StringArray* pBuffer,TBufferPos Location,BOOL Linked):
                JournalEntry(pBuffer,Location,Linked){

    NumChars = HowMany;

}

void JEInsert::Undo(TEditClient* Parent){

    Parent->CaretPos = Where;
    Parent->HiStart  = Where;
    Parent->HiEnd    = Where;
    Parent->HiEnd    += NumChars;

    Parent->DoDelete(FALSE);
    Parent->CaretPos = Where;
    Parent->ScrollToCaret();

}

JEDelete::JEDelete(char * What,StringArray* pBuffer,TBufferPos Location,BOOL Linked):
                JournalEntry(pBuffer,Location,Linked){ 

    NumChars = strlen(What);
    Deleted = new char[NumChars+1];
    strcpy(Deleted,What);

}

JEDelete::~JEDelete(){

    delete Deleted;

}

void JEDelete::Undo(TEditClient* Parent){

BOOL OverType;

    Parent->CaretPos = Where;
    Parent->HiStart  = Where;
    Parent->HiEnd    = Where;

    //We have to turn off overtype mode for the undo

    OverType = Parent->OverType;
    Parent->OverType = FALSE;

    Parent->DoInsert(Deleted,FALSE);

    //Now put it back where it was

    Parent->OverType = OverType;

    Parent->CaretPos = Where;
    Parent->HiStart  = Where;
    Parent->HiEnd    = Where;

    Parent->ScrollToCaret();

}


