/* lzpack.c - compress encode/decode file for Tar program (see file tar.c)
 * Programmer: T.V.Shaporev
 * Prepared for release 19 Oct 1990
 * Called by store() (see file store.c) and extract() (see file extract.c)
 */
/* Source:
 *	LZARI.C -- A Data Compression Program
 *	4/7/1989 Haruhiko Okumura
 *	PC-VAN        SCIENCE
 *	NIFTY-Serve    PAF01022
 *	CompuServe    74050,1022
 */

/* Modified 30.9.90 Tim Shaporev */

#include "sysup.h"

extern char v_flag;

#include <stdio.h>

#ifdef UNIX
	char *malloc();
	void free();
#endif

#ifdef MSDOS
#	include <stdlib.h>
#endif

#define	pare(x)  ((x)<<1)
#define half(x)  ((x)>>1)

#include "sysup.h"

#ifdef MSDOS
#	define __ARGS__(x) x
#endif
#ifndef __ARGS__
#	define __ARGS__(x) ()
#endif

#include "lzpack.h"

static  void	PutBit		__ARGS__(( int ));
static  void	FlushBitBuffer	__ARGS__(( void ));
static  int	GetBit		__ARGS__(( void ));
static  int	InitTree	__ARGS__(( void ));
static  void	InsertNode	__ARGS__(( int ));
static  void	DeleteNode	__ARGS__(( int ));
static  void	StartModel	__ARGS__(( void ));
static  void	UpdateModel	__ARGS__(( int ));
static  void	Output		__ARGS__(( int ));
static  void	EncodeChar	__ARGS__(( int ));
static  void	EncodePosition	__ARGS__(( int ));
static  void	EncodeEnd 	__ARGS__(( void ));
static  int	BinarySearchSym	__ARGS__(( unsigned ));
static  int	BinarySearchPos	__ARGS__(( unsigned ));
static  void	StartDecode	__ARGS__(( void ));
static  int	DecodeChar	__ARGS__(( void ));
static  int	DecodePosition	__ARGS__(( void ));

/********** Bit I/O **********/
static short PutBuf, PutMask;

#define InitPutBit (PutBuf=0, PutMask = 128)

static int  (*getbyte) __ARGS__(( void ));
static void (*putbyte) __ARGS__(( int ));

static void PutBit(bit)  /* Output one bit (bit = 0,1) */
register bit;
{
   if (bit) PutBuf |= PutMask;
   if ((PutMask >>= 1) == 0) {
      (*putbyte)((int)PutBuf);
      InitPutBit;
   }
}

static void FlushBitBuffer()  /* Send remaining bits */
{
   if (PutMask != 128) (*putbyte)((int)PutBuf);
}

static short GetBuf, GetMask;

#define InitGetBit (GetMask=0)

static int GetBit()  /* Get one bit (0 or 1) */
{
   if ((GetMask >>= 1) == 0) {
      GetBuf = (*getbyte)();
      GetMask = 128;
   }
   return (GetBuf & GetMask) != 0;
}

/********** LZSS with multiple binary trees **********/
#define N         4096    /* size of ring buffer */
#define F           60    /* upper limit for match_length */
#define THRESHOLD    2    /* encode string into position and length */
                          /* if match_length is greater than this */
#define NIL          N    /* index for root of binary search trees */

/* ring buffer of size N, with extra F-1 bytes for string comparison */
/* unsigned char text_buf[N+F-1]; */

short match_position, match_length; /* of longest match. */
/* These are set by the InsertNode() procedure. */

/* short lson[N+1], rson[N+257], dad[N+1]; */
/* left & right children & parents -- These constitute binary search trees. */

static short *core = (short *)0;

#define position_cum ((unsigned short *)core)
#define lson	     (core + N+1)
#define rson	     (core + N+1 + N+1)
#define dad	     (core + N+1 + N+1 + N+257)
#define text_buf ((unsigned char *)(core + N+1 + N+1 + N+257 + N+1))

int lzgetmem()
{
   if (!core) {
      core=(short*)malloc((N+1 + N+1 + N+257 + N+1) * sizeof(short) + N+F);
      if (!core) return -1;
   }
   return 0;
}

void lzrelmem()
{
   if (core) free((char *)core); core = (short *)0;
}

static int InitTree()  /* Initialize trees */
{
    register int i;

    /* For i = 0 to N - 1, rson[i] and lson[i] will be the right and
       left children of node i.  These nodes need not be initialized.
       Also, dad[i] is the parent of node i.  These are initialized to
       NIL (= N), which stands for 'not used.'
       For i = 0 to 255, rson[N + i + 1] is the root of the tree
       for strings that begin with character i.  These are initialized
       to NIL.  Note there are 256 trees. */
    if (!core) return -1;

    for (i=N+1; i<=N+256; i++) rson[i] = NIL;	/* root */
    for (i=0; i<N; i++)        dad[i] = NIL;	/* node */
    return 0;
}

static void InsertNode(r)
register int r;
/* Inserts string of length F, text_buf[r..r+F-1], into one of the
   trees (text_buf[r]'th tree) and returns the longest-match position
   and length via the global variables match_position and match_length.
   If match_length = F, then removes the old node in favor of the new
   one, because the old one will be deleted sooner.
   Note r plays double role, as tree node and position in buffer. */
{
    register int p;
    register int i;
    register int cmp, temp;
    register unsigned char *key;

    cmp = 1;  key = &text_buf[r];  p = N + 1 + key[0];
    rson[r] = lson[r] = NIL;  match_length = 0;
    for ( ; ; ) {
        if (cmp >= 0) {
            if (rson[p] != NIL) p = rson[p];
            else {  rson[p] = r;  dad[r] = p;  return;  }
        } else {
            if (lson[p] != NIL) p = lson[p];
            else {  lson[p] = r;  dad[r] = p;  return;  }
        }
        for (i = 1; i < F; i++)
            if ((cmp = key[i] - text_buf[p + i]) != 0)  break;
        if (i > THRESHOLD) {
            if (i > match_length) {
                match_position = (r - p) & (N - 1);
                if ((match_length = i) >= F) break;
            } else if (i == match_length) {
                if ((temp = (r - p) & (N - 1)) < match_position)
                    match_position = temp;
            }
        }
    }
    dad[r] = dad[p];  lson[r] = lson[p];  rson[r] = rson[p];
    dad[lson[p]] = r;  dad[rson[p]] = r;
    if (rson[dad[p]] == p) rson[dad[p]] = r;
    else                   lson[dad[p]] = r;
    dad[p] = NIL;  /* remove p */
}

static void DeleteNode(p)  /* Delete node p from tree */
register int p;
{
   register int q;

   if (dad[p] == NIL) return;  /* not in tree */
   if (rson[p] == NIL) q = lson[p];
   else if (lson[p] == NIL) q = rson[p];
   else {
      q = lson[p];
      if (rson[q] != NIL) {
         do {  q = rson[q];  } while (rson[q] != NIL);
         rson[dad[q]] = lson[q];  dad[lson[q]] = dad[q];
         lson[q] = lson[p];  dad[lson[p]] = q;
      }
      rson[q] = rson[p];  dad[rson[p]] = q;
   }
   dad[q] = dad[p];
   if (rson[dad[p]] == p) rson[dad[p]] = q;
   else                   lson[dad[p]] = q;
   dad[p] = NIL;
}

/********** Arithmetic Compression **********/

/*  If you are not familiar with arithmetic compression, you should read
        I. E. Witten, R. M. Neal, and J. G. Cleary,
            Communications of the ACM, Vol. 30, pp. 520-540 (1987),
    from which much have been borrowed.  */

#define M   15

/*    Q1 (= 2 to the M) must be sufficiently large, but not so
    large as the unsigned long 4 * Q1 * (Q1 - 1) overflows.  */

#define Q1  (1L << M)
#define Q2  (2 * Q1)
#define Q3  (3 * Q1)
#define Q4  (4 * Q1)
#define MAX_CUM (Q1 - 1)

#define N_CHAR  (256 - THRESHOLD + F)
    /* character code = 0, 1, ..., N_CHAR - 1 */

unsigned long /*int*/  low = 0, high = Q4, value = 0;
short shifts = 0;  /* counts for magnifying low and high around Q2 */
short char_to_sym[N_CHAR], sym_to_char[N_CHAR + 1];
unsigned short sym_freq[N_CHAR + 1];  /* frequency for symbols */
unsigned short sym_cum[N_CHAR + 1];   /* cumulative freq for symbols */
#if 0
unsigned short position_cum[N + 1];   /* cumulative freq for positions */
#endif

static void StartModel()  /* Initialize model */
{
   register i; register ch, sym;

   sym_cum[N_CHAR] = 0;
   for (sym = N_CHAR; sym >= 1; sym--) {
      ch = sym - 1;
      char_to_sym[ch] = sym;  sym_to_char[sym] = ch;
      sym_freq[sym] = 1;
      sym_cum[sym - 1] = sym_cum[sym] + sym_freq[sym];
   }
   sym_freq[0] = 0;  /* sentinel (!= sym_freq[1]) */
   position_cum[N] = 0;
   for (i = N; i >= 1; i--)
      position_cum[i - 1] = position_cum[i] + 10000 / (i + 200);
   /* empirical distribution function (quite tentative) */
   /* Please devise a better mechanism! */
}

static void UpdateModel(sym)
register int sym;
{
   register i; register c; register ch_i, ch_sym;

    if (sym_cum[0] >= MAX_CUM) {
        c = 0;
        for (i = N_CHAR; i > 0; i--) {
            sym_cum[i] = c;
            c += (sym_freq[i] = (sym_freq[i] + 1) >> 1);
        }
        sym_cum[0] = c;
    }
    for (i = sym; sym_freq[i] == sym_freq[i - 1]; i--) ;
    if (i < sym) {
        ch_i = sym_to_char[i];    ch_sym = sym_to_char[sym];
        sym_to_char[i] = ch_sym;  sym_to_char[sym] = ch_i;
        char_to_sym[ch_i] = sym;  char_to_sym[ch_sym] = i;
    }
    sym_freq[i]++;
    while (--i >= 0) sym_cum[i]++;
}

static void Output(bit) /* Output 1 bit, followed by its complements */
register bit;
{
   PutBit(bit); for (; shifts > 0; shifts--) PutBit(!bit);
}

static void EncodeChar(ch)
int ch;
{
   register int  sym;
   register unsigned long range;

   sym = char_to_sym[ch];
   range = high - low;
   high = low + (range * sym_cum[sym - 1]) / sym_cum[0];
   low +=       (range * sym_cum[sym    ]) / sym_cum[0];
   for (;;) {
      if (high <= Q2) {
         Output(0);
      } else if (low >= Q2) {
         Output(1);
         low -= Q2;  high -= Q2;
      } else if (low >= Q1 && high <= Q3) {
         shifts++;  low -= Q1;  high -= Q1;
      } else break;
      low += low;  high += high;
   }
   UpdateModel(sym);
}

static void EncodePosition(position)
int position;
{
   register unsigned long int  range;

   range = high - low;
   high = low + (range * position_cum[position    ]) / position_cum[0];
   low +=       (range * position_cum[position + 1]) / position_cum[0];
   for (;;) {
      if (high <= Q2) {
         Output(0);
      } else if (low >= Q2) {
         Output(1);
         low -= Q2;  high -= Q2;
      } else if (low >= Q1 && high <= Q3) {
         shifts++;  low -= Q1;  high -= Q1;
      } else break;
      low += low;  high += high;
   }
}

static void EncodeEnd()
{
   shifts++;
   Output(low>=Q1);
   FlushBitBuffer();
}

static int BinarySearchSym(x)
register unsigned int x;
/* 1      if x >= sym_cum[1],
   N_CHAR if sym_cum[N_CHAR] > x,
   i such that sym_cum[i - 1] > x >= sym_cum[i] otherwise */
{
    register i; register j; register k;

    i = 1;  j = N_CHAR;
    while (i<j) if (sym_cum[k=half(i+j)] > x) i=k+1; else j=k;
    return i;
}

static int BinarySearchPos(x)
register unsigned int x;
/* 0 if x >= position_cum[1],
   N - 1 if position_cum[N] > x,
   i such that position_cum[i] > x >= position_cum[i + 1] otherwise */
{
    register i; register j; register k;

    i = 1;  j = N;
    while (i<j) if (position_cum[k = half(i+j)] > x) i=k+1; else j=k;
    return i-1;
}

static void StartDecode()
{
    register i;

    for (i=0; i<M+2; i++) {
       value = pare(value) + GetBit();
    }
}

static int DecodeChar()
{
    register unsigned long range;
    register sym, ch;

    range = high - low;
    sym = BinarySearchSym((unsigned int)
        (((value - low + 1) * sym_cum[0] - 1) / range));
    high = low + (range * sym_cum[sym - 1]) / sym_cum[0];
    low +=       (range * sym_cum[sym    ]) / sym_cum[0];
    for (;;) {
        if (low >= Q2) {
            value -= Q2;  low -= Q2;  high -= Q2;
        } else if (low >= Q1 && high <= Q3) {
            value -= Q1;  low -= Q1;  high -= Q1;
        } else if (high > Q2) break;
        low += low;  high += high;
        value = pare(value) + GetBit();
    }
    ch = sym_to_char[sym];
    UpdateModel(sym);
    return ch;
}

static int DecodePosition()
{
    register unsigned long range;
    register position;

    range = high - low;
    position = BinarySearchPos((unsigned int)
        (((value - low + 1) * position_cum[0] - 1) / range));
    high = low + (range * position_cum[position    ]) / position_cum[0];
    low +=       (range * position_cum[position + 1]) / position_cum[0];
    for (;;) {
        if (low >= Q2) {
            value -= Q2;  low -= Q2;  high -= Q2;
        } else if (low >= Q1 && high <= Q3) {
            value -= Q1;  low -= Q1;  high -= Q1;
        } else if (high > Q2) break;
        low += low;  high += high;
        value = pare(value) + GetBit();
    }
    return position;
}

/********** Encode and Decode **********/

long lzencode(fget, fput)
int  (*fget) __ARGS__ ((void));
void (*fput) __ARGS__ ((int));
{
   register i; register r; register s; register c; register len;
   register last_match_length;
   register long textsize;

   getbyte = fget; putbyte = fput;
   textsize = 0;

   low = 0; high = Q4; value = 0; shifts = 0;
   InitGetBit; InitPutBit;

   StartModel();
   if (InitTree() != 0) return -1L;
   s = 0;  r = N - F;
   for (i = s; i < r; i++) text_buf[i] = ' ';
   for (len=0; len < F && (c=getbyte()) != EOF; len++)
      text_buf[r + len] = c;
   textsize = len;
   for (i=1; i<=F; i++) InsertNode(r-i);
   InsertNode(r);
   do {
      if (match_length > len) match_length = len;
      if (match_length <= THRESHOLD) {
         match_length = 1;
         EncodeChar((int)text_buf[r]);
      } else {
         EncodeChar(255-THRESHOLD+match_length);
         EncodePosition(match_position - 1);
      }
      last_match_length = match_length;
      for (i=0; i<last_match_length && (c = getbyte())!=EOF; i++) {
         DeleteNode(s);  text_buf[s] = c;
         if (s < F - 1) text_buf[s + N] = c;
         s = (s + 1) & (N - 1);
         r = (r + 1) & (N - 1);
         InsertNode(r);
      }
      textsize += i;
      while (i++ < last_match_length) {
         DeleteNode(s);
         s = (s + 1) & (N - 1);
         r = (r + 1) & (N - 1);
         if (--len) InsertNode(r);
      }
   } while (len > 0);
   EncodeEnd();
   return textsize;
}

long lzdecode(fget, fput, textsize)
/* Attemt to read after end of file IS NOT ERROR !!! Horror */
int  (*fget) __ARGS__ ((void));
void (*fput) __ARGS__ ((int));
long textsize;
{
   register i; register k; register r; register c; register j;
   register unsigned long count;

   getbyte = fget; putbyte = fput;

   low = 0; high = Q4; value = 0; shifts = 0;
   InitGetBit; InitPutBit;

   count = 0;
   StartDecode();
   StartModel();
   for (i=0; i<N-F; i++) text_buf[i] = ' ';
   r = N-F;
   while (count < textsize) {
      if ((c = DecodeChar()) < 256) {
         (*putbyte)(c);
         text_buf[r++] = c;
         r &= N-1;
         count++;
      } else {
         i = DecodePosition();
         i = (r - i - 1) & (N-1);
         j = c - 255 + THRESHOLD;
         for (k=0; k<j; k++) {
            c = text_buf[(i+k) & (N-1)];
            (*putbyte)(c);
            text_buf[r++] = c;
            r &= N-1;
            count++;
         }
      }
   }
   return count;
}
