#include <stdlib.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <dir.h>
#include <dos.h>

//NOTE:  Uses the compact memory module


/****************************************************************************\

											  DEFINES

\****************************************************************************/
#define TRUE 1
#define FALSE 0
#define BUFSIZE 1024
#define NUMKEYWORDS 83

#define MAXFILES 100
#define MAXNAMES 1000
#define MAXNAMELEN 32

#define TEXT_FILENAME	 "\n\n%s:"
#define TEXT_FUNCNAME		"\n %s()"
#define TEXT_FUNCCALLS     "\n    %s()"
#define TEXT_FUNCNOCALLS	"\n    (NONE)"

#define OUT_OF_MEM			"Out of memory!"
#define FILE_NOT_FOUND		"Unable to open file -> %s"

typedef struct filelist {
	char file[MAXPATH];
	struct filelist *next;
}FILELIST;
typedef struct funclist {
	char func[MAXNAMELEN];
	struct funclist *next;
}FUNCLIST;


/****************************************************************************\

									 FUNCTION DECLARATIONS

\****************************************************************************/
void Display_Format(const char *, ... );	//Displays program syntax
void Handle_Command_Line(int,char *[]);	//Parses command line
void exit_error(const char *, ... );		//Exits with error message
int  getline(void);								//Reads a text line from infile
int  get_filelist(void);						//Compiles filelist from filespec
void read_fnames(void);							//Scans files for function names
void show_tree(void);							//Displays functions called
char *get_parenword(char *,char *);			//Get's word in front of '('
char *strip_spaces(char *);					//Removes trailing spaces from string
void list_calls(void);							//Lists one functions calls
int  is_func(void);								//Collects all calls on a line
FUNCLIST *find_func(int);


/****************************************************************************\

										 GLOBAL VARIABLES

\****************************************************************************/
FILELIST file_head;
FUNCLIST func_head;
unsigned numfiles;								//Number of file to be processed
unsigned numfuncs;								//Number of user functions found
FILE *infile;										//File pointer of current file
char *infilespec;									//File spec from command line
char buffer1[BUFSIZE];							//Input buffer1
char buffer2[BUFSIZE];							//Input buffer2
char *bufptr;										//General buffer pointer
unsigned curlinenum;								//Line number currently being read
int  funcscalled[200];							//Numeric list of calls in function
char **comfunclist;
char *listfilename;

char version_major=1;							//Version
char version_minor=2;


/****************************************************************************\


\****************************************************************************/
/****************************************************************************\

	void main(void)

\****************************************************************************/
void main (int argc, char *argv[])
{

	printf("C/C++ Function Tree  Version %d.%02d  Copyright (c) 1993  Douglas Peterson",version_major,version_minor);
	Handle_Command_Line(argc,argv);

	if (!get_filelist())
		exit_error("Unable to open file(s) -> %s",infilespec);

	read_fnames();
	show_tree();

}

/****************************************************************************\

	void Handle_Command_Line(int argc, char *argv[])

			"Parses the command line".  This is my generic command line
		handler, feel free to incorporate it in your own software.

	argc:	argc passed from main()
	argv:	argv passed from main()

\****************************************************************************/
void Handle_Command_Line(int argc, char *argv[])
{

	while(--argc) {							//Count down the number of args
		argv++;									//Always skips the first arg

		if (**argv!='-'&&**argv!='/') {	//Is it not a switch?
			if (!infilespec) {				//Has the infile been accounted for?
				strupr(*argv);
				infilespec=*argv;				//No?, assign this arg to it
			}
			else if (!listfilename&&**argv=='@') {
				strupr(*argv);
				listfilename=(*argv+1);
			}
			else if (!comfunclist)
				comfunclist=argv;
		}//If not a switch, must be a filename
		else {
			strupr(*argv);
			switch (*(*argv+1)) {			//Must be a switch
				default:                   //Invalid switch any other
					Display_Format("Invalid switch %s",*argv);
			}//switch ()
		}//else it was a switch.
	}//next arg count;

	if (!infilespec)							//Check if in/out files specified
		Display_Format(NULL);

}

/****************************************************************************\

	void Display_Format(const char *fmt, ... )

			Displays the syntax statement, along with an optional error
		message.  Called by Handle_Command_Line() when an argument is in
		error.

	fmt:  Formatted error string

\****************************************************************************/
#define Format_Text "\n\
\nSyntax:  CTREE filespec [@listfile] [additional function(s)]\
\n\
\n     Will display user functions in filespec file(s) along with the\
\n  user function(s) called by each.  Filespec can include wildcard characters.\
\n  additional function names can be specifed following the filespec and\
\n  those will be shown as well.  Even more names can be provided by means\
\n  of a textfile with each function name on a separate line.\
\n\
\n  EXAMPLE: ctree *.c @names.txt strcpy printf\
\n\
"
void Display_Format(const char *fmt, ... )
{

	char buffer[81];						//Buffer to format string to
	va_list argptr;						//Argument pointer for vsprintf
	va_start(argptr,fmt);				//Point to first variable argument
	vsprintf(buffer,fmt,argptr);		//Format string to buffer

	printf(Format_Text);					//Print the format text string
	if (fmt)									//Unless NULL was passed as the fmt string
		printf("\nERROR: %s",buffer);	//Print the passed string as an error
	printf("\n");							//Add an additional LF for neatness
	va_end(argptr);						//End formatting

	exit(1);									//Terminate program

}

/****************************************************************************\

	void exit_error(const char *fmt, ... )

			Exits the program after displaying a message to the text screen.

	fmt:  Formatted error string

\****************************************************************************/
void exit_error(const char *fmt, ... )
{

	char str[81];
	va_list argptr;
	va_start(argptr,fmt);
	vsprintf(str,fmt,argptr);
	va_end(argptr);

	if (fmt)
		printf("\n\nERROR: %s",str);
	printf("\n");

	exit(1);

}

/****************************************************************************\

	int getline(void)

			Reads an asc][ line from typesfile, and stores it in linebuffer.
		Also increments bufptr to start of text on line.  It will continue
		to read lines from the file until a line contains an isascii()
		character and is not a comment.

	returns:	FALSE if EOF encountered, else TRUE

\****************************************************************************/
int getline(void)
{

	strcpy(buffer2,buffer1);							//Save last line
	while (1) {

		if (fgets(buffer1,BUFSIZE,infile)==NULL)	//Read line from file
			return(FALSE);
		curlinenum++;                          	//Count linenumber
		bufptr=strrchr(buffer1,'\n');					//Remove trailing LF
		if (bufptr) *bufptr=NULL;
		strtok(buffer1,"//");							//Chop off any comments
		if (*buffer1=='/'&&*buffer1=='/')
			*buffer1=NULL;									//Compensate for strtok()
		bufptr=buffer1;									//Points to 1st valid char
		while(*bufptr) {									//Search line for valid char
			if (*bufptr>0x20&&*bufptr<0x7F) {
				return(TRUE);
			}
			bufptr++;										//Increment bufptr
		}//Search buffer for valid starting character

	}//Forever

}

/****************************************************************************\

	char *get_parenword(char *buf, char *bptr)

			Returns word in front of '(' character pointed to by bptr.  If
		bptr is NULL, it will search for the first occurance of '(' in buf.
		NOTE: The '(' character and any preceeding spaces in the buffer will
		be set to NULL!

	buf:		Points to buffer where word( occurs.
	bptr:		Points to '(' character to get word from, or NULL if search is
				 needed

	returns: Pointer to NULL terminated word in front of '(' character, or
				 NULL if none found.

\****************************************************************************/
char *get_parenword(char *buf, char *bptr)
{

	int found=FALSE;

	if (!bptr) bptr=strchr(buf,'(');					 //If no bptr find '('
	if (!bptr) return(NULL);							 //If none found return NULL
	*bptr=NULL;												 //NULL terminate word
	do {
		if (isalpha(*bptr)||*bptr=='_') found=TRUE;//Found last char in word
		if (found&&(!(isalpha(*bptr)||*bptr=='_')))//Find non alpha char at
			return(strip_spaces(bptr+1));				 //begining, return it
		bptr--;												 //Search backwards
	}while (bptr>=buf);									 //Stop if start of buffer
	if (found) return(strip_spaces(bptr+1));		 //If last char was found
																 //word starts at buf begining
	return(NULL);											 //Couldn't find it

}

/****************************************************************************\

	char *strip_spaces(char *string)

			Removes trailing spaces from string by placing a NULL character
		at the first of the end spaces.

	string:	Points to string to be cropped.

	returns:	string

\****************************************************************************/
char *strip_spaces(char *string)
{

	/* Strip trailing spaces */
	int space=-1;											//Reset space index
	int x=0;
	while (string[x]) {									//Loop through string
		if (string[x]==' '&&space<0)					//If first space found
			space=x;											//Set index to position
		else if (string[x]&&string[x]!=' ')			//Otherwise
			space=-1;										//reset index
		x++;
	}
	if (space>=0)											//If index set, trailing
		string[space]=NULL;								//spaces begin at index

	return(string);										//Return the string

}

/****************************************************************************\

	void read_fnames(void)

			Scans all files in filelist, reading the names of each user
		function, and storing those names in funcnamelist[].

\****************************************************************************/
void read_fnames(void)
{

	char *funcname;
	FILELIST *fileptr=&file_head;
	FUNCLIST *funcptr=&func_head;

	while ((fileptr=fileptr->next)!=NULL) {

		if ((infile=fopen(fileptr->file,"r"))==NULL)
			exit_error(FILE_NOT_FOUND,fileptr->file);
		curlinenum=0;

		while (getline()) {					//Loop through file a line at a time
			if (*buffer1=='{') {				//Look for '{' in column 0
				/* Check previous line for function name */
				if ((funcname=get_parenword(buffer2,NULL))!=NULL) {
					if ((funcptr->next=calloc(1,sizeof(FUNCLIST)))==NULL)
						exit_error(OUT_OF_MEM);
					funcptr=funcptr->next;
					strncpy(funcptr->func,funcname,MAXNAMELEN);
					numfuncs++;
				}
			}// if '{' found on line
		}//While !EOF

		fclose(infile);

	}//next file

	/* Add exit function to list */
	if ((funcptr->next=calloc(1,sizeof(FUNCLIST)))==NULL)
		exit_error(OUT_OF_MEM);
	funcptr=funcptr->next;
	strcpy(funcptr->func,"exit");
	numfuncs++;

	/* Add command line functions to list */
	while (*comfunclist) {
		if ((funcptr->next=calloc(1,sizeof(FUNCLIST)))==NULL)
			exit_error(OUT_OF_MEM);
		funcptr=funcptr->next;
		strncpy(funcptr->func,*comfunclist++,MAXNAMELEN);
		numfuncs++;
	}

	/* Add listfile functions to list */
	if (listfilename) {
		if ((infile=fopen(listfilename,"r"))==NULL)
			exit_error(FILE_NOT_FOUND,listfilename);
		while (getline()) {
			if ((funcptr->next=calloc(1,sizeof(FUNCLIST)))==NULL)
				exit_error(OUT_OF_MEM);
			funcptr=funcptr->next;
			strncpy(funcptr->func,strip_spaces(bufptr),MAXNAMELEN);
			numfuncs++;
		}//while !EOF
	}// if listfile specified

}

/****************************************************************************\

	void show_tree(void)

			Scans each function of each file, and lists the names of the
		functions called by them.

\****************************************************************************/
void show_tree(void)
{

	FILELIST *fileptr=&file_head;
	FUNCLIST *funcptr=&func_head;

	while ((fileptr=fileptr->next)!=NULL) {

		if ((infile=fopen(fileptr->file,"r"))==NULL)
			exit_error("Unable to open file -> %s",fileptr->file);
		curlinenum=0;

		printf(TEXT_FILENAME,fileptr->file);//Display current filename
		while (getline()) {
			if (*buffer1=='{') {					//Check for beginning of function
				funcptr=funcptr->next;
				if (funcptr) {
					printf(TEXT_FUNCNAME,funcptr->func);//Display function
					list_calls();					//Show all calls made by function
				}
			}
		}//While !EOF

		fclose(infile);

	}//next file
	printf("\n");

}

/****************************************************************************\

	int is_func(void)

			Makes a numeric list of each function called on line contained in
		buffer1[].  This list is contained in funcsonline[].

	return:	The number of user functions found on the line

\****************************************************************************/
int funcsonline[50];
int is_func(void)
{

	int x;
	int count=0;
	char *word;
	FUNCLIST *funcptr;

	while (*bufptr) {										//Loop through line
		if (*bufptr=='(') {								//Find each '('
			word=get_parenword(buffer1,bufptr);    //Get word in front of '('
			funcptr=&func_head; x=1;
			while ((funcptr=funcptr->next)!=NULL) {
				if (strcmp(word,funcptr->func)==0) {
					funcsonline[count]=x;				//Table numeric function value
					count++;
				}
			x++;
			}//next func
		}
		bufptr++;
	}

	return(count);											//Return number found

}

/****************************************************************************\

	void list_calls(void)

			This functions finds and displays all the functions called by the
		currently scanned functions.  It doesn't display duplicate calls.

\****************************************************************************/
void list_calls(void)
{

	int x;
	int num;
	int found;
	int count=0;
	int *ptr=funcscalled;

	memset(funcscalled,0,sizeof(funcscalled));	//Zero numeric call table

	while (getline()) {
		if (*buffer1=='}') break;						//Read lines up to end of func

		if ((num=is_func())>0) {						//Read calls on line
			for (x=0; x<num; x++) {						//Table each in funcscalled
				ptr=funcscalled; found=FALSE;			//checking first for dupes
				while (*ptr) {								//Loop through call table
					if (funcsonline[x]==*ptr) {		//Is it dupe?
						found=TRUE;							//Yes, skip it
						break;
					}
					ptr++;
				}//while (step thru funcs called list
				if (!found) {								//Not found in table,
					*ptr=funcsonline[x];					//add it
					count++;
				}//if func call on line is not in funcscalled list
			}//next x
		}//if any user functions were found on line

	}//while !EOF or '}' end of function

	if (!count) printf(TEXT_FUNCNOCALLS);			//None found
	else {
		for (x=0; x<count; x++)							//Print those found
			printf(TEXT_FUNCCALLS,find_func(funcscalled[x])->func);
	}

}

/****************************************************************************\

	int get_filelist(void)

			This function scans the filespec on the disk and compiles a list
		of filenames which it then sorts.

	returns:	FALSE if none found, or TRUE if else

\****************************************************************************/
int get_filelist(void)
{

	int done;
	struct ffblk f;
	FILELIST *fileptr=&file_head;

	/* Read the directory, finding all the files specified */
	done=findfirst(infilespec,&f,0);
	if (done) return(FALSE);
	while (!done) {
		if ((fileptr->next=calloc(1,sizeof(FILELIST)))==NULL)
			exit_error(OUT_OF_MEM);
		fileptr=fileptr->next;
		strupr(f.ff_name);
		strcpy(fileptr->file,f.ff_name);
		numfiles++;
		done=findnext(&f);
	}//While no more image files exist

	return(TRUE);

}

/****************************************************************************\

	FUNCLIST *find_func(int index)

			Searches for a link in the link-list FUNCLIST, by index.

	index:	0=header, 1 - x link number of list.

	returns:	Pointer to link[index] or NULL if index beyond end of list.

\****************************************************************************/
FUNCLIST *find_func(int index)
{

	int x;
	FUNCLIST *ptr=&func_head;
	for (x=0; x<index; x++)
		if ((ptr=ptr->next)==NULL)
			return(NULL);
	return(ptr);

}

