// IPX.H -- Defines for IPX routines
// by Allen Brunson  06/01/94


#ifndef _IPX_H                                     // If not already included
#define _IPX_H                                     // Define this file


/****************************************************************************/
/*                                                                          */
/***  General definitions                                                 ***/
/*                                                                          */
/****************************************************************************/

#ifndef  TRUEFALSE                                 // If TRUEFALSE not set
#define  TRUEFALSE                                 // Define it
#define  FALSE               0                     // Boolean false
#define  OFF                 0                     //  defines
#define  NO                  0
#define  TRUE                1                     // Boolean true
#define  ON                  1                     //  defines
#define  YES                 1
#endif

#ifndef  BYTE                                      // If BYTE not yet defined
#define  BYTE                                      // Set definition
typedef  unsigned char byte;                       // Typedef byte type
#endif
#ifndef  WORD                                      // If WORD not yet defined
#define  WORD                                      // Set definition
typedef  unsigned int  word;                       // Typedef word type
#endif


/****************************************************************************/
/*                                                                          */
/***  Communication size definitions                                      ***/
/*                                                                          */
/****************************************************************************

The define IPXCOMMSIZE is useful in determining how much memory to allocate
for IPXLIB before calling ipxStart().  Each send and receive ECB/buffer
pair will require IPXCOMMSIZE bytes of memory.  So if you have 8 send
ECB/buffers and 20 receive ECB/buffers, then the total bytes you would need
would be (8 + 20) * IPXCOMMSIZE.                                            */

#define  IPXCOMMSIZE                             \
 (sizeof (struct ECB) +                          \
  sizeof (struct IPXPKTHDR) + IPXDATASIZE +      \
  sizeof (struct IPXEVENT))


/****************************************************************************/
/*                                                                          */
/***  IPX error return values                                             ***/
/*                                                                          */
/****************************************************************************

These error codes are completely "made up" and bear no relation to the error
codes that the IPX driver itself returns, which can vary depending on the
circumstances.  In very rare cases, an error code other than one of the
following might be returned.  See the documentation for details.            */

#define  ipxeNOERR               0                 // No error
#define  ipxeNOIPX               2                 // IPX driver isn't loaded
#define  ipxeIPXNOTSTARTED       3                 // ipxStart() not called
#define  ipxeIPXSTARTED          4                 // ipxStart() called twice
#define  ipxeBADCOMMPARMS        5                 // Bad parms to ipxStart()
#define  ipxeMEMTOOSMALL         6                 // Memory block too small
#define  ipxeSOCKETTABLEFULL     7                 // Socket table is full
#define  ipxeSOCKETOPEN          8                 // Socket already open
#define  ipxeSOCKETNOTOPEN       9                 // Socket not open
#define  ipxeNOFREESENDECB      10                 // No free send ECBs
#define  ipxeRECVCANCEL         11                 // Receive was canceled
#define  ipxeSENDCANCEL         12                 // Send was canceled
#define  ipxeRECVPKTTOOBIG      13                 // Received packet too big
#define  ipxeSENDPKTBAD         14                 // Packet was malformed
#define  ipxeSENDBADROUTE       15                 // Packet undeliverable
#define  ipxeSENDNETFAIL        16                 // Hardware or net failure


/****************************************************************************/
/*                                                                          */
/***  IPX structure definitions                                           ***/
/*                                                                          */
/****************************************************************************

All word-sized values in the IPX packet header are stored as hi-lo, in
opposition to standard Intel byte-ordering, which is lo-hi.  Most words in
the ECB are also hi-lo, with a couple of exceptions.  All addresses in the
ECB are stored in standard Intel offset-segment format.                     */

struct IPXADDR                                     // IPX address
  {
    byte net[4];             // hi-lo              //  0: Network number
    byte node[6];            // hi-lo              //  4: Node number
    word socket;             // hi-lo              // 10: Socket number
  };

struct IPXADDRFULL                                 // IPX address/route info
  {
    byte net[4];             // hi-lo              //  0: Network number
    byte node[6];            // hi-lo              //  4: Node number
    word socket;             // hi-lo              // 10: Socket number
    byte immedAddr[6];       // hi-lo              // 12: Immediate address
  };

struct IPXEVENT                                    // IPX event queue entry
  {
    word eCompCode;          // lo-hi              // Event completion code
    struct ECB near *ECBptr; // lo-hi              // Pointer to ECB
  };

struct IPXPKTHDR                                   // IPX packet header
  {
    word checksum;           // hi-lo              //  0: Packet checksum
    word length;             // hi-lo              //  2: Packet length
    byte transCtrl;                                //  4: Transport control
    byte pktType;                                  //  5: Packet type
    struct IPXADDR dst;                            //  6: Dest address
    struct IPXADDR src;                            // 18: Source address
  };

struct ECB                                         // Event Control Block
  {
    void far *linkAddr;      // off-seg            //  0: Link address
    void far *ESRaddr;       // off-seg            //  4: ESR address
    byte inUse;                                    //  8: In use flag
    byte compCode;                                 //  9: Completion code
    word socket;             // hi-lo              // 10: Socket number
    byte ipxWork[4];                               // 12: IPX workspace
    byte driverWork[12];                           // 16: Driver workspace
    byte immed[6];           // hi-lo              // 28: Immediate address
    word fragCnt;            // lo-hi              // 34: Fragment count
    void far *pktHdr;        // off-seg            // 36: Pkt header address
    word pktHdrSize;         // lo-hi              // 40: Pkt header size
    void far *pktData;       // off-seg            // 42: Pkt data address
    word pktDataSize;        // lo-hi              // 46: Pkt data size
  };


/****************************************************************************/
/*                                                                          */
/***  Global data                                                         ***/
/*                                                                          */
/****************************************************************************/

extern word ipxRecvMax;                            // Max receive ECBs in use
extern word ipxSendMax;                            // Max send ECBs in use


/****************************************************************************/
/*                                                                          */
/***  Function declarations                                               ***/
/*                                                                          */
/****************************************************************************/

// IPX.ASM                                         // Assembler IPX routines

word ipxAddrLocal(struct IPXADDRFULL               // Gets local IPX address
 far *ipxAddr);
word ipxLibVer(void);                              // Gets IPXLIB version
word ipxRecvChk(void);                             // Checks for waiting pkts
word ipxRecvPkt(void far *data, word dataSize,     // Receives a packet
 struct IPXADDRFULL far *ipxAddr,
 word far *pktSize, word far *error);
word ipxSendChk(void);                             // Checks for send errors
word ipxSendErr(void far *data, word dataSize,     // Retrieves send errors
 struct IPXADDRFULL far *ipxAddr,
 word far *pktSize, word far *error);
word ipxSendPkt(void far *data, word dataSize,     // Sends a packet
 struct IPXADDRFULL far *ipxAddr);
word ipxStart(word recvCnt, word sendCnt,          // Starts up IPXLIB
 word dataSize, word socket, void far *memPtr,
 word memSize);
word ipxStop(void);                                // Stops IPXLIB

// IPXUTIL.C                                       // Utility functions

void ipxAddrBrd(struct IPXADDRFULL *ipxAddr);      // Sets addr for broadcast
int ipxAddrCmp(struct IPXADDRFULL *ipxAddr1,       // Compares two IPX
 struct IPXADDRFULL *ipxAddr2);                    //  addresses
void ipxAddrCpy(struct IPXADDRFULL *ipxAddrDst,    // Copies one addr to
 struct IPXADDRFULL *ipxAddrSrc);                  //  another
char *ipxAddrStr(char *str,                        // Converts an IPX address
 struct IPXADDRFULL *ipxAddr);                     //  to a string
char *ipxAddrStrLong(char *str,                    // Converts IPX address to
 struct IPXADDRFULL *ipxAddr);                     //  long string


#endif                                             // End of _IPX_H
