// NETWORK.C -- High-level IPX routines for IPXTEST
// by Allen Brunson  06/01/94


#ifdef   __TURBOC__    // If a Borland compiler
#include <alloc.h>     // malloc(), free()
#else
#include <malloc.h>    // malloc(), free()
#endif

#include <dos.h>       // delay()
#include <stdio.h>     // sprintf()
#include <stdlib.h>    // atoi()
#include <string.h>    // strcpy()
#include <time.h>      // needed for randomize()
#include "ipxtest.h"   // IPXtest-specific defines
#include "ipx.h"       // IPXLIB defines
#include "ipxcfg.h"    // IPXLIB configuration settings


/****************************************************************************/
/*                                                                          */
/***  Network data                                                        ***/
/*                                                                          */
/****************************************************************************/

struct IPXTESTPKT recvPkt;                         // Receive packet
struct IPXADDRFULL recvAddr;                       // Receive address
char   recvAddrStr[30];                            // Receive address string

struct IPXTESTPKT sendPkt;                         // Send packet

struct IPXADDRFULL ipxAddrBroad;                   // Address for broadcasts
char   nameStr[NAMELEN + 1] = "NoName";            // Name string
struct USER user[USERTOTAL];                       // User table

byte   flurryMode = FALSE;                         // Flurry mode flag
unsigned long int flurryPackets = 0;               // Flurry packets received

#ifndef DEBUG                                      // If DEBUG not defined
void   *memPtr;                                    // Memory pointer
#else                                              // If DEBUG defined
struct IPXDATA ipxData;                            // Used for IPXLIB data
#endif


/****************************************************************************/
/*                                                                          */
/***  netStart()                                                          ***/
/*                                                                          */
/****************************************************************************

This procedure sets up the network data and displays opening messages.      */

void netStart(void)                                // Begin netStart()
  {
    char addrStr[30];                              // Address string
    word IPXLIBver;                                // IPXLIB version
    word rval;                                     // Function return value
    unsigned long int memSize;                     // Memory for IPXLIB

    message("Starting up IPX communications ..."); // Display opening message

    memSize =                                      // Calculate memory
     (IPXRECVCNT + IPXSENDCNT) * IPXCOMMSIZE;      //  necessary for IPXLIB

    if (memSize > 65500L)                          // If memory size too big
      {
        message("IPX memory required is greater "  // Display error message
        "than 64k.  Cannont continue.");
        return;                                    // Return
      }

    #ifdef __TURBOC__                              // If a Borland compiler
    randomize();                                   // Init random numbers
    #endif

    #ifdef DEBUG                                   // If DEBUG is defined

    rval = ipxStart(IPXRECVCNT, IPXSENDCNT,        // Start up IPXLIB
     IPXDATASIZE, IPXSOCKET, &ipxData,
     sizeof ipxData);

    #else                                          // If DEBUG isn't defined

    sprintf(str, "Allocating %d bytes of "         // Format message
     "memory ...", memSize);
    message(str);                                  // Print message

    memPtr = malloc((int) memSize);                // Allocate memory

    if (memPtr == NULL)                            // If alloc failed
      {
        message("Error allocating memory.");       // Print error
        endProgram = TRUE;                         // Set end program flag
        return;                                    // Return
      }

    rval = ipxStart(IPXRECVCNT, IPXSENDCNT,        // Start up IPXLIB
     IPXDATASIZE, IPXSOCKET, memPtr,
     (word) memSize);

    #endif                                         // End if DEBUG

    if (rval)                                      // If ipxStart() error
      {
        err(rval);                                 // Display error
        endProgram = TRUE;                         // Set end program flag
        return;                                    // Return to caller
      }

    err(ipxAddrLocal(&user[0].addr));              // Get local address

    ipxAddrStr(addrStr, &user[0].addr);            // Convert address to str
    sprintf(str, "Local address is %s", addrStr);  // Set up string
    message(str);                                  // Print message

    user[0].inUse = TRUE;                          // First entry is this PC
    strcpy(user[0].name, nameStr);                 // Set name string

    ipxAddrCpy(&ipxAddrBroad, &user[0].addr);      // Set up ipxAddrBroad
    ipxAddrBrd(&ipxAddrBroad);                     //  for broadcasting

    IPXLIBver = ipxLibVer();                       // Get version number
    sprintf(str, "IPXLIB version is v%d.%02d",     // Set up string
     IPXLIBver >> 8, IPXLIBver & 0x00FF);
    message(str);                                  // Print message
    message("");                                   // Blank line

    message("Use the NAME command to "             // Display name and
     "set your name,");                            //  ping reminder
    message("then use PING to find other users.");
    message("");                                   // Blank line
  }                                                // End netStart()


/****************************************************************************/
/*                                                                          */
/***  netStop()                                                           ***/
/*                                                                          */
/****************************************************************************

This procedure shuts down IPXLIB.                                           */

void netStop(void)                                 // Begin netStop()
  {
    message("");                                   // Print blank line
    message("Stopping IPX communications ...");    // Display final message
    err(ipxStop());                                // Stop IPX

    #ifndef DEBUG                                  // If DEBUG not defined
    if (memPtr != NULL) free(memPtr);              // Free IPXLIB memory
    #endif                                         // End if not DEBUG
  }                                                // End netStop()


/****************************************************************************/
/*                                                                          */
/***  recvBroadcast()                                                     ***/
/*                                                                          */
/****************************************************************************

This procedure processes a received broadcast packet.                       */

void recvBroadcast(void)                           // Begin recvBroadcast()
  {
    sprintf(str, "Broadcast from %s: %s",          // Format string
     recvPkt.name, recvPkt.msg);
    message(str);                                  // Display string

    strcpy(cmdStr, recvPkt.msg);                   // Interpret this string
    inputFlag = 2;                                 //  as a command
  }                                                // End recvBroadcast()


/****************************************************************************/
/*                                                                          */
/***  recvMessage()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure processes a received message packet.                         */

void recvMessage(void)                             // Begin recvMessage()
  {
    sprintf(str, "Message from %s: %s",            // Format string
     recvPkt.name, recvPkt.msg);
    message(str);                                  // Display string

    strcpy(cmdStr, recvPkt.msg);                   // Interpret this string
    inputFlag = 2;                                 //  as a command
  }                                                // End recvMessage()


/****************************************************************************/
/*                                                                          */
/***  recvPacket()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure is the front end for processing incoming packets.            */

void recvPacket(void)                              // Begin recvPacket()
  {
    word pktSize;                                  // Received packet size
    word error;                                    // Error value
    word rval;                                     // Return value

    rval = ipxRecvPkt(&recvPkt, sizeof recvPkt,    // Get waiting
     &recvAddr, &pktSize, &error);                 //  packet
    err(error);                                    // Display error message

    if (!rval || rval == ipxeIPXNOTSTARTED)        // Return if none
      return;

    ipxAddrStr(recvAddrStr, &recvAddr);            // Convert address to str

    if (recvPkt.signature != SIGNATURE)            // If signature wrong
      {
        sprintf(str, "Packet with unknown "        // Format error string
         "signature received from %s",
         recvAddrStr);
        message(str);                              // Display it
        return;                                    // Return
      }

    userSave();                                    // Record this user

    if (recvPkt.type == pFLURRY)                   // If flurry packet
      {
        flurryPackets++;                           // Increment flurry count

        if (!(flurryPackets % 1000))               // If 1000 have gone by
          {
            sprintf(str,                           // Format string
             "Flurry packet total: %lu",
             flurryPackets);

            message(str);                          // Display string
          }

        return;                                    // Return
      }

    sprintf(str, "Received packet from "           // Regular packet message
     "%s, size %d bytes",
      recvAddrStr, pktSize);
    message(str);                                  // Display message

    switch (recvPkt.type)                          // Decision on packet type
      {
        case pBROADCAST:                           // Broadcast packet
          recvBroadcast();                         // Process it
          break;

        case pMESSAGE:                             // Message packet
          recvMessage();                           // Process it
          break;

        case pPING:                                // Ping packet
          recvPing();                              // Process the ping
          break;

        case pPINGRESPONSE:                        // Ping response
          recvPingResponse();                      // Process the response
          break;

        default:                                   // Unknown type
          sprintf(str, "Unknown packet type from " // Format string
           "%s", recvAddrStr);
          message(str);                            // Display message
          break;
      }
  }                                                // End recvPacket()


/****************************************************************************/
/*                                                                          */
/***  recvPing()                                                          ***/
/*                                                                          */
/****************************************************************************

This procedure processes received ping packets.                             */

void recvPing(void)                                // Begin recvPing()
  {
    message("Got ping packet");                    // Display message

    if (!ipxAddrCmp(&user[0].addr, &recvAddr))     // Return if sender is 
      return;                                      //  this PC

    // Wait up to one full second so the PC doing
    //  the pinging isn't deluged with replies all
    //  arriving at the exact same instant.
    
    #ifdef __TURBOC__                              // If a Borland compiler
    delay(random(1000));                           // Wait for up to a second
    #endif

    sprintf(str, "Sending ping response to %s",    // Format string
     recvAddrStr);
    message(str);                                  // Display string

    sendPkt.signature = SIGNATURE;                 // Set up send packet
    sendPkt.type = pPINGRESPONSE;
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);

    ipxSendPkt(&sendPkt,                           // Send ping response
     sizeof sendPkt - (MSGLEN + 1), &recvAddr);                                   //  to sender
  }                                                // End recvPing()


/****************************************************************************/
/*                                                                          */
/***  recvPingResponse()                                                  ***/
/*                                                                          */
/****************************************************************************

This procedure processes received ping responses.                           */

void recvPingResponse(void)                        // Begin recvPingResponse()
  {
    message("Got ping response");                  // Display message
  }                                                // End recvPingResponse()


/****************************************************************************/
/*                                                                          */
/***  sendBroadcast()                                                     ***/
/*                                                                          */
/****************************************************************************

This procedure broadcasts a message to all users.                           */

void sendBroadcast(void)                           // Begin sendBroadcast()
  {
    word i = 0;                                    // String index
    word len;                                      // String length
    char *sendStr;                                 // String to send

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past B command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    sendStr = &cmdStr[i];                          // Set send string

    len = strlen(sendStr);                         // Get string length

    if (!len)                                      // If no string
      {
        message("You must enter a message to "     // Display error
         "broadcast after the B command.");        //  message
        return;                                    // Return to caller
      }

    if (len > MSGLEN) sendStr[MSGLEN] = 0;         // Truncate if too long

    sendPkt.signature = SIGNATURE;                 // Set up packet for
    sendPkt.type = pBROADCAST;                     //  send
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);
    strcpy(sendPkt.msg, sendStr);

    for (i = 0; i < USERTOTAL; i++)                // Loop for all users
      {
        if (!user[i].inUse) continue;              // Don't do empty slots

        err(ipxSendPkt(&sendPkt, sizeof sendPkt,   // Send the packet
         &user[i].addr));                          //  to this user
      }
  };                                               // End sendBroadcast()


/****************************************************************************/
/*                                                                          */
/***  sendErr()                                                           ***/
/*                                                                          */
/****************************************************************************

This procedure checks for send errors.  If one is found, an error message
is printed.                                                                 */

void sendErr(void)                                 // Begin sendErr()
  {
    word rval;                                     // Return value
    word pktSize;                                  // Size of packet
    word error;                                    // Error value
    char sendAddrStr[30];                          // Send address str
    struct IPXADDRFULL sendAddrErr;                // IPX address

    user[0].ipxRecvMax = ipxRecvMax;               // Update this PC's
    user[0].ipxSendMax = ipxSendMax;               //  statistics

    rval = ipxSendChk();                           // Check for send errors
    if (!rval || rval == ipxeIPXNOTSTARTED)        // Return if none
      return;

    rval = ipxSendErr(&sendPkt, sizeof sendPkt,    // Get the undelivered
     &sendAddrErr, &pktSize, &error);              //  packet

    ipxAddrStr(sendAddrStr, &sendAddrErr);         // Convert addr to string
    sprintf(str, "Error sending packet to %s",     // Format string
     sendAddrStr);
    message(str);                                  // Display string
    err(error);                                    // Display error
  }                                                // End sendErr()


/****************************************************************************/
/*                                                                          */
/***  sendFlurry()                                                        ***/
/*                                                                          */
/****************************************************************************

This procedure will send "flurry" packets to all known users if flurry mode
is on.  Note: You should not duplicate this kind of thing in your own
program!  Running IPXTEST will show that continuously sending packets will
quickly overrun the total number of both send and receive ECBs.             */

void sendFlurry(void)                              // Begin sendFlurry()
  {
    static word i = 0;                             // Loop counter

    if (flurryMode == FALSE) return;               // Return if no flurries

    sendPkt.signature = SIGNATURE;                 // Set signature
    sendPkt.type = pFLURRY;                        // Set packet type
    sendPkt.ipxRecvMax = ipxRecvMax;               // Set max recv ECB count
    sendPkt.ipxSendMax = ipxSendMax;               // Set max send ECB count
    strcpy(sendPkt.name, nameStr);                 // Copy user name

    i++; if (i >= USERTOTAL) i = 0;                // Increment i

    if (user[i].inUse == FALSE) return;            // Don't do empty slots

    err(ipxSendPkt(&sendPkt,                       // Send packet to this
     sizeof sendPkt - (MSGLEN + 1),                //  user
     &user[i].addr));
  }                                                // End sendFlurry()


/****************************************************************************/
/*                                                                          */
/***  sendFlurryMode()                                                    ***/
/*                                                                          */
/****************************************************************************

This procedure turns flurry mode on or off or resets the flurry packet
count.                                                                      */

char badFlurryStr[] = "You must enter ON, OFF, "   // Error string
 "or RESET after the FLURRY command.";

void sendFlurryMode(void)                          // Begin sendFlurryMode()
  {
    word i = 0;                                    // String index
    char *parmStr;                                 // Parameter string

    strupr(cmdStr);                                // String to uppercase

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past F command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    parmStr = &cmdStr[i];                          // Set parameter string

    i = strlen(parmStr);                           // Get parm str length
    while (parmStr[i - 1] == ' ' && i >= 1)        // Remove trailing
      {parmStr[i - 1] = 0; i--;}                   //  spaces

    if (!i)                                        // If no parms
      {message(badFlurryStr); return;}             // Return

    if (!strcmp(parmStr, "ON"))                    // If ON parameter given
      {
        flurryMode = ON;                           // Turn on flurry mode
        message("Flurry mode turned on.");         // Display message
        return;                                    // Return
      }

    if (!strcmp(parmStr, "OFF"))                   // If OFF parameter given
      {
        flurryMode = OFF;                          // Turn off flurry mode
        message("Flurry mode turned off.");        // Display message
        return;                                    // Return
      }

    if (!strcmp(parmStr, "RESET"))                 // If RESET given
      {
        flurryPackets = 0;                         // Reset packet count
        message("Flurry packet count "             // Display message
         "set to zero.");
        return;                                    // Return
      }

    message(badFlurryStr);                         // Display error message
  }                                                // End sendFlurryMode()


/****************************************************************************/
/*                                                                          */
/***  sendMessage()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure sends a message to one user.                                 */

void sendMessage(void)                             // Begin sendMessage()
  {
    word i = 0;                                    // String index
    word len;                                      // String length
    char *sendStr;                                 // String to send
    int  userNum;                                  // User number

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past M command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    userNum = atoi(&cmdStr[i]);                    // Get user number

    if (userNum <= 0 || userNum > USERTOTAL ||     // If user number out of
     !user[userNum - 1].inUse)                     //  range or not in use
      {
        message("Invalid user number.  "           // Display message
         "Enter D for list."); 
        return;                                    // Return
      }

    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past user number
    while (cmdStr[i] == ' ') i++;                  // Go past spaces

    sendStr = &cmdStr[i];                          // Set string to send

    len = strlen(sendStr);                         // Get string length

    if (!len)                                      // If no string
      {
        message("You must enter a message to "     // Display error
         "send after the user number.");           //  message
        return;                                    // Return to caller
      }

    if (len > MSGLEN) sendStr[MSGLEN] = 0;         // Truncate if too long

    sendPkt.signature = SIGNATURE;                 // Set up packet for
    sendPkt.type = pMESSAGE;                       //  sending
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);
    strcpy(sendPkt.msg, sendStr);

    err(ipxSendPkt(&sendPkt, sizeof sendPkt,       // Send the packet
     &user[userNum - 1].addr));
  };                                               // End sendBroadcast()


/****************************************************************************/
/*                                                                          */
/***  sendPing()                                                          ***/
/*                                                                          */
/****************************************************************************

This procedure sends out an "is anybody there?" request.                    */

void sendPing(void)                                // Begin sendPing()
  {
    userClear();                                   // Clear out user table

    sendPkt.signature = SIGNATURE;                 // Set packet signature
    sendPkt.type = pPING;                          // Set packet type
    sendPkt.ipxRecvMax = ipxRecvMax;
    sendPkt.ipxSendMax = ipxSendMax;
    strcpy(sendPkt.name, nameStr);                 // Set user name

    message("Sending ping ...");                   // Display message

    err(ipxSendPkt(&sendPkt,                       // Send the ping
     sizeof sendPkt - (MSGLEN + 1),
     &ipxAddrBroad));
  }                                                // End sendPing()


/****************************************************************************/
/*                                                                          */
/***  setName()                                                           ***/
/*                                                                          */
/****************************************************************************

This procedure sets the local user's name.                                  */

void setName(void)                                 // Begin setName()
  {
    word i = 0;                                    // String index
    word len;                                      // String length
    char *name;                                    // Pointer to name

    if (strlen(cmdStr) == 0) return;               // Return if no input str

    message("");                                   // Print blank line

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past NAME command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    name = &cmdStr[i];                             // Set name string

    len = strlen(name);                            // Get string length

    if (!len)                                      // If no name given
      {
        message("You must specify a name "         // Display error
         "after the NAME command.");               //  message
        return;                                    // Return to caller
      }

    if (len > NAMELEN) name[NAMELEN] = 0;          // Truncate if too long

    strcpy(nameStr, &cmdStr[i]);                   // Copy name string
    strcpy(user[0].name, nameStr);                 // Save as first name
    sprintf(str, "Name set to '%s'", nameStr);     // Format string
    message(str);                                  // Display it
    message("");
  }                                                // End setName()


/****************************************************************************/
/*                                                                          */
/***  statDisplay()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure displays statistics on maximum ECB/packet pair usage or
resets the statistics to zeroes if the RESET parameter is given.            */

void statDisplay(void)                             // Begin statDisplay()
  {
    word recvCnt = IPXRECVCNT;                     // Receive ECB/buffers
    word sendCnt = IPXSENDCNT;                     // Send ECB/buffers
    word  i = 0;                                    // String index
    char *parmStr;                                 // Parameter str pointer

    strupr(cmdStr);                                // String to uppercase

    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    while (cmdStr[i] != ' ' && cmdStr[i] != 0) i++;// Go past STAT command
    while (cmdStr[i] == ' ') i++;                  // Go past spaces
    parmStr = &cmdStr[i];                          // Set parameter string

    i = strlen(parmStr);                           // Get parm str length
    while (parmStr[i - 1] == ' ' && i >= 1)        // Remove trailing
     {parmStr[i - 1] = 0; i--;}                    //  spaces

    if (!i)                                        // If no parms
      {
        sprintf(str, "Max receive ECBs in use: "   // Format receive message
         "%03d of %03d", ipxRecvMax, recvCnt);
        message(str);                              // Display it

        sprintf(str, "   Max send ECBs in use: "   // Format send message
         "%03d of %03d", ipxSendMax, sendCnt);
        message(str);                              // Display it

        return;                                    // Return
      }

    if (!strcmp(parmStr, "RESET"))                 // If RESET given
      {
        ipxRecvMax = 0; ipxSendMax = 0;            // Reset statistics
        message("Statistics reset.");              // Display message
        return;                                    // Return
      }

    message("Unknown STAT parameter.");            // Display message
  }                                                // End statDisplay()


/****************************************************************************/
/*                                                                          */
/***  userClear()                                                         ***/
/*                                                                          */
/****************************************************************************

This procedure clears out the user table.                                   */

void userClear(void)                               // Begin userClear()
  {
    word i;                                        // Loop counter

    for (i = 1; i < USERTOTAL; i++)                // Clear all entries
      user[i].inUse = FALSE;                       //  except the first
  }                                                // End userClear()


/****************************************************************************/
/*                                                                          */
/***  userDisplay()                                                       ***/
/*                                                                          */
/****************************************************************************

This procedure displays the list of known IPXTEST users.                    */

void userDisplay(void)                             // Begin userDisplay()
  {
    word i;                                        // Loop counter
    word total = 0;                                // Total users
    char ipxAddr[60];                              // Address string
    char local;                                    // Local PC or not

    message("");                                   // Print blank line
    message("    Known users:");                   // Display opening
    message("");
    message("    Network  Node         Sckt Immediate     Rcv Snd   Name");
    message("    -------  ----         ---- ---------     --- ---   ----");
    message("");

    for (i = 0; i < USERTOTAL; i++)                // Loop for all users
      {
        if (user[i].inUse == FALSE) continue;      // Continue if empty

        total++;                                   // Increment total

        if (!ipxAddrCmp(&user[0].addr, 
         &user[i].addr))
          local = '*'; else local = ' ';

        ipxAddrStrLong(ipxAddr, &user[i].addr);    // Convert address to str

        sprintf(str, "%2d: %s  %03d %03d  %c%s",   // Format message string
         i + 1, ipxAddr, user[i].ipxRecvMax,
         user[i].ipxSendMax, local, user[i].name);

        message(str);                              // Print string
      }

    if (total == 0) message("(None)");             // Display if no users

    message("");                                   // Print blank line
  }                                                // End userDisplay()


/****************************************************************************/
/*                                                                          */
/***  userSave()                                                          ***/
/*                                                                          */
/****************************************************************************

This procedure copies user data from the current received packet to the
user table.  If the user is already in the table, the information is
updated.                                                                    */

void userSave(void)                                // Begin userSave()
  {
    word i;                                        // Loop counter

    for (i = 0; i < USERTOTAL; i++)                // Loop for all entries
      {
        if (!ipxAddrCmp(&recvAddr, &user[i].addr)) // Find first slot that
          break;                                   //  either matches this
                                                   //  address or is empty,
        if (!user[i].inUse) break;                 //  whichever comes first
      }
    
    if (i >= USERTOTAL)                            // If no free slots
      {
        message("User table is full.");            // Display error
        return;
      }

    user[i].inUse = TRUE;                          // Indicate entry is used
    user[i].ipxRecvMax = recvPkt.ipxRecvMax;       // Save receive statistic
    user[i].ipxSendMax = recvPkt.ipxSendMax;       // Save send statistic
    ipxAddrCpy(&user[i].addr, &recvAddr);          // Save address
    strcpy(user[i].name, recvPkt.name);            // Save name
  }                                                // End userSave()
