/*********************************************************************/
/*                                                                   */
/*                          SBICONV.C                                */
/*                                                                   */
/*       A utility to convert SBI (SoundBlaster Instrument)          */
/*          files to commented C or assembly source code.            */
/*                                                                   */
/* Steve Robison 73144,2540                                          */
/* Hereby released into the public domain.                           */
/*********************************************************************/
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dos.h>
#include <dir.h>
#include <io.h>


typedef int BOOL;
const BOOL True = (1 == 1);
const BOOL False = (1 == 0);

typedef unsigned char BYTE;

struct SBI_FILE_STRUCT {
	char header[3];			// "SBI"
    BYTE one_a;             // '0x1A' ???
    char inst_name[8];		// instrument name
    char comments[24];		// comments field
    BYTE mod_prop;			// modulator sound properties
	BYTE car_prop;			// carrier sound properties
	BYTE mod_vol;			// modulator volume
	BYTE car_vol;			// carrier volume
	BYTE mod_a_d;			// modulator attack/decay
	BYTE car_a_d;			// carrier attack/decay
	BYTE mod_s_r;			// modulator sustain/release
	BYTE car_s_r;			// carrier sustain/release
	BYTE mod_wav;			// modulator waveform
	BYTE car_wav; 			// carrier waveform
	BYTE synth_mode;		// synthesis mode
	BYTE reserved[5];		// reserved for future use
};
// SBI file len
#define SBI_LEN  52
// output file buffer len
#define OUT_LEN 1000

void help(void);
// convert SBI to Assembly code
void sbi2asm(struct SBI_FILE_STRUCT *sbi, char out[]);
// convert SBI to C code
void sbi2c(struct SBI_FILE_STRUCT *sbi, char out[]);

main(int argc, char *argv[])
{
	char filename[MAXPATH];  			// SBI filename
    char drive[MAXDRIVE];				// SBI filname parts
	char dir[MAXDIR];
	char file[MAXFILE];
	char ext[MAXEXT];
    FILE *sbi_file;
    struct SBI_FILE_STRUCT sbi_struct;
    char outfilename[MAXPATH]; 			// ascii out put file (c or asm)
    FILE *out_file;
    char out_array[OUT_LEN];            // char array to store output
	BOOL asmflag = False;
    BOOL cflag = True;   				// default
    BOOL helpflag = False;
    BOOL infileflag = False;            // supplied on command line?
    BOOL outfileflag = False;
    int i;
    // parse command line
    if (argc == 1)
		helpflag = True;
    for (i = 1; i < argc; i++) {
		if (!strcmpi(argv[i], "/a")) {
        	asmflag = True;
            cflag = False;
        }
		else if (!strcmpi(argv[i], "/c")) {
        	cflag = True;
            asmflag = False;
        }
		else if ( (!strcmpi(argv[i], "/h")) || (!strcmp(argv[i], "/?")) )
        	helpflag = True;
        else if (!infileflag) {
        	strncpy(filename, argv[i], MAXPATH);
            strupr(filename);
            infileflag = True;
        }
        else if (!outfileflag) {
        	strncpy(outfilename, argv[i], MAXPATH);
            strupr(outfilename);
            outfileflag = True;
        }
    }
    //  call help
    if (helpflag) {
    	help();
        return(EXIT_FAILURE);
    }
    // check for infileflag
	if (!infileflag) {
		puts("Must specify input SBI file.");
		return(EXIT_FAILURE);
	}
    // parse infile name
	fnsplit(filename, drive, dir, file, ext);
    // assume 'SBI' extension if none given
    if (ext[0] == 0) {
    	strcpy(ext, ".SBI");
		fnmerge(filename, drive, dir, file, ext);
    }
    // open file
    sbi_file = fopen(filename, "rb");
    if (!sbi_file) {
    	printf("Unable to open file %s.\n", filename);
		return(EXIT_FAILURE);
	}
    // check size
    if (filelength(fileno(sbi_file)) != SBI_LEN) {
    	printf("File %s is incorrect size.\n", filename);
        fclose(sbi_file);
        return(EXIT_FAILURE);
    }
    // read file into structure
    fseek(sbi_file, 0, SEEK_SET);
    if (fread(&sbi_struct, sizeof(struct SBI_FILE_STRUCT), 1, sbi_file) != 1) {
		printf("Error reading file %s.\n", filename);
        fclose(sbi_file);
        return(EXIT_FAILURE);
    }
    // close input file
    fclose(sbi_file);
    // verify header
    if (memcmp(sbi_struct.header, "SBI", 3)) {
    	printf("File %s is not an SBI file.\n", filename);
        fclose(sbi_file);
        return(EXIT_FAILURE);
    }
    // c or asm format?
    if (!outfileflag) {
       	strcpy(outfilename, file);
    	if (asmflag)
            strcat(outfilename, ".ASM");
		else if (cflag)
            strcat(outfilename, ".C");
    }
    // open output file
    out_file = fopen(outfilename, "wt");
    if (!out_file) {
    	printf("Unable to open output file %s.\n", outfilename);
		return(EXIT_FAILURE);
	}
    // initialize output array
    memset(out_array, 0, OUT_LEN);
    // translate data to c or asm text
    if (asmflag)
    	sbi2asm(&sbi_struct, out_array);
    else
    	sbi2c(&sbi_struct, out_array);
    // write output array to output file
    if (fwrite(&out_array, strlen(out_array), 1, out_file) != 1) {
		printf("Error writing output file %s.\n", outfilename);
        fclose(out_file);
        return(EXIT_FAILURE);
    }

    // close output file
    fclose(out_file);
    // done
    return(EXIT_SUCCESS);

}

void help()
{
	puts("\nThis program converts SoundBlaster(TM) Instrument Files");
	puts("  to ascii text suitable for inclusion in source code.\n");
    puts("      SBICONV filename.sbi [outfile.ext] [/a] [/c]");
    puts("                /a = asm source.");
    puts("                /c = C/C++ source.\n");
    puts("Steve Robison 1994");
	return;
}

void sbi2asm(struct SBI_FILE_STRUCT *sbi, char out[])
{
    char temp[4];
	strcpy(out, "\n; *** sbi instrument asm source created by SBICONV ***\n\n");
	strcat(out, "inst_name        DB \'");
    strncat(out, sbi->inst_name, 8);
    strcat(out, "\'\n");
    strcat(out, "inst_comments    DB \'");
    strncat(out, sbi->comments, 24);
    strcat(out, "\'\n");
    strcat(out, "cmf_inst         DB 0");
	itoa(sbi->mod_prop, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; modulator sound properties\n");
    strcat(out, "                 DB 0");
	itoa(sbi->car_prop, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; carrier sound properties\n");
    strcat(out, "                 DB 0");
	itoa(sbi->mod_vol, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; modulator volume\n");
    strcat(out, "                 DB 0");
	itoa(sbi->car_vol, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; carrier volume\n");
    strcat(out, "                 DB 0");
	itoa(sbi->mod_a_d, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; modulator attack/decay\n");
    strcat(out, "                 DB 0");
	itoa(sbi->car_a_d, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; carrier attack/decay\n");
    strcat(out, "                 DB 0");
	itoa(sbi->mod_s_r, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; modulator sustain/release\n");
    strcat(out, "                 DB 0");
	itoa(sbi->car_s_r, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; carrier sustain/release\n");
    strcat(out, "                 DB 0");
	itoa(sbi->car_wav, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; modulator waveform\n");
    strcat(out, "                 DB 0");
	itoa(sbi->car_wav, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; carrier waveform\n");
    strcat(out, "                 DB 0");
	itoa(sbi->synth_mode, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, "h                   ; synthesis mode\n");
    strcat(out, "                 DB dup (0)                ; reserved\n");
    return;
}

void sbi2c(struct SBI_FILE_STRUCT *sbi, char out[])
{
    char temp[4];
	strcpy(out, "\n/**** SBI instrument C/C++ source created by SBICONV ****/\n");
    strcat(out, "/*      Name = \'");
    strncat(out, sbi->inst_name, 8);
    strcat(out, "\' */\n");
    strcat(out, "/*      Comments = \'");
    strncat(out, sbi->comments, 24);
    strcat(out, "\' */\n");
	strcat(out, "FM_Instrument ");
    strncat(out, sbi->inst_name, 8);
    strcat(out, " = {\n");
    strcat(out, "    { 0x");
	itoa(sbi->mod_prop, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, ",                      /* modulator sound properties */\n");
    strcat(out, "      0x");
	itoa(sbi->car_prop, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, " },                    /* carrier sound properties */\n");
    strcat(out, "    { 0x");
	itoa(sbi->mod_vol, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, ",                      /* modulator volume */\n");
    strcat(out, "      0x");
	itoa(sbi->car_vol, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, " },                    /* carrier volume */\n");
    strcat(out, "    { 0x");
	itoa(sbi->mod_a_d, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, ",                      /* modulator attack/decay */\n");
    strcat(out, "      0x");
	itoa(sbi->car_a_d, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, " },                    /* carrier attack/decay */\n");
    strcat(out, "    { 0x");
	itoa(sbi->mod_s_r, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, ",                      /* modulator sustain/release */\n");
    strcat(out, "      0x");
	itoa(sbi->car_s_r, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, " },                    /* carrier sustain/release */\n");
    strcat(out, "   {  0x");
	itoa(sbi->car_wav, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, ",                      /* modulator waveform */\n");
    strcat(out, "      0x");
	itoa(sbi->car_wav, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, " },                    /* carrier waveform */\n");
    strcat(out, "    { 0x");
	itoa(sbi->synth_mode, temp, 16);
    strupr(temp);
    if (strlen(temp) == 1)
    	strcat(out, "0");
    strcat(out, temp);
    strcat(out, " },                    /* synthesis mode */\n");
    strcat(out, "    { 0x00,                      /* reserved*/\n");
    strcat(out, "      0x00,\n");
    strcat(out, "      0x00,\n");
    strcat(out, "      0x00,\n");
    strcat(out, "      0x00 }\n");
    strcat(out, "};\n\n");
	return;
}
