// ---------------------------------------------------------------------------
// Module:		FREEUP.C
//
// Description:	FreeUp File Manager is a MS Windows 3.1 file manager with
//				the standard copy, move, delete, rename, view, etc. file
//				and directory functions.  Along with those are options to
//				display directory size information along side the directory
//				tree display, the ability to .ZIP selected files or directories
//				with several options, execute, associate and many, many
//				more useful functions.
//
// The "Real"
// Description:	FreeUp was originally designed to become a utility to help
//				to "Free Up" ever-shrinking disk drive space in which I
//				found myself shelling to DOS and running various DOS-based
//				file managers and utilities to do the job.  My second
//				priority was to build a substantial application to allow me
//				to learn as many Windows programming techniques as was
//				possible in a program like this.
//				I started this program back in january, it's now nearly july
//				and I guess I'm just becoming tired of tooling around with
//				it.  So here it is.  You can use parts of this program for
//				your own use, but not all of it (see LEGAL.TXT). If you
//				decide to use some of the code in your programs (modified or
//				not), please mention my name or FreeUp's name in the documentation. 
//
// Known bugs:	Upon exiting the program, FreeUp doesn't "free up" a certain
//				GDI resource which I still can't find!  Secondly, the
//				Toolbar drive button creation function in FU_TBAR.C can
//				correctly identify RAM disks & CD-ROM drives but does not
//				properly display them.  
//					
// Last modified: 06/28/94
//
// Please contact me if you have any questions or comments:
//
// Compuserve ID:	71042,151
// Phone Number:	408-629-7460
//
// (C) Copyright 1994 Thomas R. Grubbe, All Rights Reserved
//
// ---------------------------------------------------------------------------

#define STRICT

#include <windowsx.h>
#include <shellapi.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dir.h>
#include <dos.h>
#include <ctl3d.h>
#include <math.h>
#include "freeup.h"
#include "fu_draw.h"
#include "fu_util.h"
#include "fu_file.h"
#include "fu_cmd.h"
#include "fu_tbar.h"
#include "fu_zip.h"

TreeStruct _huge Tree[MAXPATHS];		// See FREEUP.H

// ========================= Function Declarations ======================
LRESULT CALLBACK _export WndProc(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export ScanTreeDlg(HWND, UINT, WPARAM, LPARAM);
BOOL CALLBACK _export AboutDlgProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK _export CenterWndProc(HWND, UINT, WPARAM, LPARAM);
LRESULT CALLBACK _export StatusBarWndProc(HWND hwnd, UINT msg,
									   WPARAM wParam, LPARAM lParam);
extern LRESULT CALLBACK _export ToolBarWndProc(HWND hwnd, UINT msg,
									   WPARAM wParam, LPARAM lParam);

int Compare(TreeStruct _huge *str1, TreeStruct _huge *str2);
void SortDir(TreeStruct _huge *Sptr, int Num);
void ScanTree(void);
void Initialize(TreeStruct _huge *Tptr);
unsigned long GetDirSize(char *Path, struct find_t f);
void AddListBoxItems(BOOL TopIdx);
int GetParentDir(int Idx);
int GetSubCnt(int Idx);
BOOL ReadDataFile(void);
void WriteDataFile(void);
int GetDirectoryCount(void);
void SystemChangeColor(void);
void DrawDirTotals(void);
void DrawFileTotals(void);


void DirOnDrawItem(HWND hwnd, DRAWITEMSTRUCT FAR* lpDrawItem);
BOOL DirOnCommand(HWND hwnd, UINT id, HWND hwndCtl, UINT code);
void DirActionItem(HWND hwnd, DWORD dwData, WORD wItemNum);

extern UINT CALLBACK _export ChooseFontHook(HWND, UINT, WPARAM, LPARAM);
extern void GetFont(HWND hWnd);
extern BOOL IsCDRomDrive(int);
void InitFont(void);
void FreeUpSetFont(HWND hwnd);

#define ROWS 8		// Number of rows of folder bitmaps in FOLDERS.BMP
#define COLS 6		// Number of cols of folder bitmaps in FOLDERS.BMP

TREEDRAWSTRUCT TreeDrawStruct;		// See FU_DRAW.H
FILESDRAWSTRUCT FilesDrawStruct;	// See FU_FILE.H 

// ========================== Global variables ==========================
HINSTANCE ghInstance;
HWND ghwndMain, ghwndDirBox, ghwndFileBox;
HWND ghwndStatusBar, ghwndSeparator, ghwndToolBar;
HWND hTotalFiles, hTotalDirs, hCurrentDir;

char szAppName[] = "FreeUp";
WNDCLASS CenterClass;
WNDCLASS StatusBarClass;
WNDCLASS ToolBarClass;
HCURSOR hResizeCur;

int Count, Deep, Index, Cnt;
int TotalFiles;
int TotalTaggedFiles;
unsigned long TotalTaggedBytes;
int StatBarHeight = 30;
int ToolBarHeight = 65;
char CurPath[128];
// unsigned long DiskSize, FreeSpace, BytesUsed, BytesRead;
char GlobalSearchText[80];
BOOL Quit = FALSE;
int SortOrder = DECENDING;
int SortBy = SORTSIZE;
BOOL TreeModified = FALSE;
int CurrentDrive;

char szFontStyle[LF_FACESIZE];		// Regular, Italic, Bold, etc...
LOGFONT logFont = {0,0,0,0, FW_NORMAL, 0,0,0, ANSI_CHARSET,
				   OUT_DEFAULT_PRECIS, CLIP_DEFAULT_PRECIS, PROOF_QUALITY,
				   VARIABLE_PITCH | FF_DONTCARE, "System"};
HFONT ghFont, ghFontOld;

extern unsigned _stklen = 48000U;	// Increase stack size for global vars 

// FREEUP.INI Profile strings/ints (should be self-explanatory)
char StartUpDir[128];
char DefaultEditor[128];
char DefaultViewer[128];
int Center = 200;
BOOL LowerCase = FALSE;
BOOL DirSizeInfo = TRUE;
BOOL UseCustomPathNames = TRUE;
int NumDrives;
RECT WindowRect, DefRect;
RECT DirTotalsRect, FileTotalsRect;
RECT FreeSpaceRect, CurrentDirRect;
int WindowState = SW_SHOW;


#pragma argsused
int PASCAL WinMain (HINSTANCE hInstance, HINSTANCE hPrevInstance,
					LPSTR lpszCmd, int nCmdShow)
{
	MSG			msg;
	WNDCLASS	wc;
	HANDLE		hAccel;
	RECT		r;
	char		Str[80];

	ghInstance = hInstance;

	Ctl3dRegister(hInstance);
	Ctl3dAutoSubclass(hInstance);

    // Initialize TreeDrawStruct structure
	if (!TreeDraw_DrawInit(hInstance,
					  IDC_FOLDERS,
					  ROWS,
					  COLS,
                      TRUE,
					  &TreeDrawStruct,
					  TRUE))
		return (-1);
    // Initialize FileDrawStruct structure
	if (!FileDraw_DrawInit(hInstance,
					  IDC_FOLDERS,
					  ROWS,
					  COLS,
					  TRUE,
					  &FilesDrawStruct))
		return (-1);

	if (!hPrevInstance)
	{
		wc.style         = CS_HREDRAW | CS_VREDRAW ;
		wc.lpfnWndProc   = (WNDPROC)WndProc ;
		wc.cbClsExtra    = 0 ;
		wc.cbWndExtra    = 0 ;
		wc.hInstance     = hInstance ;
		wc.hIcon         = LoadIcon(hInstance, MAKEINTRESOURCE(IDC_FREEUPICON));
		wc.hCursor       = LoadCursor (NULL, IDC_ARROW) ;
		wc.hbrBackground = (HBRUSH)(COLOR_WINDOW + 1);
		wc.lpszMenuName  = "MAINMENU";
		wc.lpszClassName = szAppName ;

		if (!RegisterClass (&wc))
			return (-1);

		CenterClass.style         = CS_VREDRAW;
		CenterClass.lpfnWndProc   = (WNDPROC)CenterWndProc;
		CenterClass.cbClsExtra    = 0;
		CenterClass.cbWndExtra    = 0;
		CenterClass.hInstance     = hInstance;
		CenterClass.hIcon         = NULL; 
		CenterClass.hCursor       = (HCURSOR)LoadCursor (NULL, "IDC_SEPARATOR");
		CenterClass.hbrBackground = (HBRUSH)(GetStockObject(LTGRAY_BRUSH));
		CenterClass.lpszMenuName  = NULL;
		CenterClass.lpszClassName = "SeparatorClass";

		if (!RegisterClass (&CenterClass))
			return (-1);

		StatusBarClass.lpfnWndProc   = (WNDPROC)StatusBarWndProc ;
		StatusBarClass.hInstance     = hInstance ;
		StatusBarClass.hCursor       = (HCURSOR)LoadCursor (NULL, IDC_ARROW) ;
		StatusBarClass.hbrBackground = (HBRUSH)(GetStockObject(LTGRAY_BRUSH));
		StatusBarClass.lpszClassName = "StatusBarClass";

		if (!RegisterClass (&StatusBarClass))
			return (-1);

		ToolBarClass.lpfnWndProc   = (WNDPROC)ToolBarWndProc ;
		ToolBarClass.hInstance     = hInstance ;
		ToolBarClass.hCursor       = (HCURSOR)LoadCursor (NULL, IDC_ARROW) ;
		ToolBarClass.hbrBackground = (HBRUSH)(GetStockObject(LTGRAY_BRUSH));
		ToolBarClass.lpszClassName = "ToolBarClass";

		if (!RegisterClass (&ToolBarClass))
			return (-1);
	}

    // Get the number of disk drives on the system
	NumDrives = setdisk(getdisk());
    // Read FREEUP.INI info
	InitFreeUp();

	ghwndMain = CreateWindow (szAppName,
								szAppName,
								WS_OVERLAPPEDWINDOW,
								CW_USEDEFAULT,
								CW_USEDEFAULT,
								CW_USEDEFAULT,
								CW_USEDEFAULT,
								NULL,
								NULL,
								hInstance,
								NULL);


	// Get a default window RECT in case FREEUP.INI does not
    // exist or does not contain the previous window size.
	GetWindowRect(ghwndMain, &DefRect);
	MoveWindow(ghwndMain, WindowRect.left, WindowRect.top,
				WindowRect.right, WindowRect.bottom, TRUE);

    // Empty everything from Tree[] structure
	Initialize((TreeStruct _huge *)Tree);

	// If the data file for the current drive doen't exist,
	// scan the current drive. 
	if (!ReadDataFile())
		SendMessage(ghwndMain, WM_COMMAND, IDM_READDIRS, 0L);
	else
		AddListBoxItems(TRUE);

    // Create the initial font for the Dir & File listboxes
    InitFont();
	ShowWindow (ghwndMain, WindowState);
	UpdateWindow (ghwndMain);

    hAccel = LoadAccelerators(hInstance, "FreeUpAccelerators");

	while (GetMessage ((LPMSG)&msg, NULL, 0, 0))
	{
		if (!TranslateAccelerator(ghwndMain, hAccel, &msg))
        {
			TranslateMessage ((LPMSG)&msg);
			DispatchMessage ((LPMSG)&msg);
        }
	}
	DeleteObject(ghFont);
	Ctl3dUnregister(hInstance);
	UnregisterClass("SeparatorClass", hInstance);
	UnregisterClass("StatusBarClass", hInstance);
	UnregisterClass("ToolBarClass",   hInstance);
	return msg.wParam ;
}


LRESULT CALLBACK _export WndProc (HWND hwnd, UINT message,
								  WPARAM wParam, LPARAM lParam)
{
	static HANDLE hMem;
	static HMENU hMenu;
	int CenterWidth;
	LPDRAWITEMSTRUCT DrawStruct;
    LPMEASUREITEMSTRUCT MeasureStruct;
    int i;

	switch (message)
	{
		case WM_CREATE:
		{
			RECT rcMain;

            // Create the directory listbox
			ghwndDirBox = CreateWindow("listbox", NULL,
							WS_CHILD | WS_VISIBLE | WS_VSCROLL |
							LBS_NOINTEGRALHEIGHT | LBS_NOTIFY |
                            LBS_OWNERDRAWFIXED | LBS_WANTKEYBOARDINPUT,
							0, 0, 0, 0, hwnd, (HMENU)IDC_DIRBOX,
							ghInstance, NULL);

            // Create the files listbox
			ghwndFileBox = CreateWindow("listbox", NULL,
							WS_CHILD | WS_VISIBLE | WS_VSCROLL | 
							LBS_NOINTEGRALHEIGHT | LBS_NOTIFY |
							LBS_MULTIPLESEL |
							LBS_OWNERDRAWFIXED | LBS_WANTKEYBOARDINPUT,
							0, 0, 0, 0, hwnd, (HMENU)IDC_FILEBOX,
							ghInstance, NULL);

			CenterWidth = GetSystemMetrics(SM_CXFRAME);
			GetClientRect(hwnd, &rcMain);
            // Create the little separator between the above
			ghwndSeparator = CreateWindow("SeparatorClass", NULL,
							WS_CHILD | WS_VISIBLE | WS_BORDER | WS_CLIPSIBLINGS,
							Center, -400, CenterWidth, 2000, hwnd, (HMENU)IDC_SEPARATOR,
							ghInstance, NULL);

            // Create the statusbar window
			ghwndStatusBar = CreateWindow("StatusBarClass", NULL,
							WS_CHILD | WS_VISIBLE | WS_BORDER,
							rcMain.left, rcMain.bottom - StatBarHeight, rcMain.right, rcMain.bottom, hwnd,
							(HMENU)IDC_STATUSBAR, ghInstance, NULL);

            // Create the toolbar window
			ghwndToolBar = CreateWindow("ToolBarClass", NULL,
							WS_CHILD | WS_VISIBLE | WS_BORDER,
							rcMain.left, rcMain.top, rcMain.right, rcMain.top + ToolBarHeight, hwnd,
							(HMENU)IDC_TOOLBAR, ghInstance, NULL);

			DragAcceptFiles(ghwndFileBox, TRUE);

			// Check the SortBy menu option to reflect
            // what was in FREEUP.INI
			hMenu = GetSubMenu(GetMenu(hwnd), 1);
			CheckMenuItem(hMenu, SortBy, MF_CHECKED);
			CheckMenuItem(hMenu, (SortOrder == ASCENDING) ?
						  IDM_ASCENDING : IDM_DECENDING, MF_CHECKED);

            // Gray the Use custom path names menu option if FALSE
            if (!UseCustomPathNames)
				EnableMenuItem(hMenu, (WORD)IDM_CUSTOMNAME, MF_BYCOMMAND | MF_GRAYED);

            CurrentDrive = getdisk();
/*
			// Allocate global memory for directory structure
//			GlobalCompact(-1L);
			hMem = GlobalAlloc(GMEM_MOVEABLE | GMEM_ZEROINIT, MAXALLOC);
			if (hMem == NULL)	{
				MessageBox(hwnd, "Could not allocate enough memory for directory tree.",
							"Error", MB_OK);
				SendMessage(hwnd, WM_CLOSE, 0, 0L);
			}
			Tree = (TreeStruct _huge *)GlobalLock(hMem);
			if (Tree == NULL)	{
            	GlobalFree(hMem);
				MessageBox(hwnd, "Could not allocate enough memory for directory tree.",
							"Error", MB_OK);
				SendMessage(hwnd, WM_CLOSE, 0, 0L);
			}
*/
			return 0L;
		}
		case WM_SIZE:
		{
			RECT rcMain, rcDir, rcFile, rcSeparator, rcSbar, rcTbar;

			GetClientRect(hwnd, &rcMain);
			GetClientRect(ghwndDirBox, &rcDir);
			GetClientRect(ghwndFileBox, &rcFile);
			GetClientRect(ghwndSeparator, &rcSeparator);
			GetClientRect(ghwndStatusBar, &rcSbar);
			GetClientRect(ghwndToolBar, &rcTbar);

            rcTbar.left   = rcMain.left;
			rcTbar.top    = rcMain.top;
			rcTbar.right  = rcMain.right;
            rcTbar.bottom = ToolBarHeight;

            rcDir.left   = 0;
			rcDir.top    = rcMain.top + ToolBarHeight;
			rcDir.right  = Center;
			rcDir.bottom = rcMain.bottom - StatBarHeight - ToolBarHeight;

			CenterWidth = GetSystemMetrics(SM_CXFRAME);
			rcSeparator.left   = rcDir.right;
			rcSeparator.top    = rcMain.top + ToolBarHeight;
			rcSeparator.right  = CenterWidth;
			rcSeparator.bottom = rcMain.bottom - StatBarHeight - ToolBarHeight;

			rcFile.top    = rcMain.top + ToolBarHeight;
			rcFile.left   = rcDir.right + rcSeparator.right;
			rcFile.right  = rcMain.right - rcDir.right - rcSeparator.right;
			rcFile.bottom = rcMain.bottom - StatBarHeight - ToolBarHeight;

			rcSbar.top    = rcMain.bottom - StatBarHeight;
			rcSbar.left   = rcMain.left;
			rcSbar.right  = rcMain.right;
			rcSbar.bottom = StatBarHeight;

			MoveWindow(ghwndDirBox, 0, rcDir.top, rcDir.right, rcDir.bottom, TRUE);
			SetWindowPos(ghwndSeparator, NULL, rcSeparator.left, rcSeparator.top,
            			 rcSeparator.right, rcSeparator.bottom, SWP_NOZORDER | SWP_SHOWWINDOW);
			MoveWindow(ghwndFileBox, rcFile.left, rcFile.top, rcFile.right, rcFile.bottom, TRUE);
			MoveWindow(ghwndStatusBar, rcSbar.left, rcSbar.top,
						 rcSbar.right, rcSbar.bottom, TRUE);
			MoveWindow(ghwndToolBar, rcTbar.left, rcTbar.top,
						 rcTbar.right, rcTbar.bottom, TRUE);

			InvalidateRect(hwnd, &rcMain, TRUE);
			return 0L;
		}
		case WM_SYSCOMMAND:
			// If WindowState was previously SW_MAXIMIZED or SW_MINIMIZED,
			// DeInitFreeUP() doesn't save the main window dimensions. So
			// WindowRect is always the correct restored size. 
			if (wParam == SC_RESTORE)
				MoveWindow(ghwndMain, WindowRect.left, WindowRect.top,
							WindowRect.right, WindowRect.bottom, TRUE);
			break;
		case WM_DROPFILES:
			MessageBeep(MB_OK);
			DragFinish((HDROP)wParam);
			return 0L;
		case WM_PAINT:
		{
			HDC hdc;
			PAINTSTRUCT ps;

			hdc = BeginPaint(hwnd, &ps);
			EndPaint(hwnd, &ps);
			return 0L;
		}
		case WM_COMMAND:
			switch (wParam)
			{
				case IDM_READDIRS:
                {
					FARPROC ScanProc;

					TreeDraw_DrawCloseAll(&TreeDrawStruct);

                    ScanProc = MakeProcInstance((FARPROC)ScanTreeDlg, ghInstance);
					DialogBox(ghInstance, "ScanDirectories", hwnd, (DLGPROC)ScanProc);
					FreeProcInstance(ScanProc);

					SendMessage(ghwndDirBox, WM_SETREDRAW, FALSE, 0L);
					SendMessage(ghwndFileBox, WM_SETREDRAW, FALSE, 0L);
					LoadWaitCursor();
					SortDir((TreeStruct _huge *)Tree, Count);
					SendMessage(ghwndDirBox, LB_RESETCONTENT, 0, 0L);

                    if (LowerCase)
						wsprintf(Tree[0].Path, "%c:\\", getdisk() + 'a');
					else
						wsprintf(Tree[0].Path, "%c:\\", getdisk() + 'A');
					lstrcpy(Tree[0].Name, Tree[0].Path);
					Tree[0].LastChild = FALSE;
					Tree[0].Level = 0;
	
					AddListBoxItems(TRUE);

					WriteDataFile();
					LoadArrowCursor();
					return 0L;
				}
				case IDM_ASCENDING:
					CheckMenuItem(hMenu, IDM_DECENDING, MF_UNCHECKED);
					SortOrder = ASCENDING;
					CheckMenuItem(hMenu, IDM_ASCENDING, MF_CHECKED);
					SortFiles();
					return 0L;
				case IDM_DECENDING:
					CheckMenuItem(hMenu, IDM_ASCENDING, MF_UNCHECKED);
					SortOrder = DECENDING;
					CheckMenuItem(hMenu, IDM_DECENDING, MF_CHECKED);
					SortFiles();
					return 0L;
				case IDM_BYNAME:
				case IDM_BYEXTENSION:
				case IDM_BYSIZE:
				case IDM_BYDATE:
					// Size & Date are always displayed in decending
                    // order, else sort in ascending order
                	hMenu = GetSubMenu(GetMenu(hwnd), 1);
					if (wParam == IDM_BYSIZE || wParam == IDM_BYDATE)
						SendMessage(hwnd, WM_COMMAND, IDM_DECENDING, 0L);
					else if (wParam != IDM_BYSIZE && wParam != IDM_BYDATE)
						SendMessage(hwnd, WM_COMMAND, IDM_ASCENDING, 0L);
                    // Un-check the old order, check the new
                	CheckMenuItem(hMenu, SortBy, MF_UNCHECKED);
					SortBy = wParam;
					CheckMenuItem(hMenu, SortBy, MF_CHECKED);
					SortFiles();
					return 0L;
				case IDM_TAGALL:
					SendMessage(ghwndFileBox, LB_SELITEMRANGE, TRUE, MAKELONG(0, FileCount));
					TotalTaggedFiles = FileCount;
					TotalTaggedBytes = DirSize;
					DrawFileTotals();
                    return 0L;
				case IDM_UNTAGALL:
					SendMessage(ghwndFileBox, LB_SELITEMRANGE, FALSE, MAKELONG(0, FileCount));
					TotalTaggedFiles = 0;
					TotalTaggedBytes = 0L;
					DrawFileTotals();
					return 0L;
				case IDM_INVERTTAGS:
					InvertTagged();
                    return 0L;
				case IDM_DRIVE_A:
				case IDM_DRIVE_B:
				case IDM_DRIVE_C:
				case IDM_DRIVE_D:
				case IDM_DRIVE_E:
				case IDM_DRIVE_F:
				case IDM_DRIVE_G:
				case IDM_DRIVE_H:
				case IDM_DRIVE_I:
				case IDM_DRIVE_J:
				case IDM_DRIVE_K:
				case IDM_DRIVE_L:
				case IDM_DRIVE_M:
				case IDM_DRIVE_N:
				case IDM_DRIVE_O:
				case IDM_DRIVE_P:
				case IDM_DRIVE_Q:
				case IDM_DRIVE_R:
				case IDM_DRIVE_S:
				case IDM_DRIVE_T:
				case IDM_DRIVE_U:
				case IDM_DRIVE_V:
				case IDM_DRIVE_W:
				case IDM_DRIVE_X:
				case IDM_DRIVE_Y:
				case IDM_DRIVE_Z:
				{
					char NewPath[128];

                    // wParam = 0-25 (drives A-Z)
					CurrentDrive = wParam;

					wsprintf(NewPath, "%c:\\", CurrentDrive + 'A');
					if (LogPath(NewPath) == -1)	{
						MessageBox(hwnd, "Error logging to new path.", "Error", MB_OK);
						return 0L;
					}

                    // Close any open folders
					TreeDraw_DrawCloseAll(&TreeDrawStruct);
					SendMessage(ghwndDirBox, WM_SETREDRAW, FALSE, 0L);
					SendMessage(ghwndFileBox, WM_SETREDRAW, FALSE, 0L);

					// ReScan if were logging to a floppy drive or
					// data file for CurrentDrive is not found.
					// Else, just read the data file. 
					if (CurrentDrive < 2 || !ReadDataFile())
						SendMessage(hwnd, WM_COMMAND, IDM_READDIRS, 0L);
					else
                    	AddListBoxItems(TRUE);
					return 0L;
				}
				case IDM_VIEW:
                case IDM_EDIT:
					ViewEdit(wParam);
					return 0L;
				case IDM_MOVE:
				case IDM_COPY:
					CopyMove(wParam);
					return 0L;
				case IDM_DELETE:
					Delete();
					return 0L;
				case IDM_MKDIR:
					MkDir();
					return 0L;
				case IDM_ZIPFILES:
					ZipFiles();
                    return 0L;
				case IDM_ATTRIBUTES:
					Attributes();
					return 0L;
				case IDM_RENAME:
					ReName();
					return 0L;
				case IDM_FIND:
					FindText();
					return 0L;
				case IDM_CUSTOMNAME:
					CustomName();
					return 0L;
				case IDM_CHOOSEFONTS:
					GetFont(hwnd);
					FreeUpSetFont(hwnd);
					return 0L;
				case IDM_DEFOPTS:
					DefOptions();
					if (!UseCustomPathNames)
						EnableMenuItem(hMenu, (WORD)IDM_CUSTOMNAME, MF_BYCOMMAND | MF_GRAYED);
					else if (UseCustomPathNames)
						EnableMenuItem(hMenu, (WORD)IDM_CUSTOMNAME, MF_BYCOMMAND | MF_ENABLED);
					return 0L;
				case IDM_CNTSRCLINES:
					SrcLineCount();
					return 0L;
				case IDM_LOADOPTS:
					InitFreeUp();
					return 0L;
				case IDM_SAVEOPTS:
					DeInitFreeUp();
                    WriteDataFile();
					return 0L;
				case IDM_ABOUT:
				{
					FARPROC AboutProc =
						MakeProcInstance((FARPROC)AboutDlgProc, ghInstance);
					DialogBox(ghInstance, "AboutDlg", hwnd, (DLGPROC)AboutProc);
					FreeProcInstance(AboutProc);
                    return 0L;
				}
				case IDM_ASSOCIATE:
				case -1:
				case -2:
				case -3:
				case -4:
				case -5:
					FreeUpMsgBox(MB_OK, szAppName, "function not yet added.");
					return 0L;
				case IDM_EXIT:
					SendMessage(hwnd, WM_CLOSE, 0, 0L);
					return 0L;
				case IDC_FILEBOX:
					FileOnCommand(hwnd, (int)wParam, (HWND)LOWORD(lParam), (UINT)HIWORD(lParam));
					return 0L;
				case IDC_DIRBOX: 
					DirOnCommand(hwnd, (int)wParam, (HWND)LOWORD(lParam), (UINT)HIWORD(lParam));
					EnableButtons(FALSE);
					return 0L;
			}
		case WM_SETFONT:
		{
			TreeDraw_DrawSetTextHeight(ghwndDirBox, (HFONT)wParam, &TreeDrawStruct);
			SetFileBoxTextHeight(ghwndFileBox, (HFONT)wParam, &FilesDrawStruct);
			SendMessage(ghwndDirBox, WM_SETFONT, (HFONT)wParam, 0L);
			SendMessage(ghwndFileBox, WM_SETFONT, (HFONT)wParam, 0L);
			return 0L;
        }
		case WM_DRAWITEM:
				DrawStruct = (LPDRAWITEMSTRUCT)lParam;

                if (DrawStruct->CtlID == (UINT)IDC_DIRBOX)
					DirOnDrawItem(ghwndDirBox, (DRAWITEMSTRUCT FAR*)(lParam));
				if (DrawStruct->CtlID == (UINT)IDC_FILEBOX)
					FileOnDrawItem(ghwndFileBox, (DRAWITEMSTRUCT FAR*)(lParam),
								   &FilesDrawStruct);
			return 0L;
		case WM_MEASUREITEM:
        	InitFont();
			MeasureStruct = (LPMEASUREITEMSTRUCT)lParam;

			if (MeasureStruct->CtlID == (UINT)IDC_DIRBOX)
				TreeDraw_OnMeasureItem(ghwndDirBox, (MEASUREITEMSTRUCT FAR*)(lParam), &TreeDrawStruct);
			if (MeasureStruct->CtlID == (UINT)IDC_FILEBOX)
				FileDraw_OnMeasureItem(ghwndFileBox, (MEASUREITEMSTRUCT FAR*)(lParam), &FilesDrawStruct);
			return 0L;
		case WM_SETFOCUS:
//			SetFocus((GetFocus() == ghwndDirBox) ? ghwndFileBox : ghwndDirBox);
            return 0L;
		case WM_SYSCOLORCHANGE:
			SystemChangeColor();
            return 0L;
		case WM_CLOSE:
		case WM_DESTROY:
			DeInitFreeUp();
			WriteDataFile();
			DestroyWindow(ghwndDirBox);
			DestroyWindow(ghwndFileBox);
			TreeDraw_DrawTerm(&TreeDrawStruct);
			FileDraw_DrawTerm(&FilesDrawStruct);
			if (hMem)	{
				GlobalUnlock(hMem);
				GlobalFree(hMem);
			}
			PostQuitMessage (0);
			return 0L;
	}
	return DefWindowProc (hwnd, message, wParam, lParam) ;
}


void AddListBoxItems(BOOL TopIdx)
{
	register int i, j;
	int cnt, icnt, Parent, Idx, Top;
    char Tmp[80];

	FlushMsgLoop();
    LoadWaitCursor();
	SendMessage(ghwndDirBox, WM_SETREDRAW, FALSE, 0L);
	Idx = GetDirIndex();
	Top = SendMessage(ghwndDirBox, LB_GETTOPINDEX, 0, 0L);

	SendMessage(ghwndDirBox, LB_RESETCONTENT, 0, 0L);

	for (i=0; i<=Count; i++)	{
		PathCh(Tree[i].Path, lstrlen(Tree[i].Path), '/', '\\');
		Tree[i].Parent = GetParentDir(i);
		Tree[i].SubCnt = GetSubCnt(i);
        if (LowerCase)	{
//			strlwr(Tree[i].Path);
			strlwr(Tree[i].Name);
		}
		else	{
//			strupr(Tree[i].Path);
			strupr(Tree[i].Name);
        }
		SendMessage(ghwndDirBox, LB_ADDSTRING, 0, (long)i);
	}
    // Find all last children in the directory tree
	for (i=0; i<=Count; i++)	{
		cnt = Tree[i].SubCnt;
		if (Tree[i+cnt].IsParent && Tree[i+cnt].ChildNum == cnt)
			Tree[i+cnt].LastChild = TRUE;
	}
    // Make sure the last directory is a last child
    if (Count > 1)	{
		Parent = Tree[Count].Parent;
	    if (Tree[Parent].Level > 0)
			Tree[Parent].LastChild = TRUE;
	}
	Tree[Count].LastChild = TRUE;

	SendMessage(ghwndDirBox, WM_SETREDRAW, TRUE, 0L);

    // TopIdx means set listbox's index to the top
    if (TopIdx)	{
		SendMessage(ghwndDirBox, LB_SETCURSEL, 0, 0L);
		SendMessage(ghwndDirBox, LB_SETTOPINDEX, 0, 0L);
		GetFiles("\\", "*.*");
	}

	LoadArrowCursor();
	UpdateStatusBars();
}

int GetParentDir(int Idx)
{
	register int i;

    // Root dir has no parent
	if (Idx == 0)
		return -1;
	// Walk the tree backward from Idx until level is 1 less
    // than Tree[Idx].Level
	for (i=Idx; i>=0; i--)	{
		if (Tree[i].Level == Tree[Idx].Level - 1)	{
        	Tree[i].IsParent = TRUE;
			return (i);
		}
	}
	return (0);
} 

int GetSubCnt(int Idx)
{
	register int i, cnt=0, icnt=0;

	// If the level below the current dir is greater than
	// the current dir, then it has subdirectories. So continue.
	if (Tree[Idx+1].Level > Tree[Idx].Level)	{
		for (i=Idx+1; i<=Count; i++)	{
        	// Increase subdirectory count & assign it to child
			if (Tree[i].Level == Tree[Idx].Level + 1)	{
				cnt++;
				Tree[i].ChildNum = cnt;
			}
			// Children may also be parents, so count those
            // using a different variable.
			if (Tree[i].Level >= Tree[Idx].Level + 1)
				icnt++;
			// If the level is less than that of the original
            // parent, then were done.
			if (Tree[i].Level <= Tree[Idx].Level)	{
				// If there were sub-parents, include those in the
				// sub tree and add them to the sub count, else just
                // count the children.
				if (icnt > cnt)
					Tree[Idx+icnt].LastChild = TRUE;
				else
					Tree[Idx+cnt].LastChild = TRUE;
				return (cnt);
			}
		}
		return (cnt);
	}
    return (0);
}


#pragma argsused
void DirOnDrawItem(HWND hwnd, DRAWITEMSTRUCT FAR* lpDrawItem)
{
	char  szName[80];
	int   nLevel;
	int   nTempLevel;
	int   nRow;
	int   nColumn;
	int   nIndex;
	DWORD dwConnectLevel = 0L;
	DWORD dwMask;
	int SubCnt;
    char Tmp[80];

    nIndex = lpDrawItem->itemID;

	if (nIndex != 0 && UseCustomPathNames && Tree[nIndex].CustomName[0] != '\0')
		lstrcpy(szName, Tree[nIndex].CustomName);
	else
		lstrcpy(szName, Tree[nIndex].Name);

	lstrcpy(Tmp, ul2str(Tree[nIndex].DirSize));

	// If the current drawn item is opened, select the correct bitmap
    // to display.
	if ( TreeDraw_IsOpened(&TreeDrawStruct, nIndex) && nIndex != 0)
		nColumn = 1;
	else
		nColumn = 0;

    // Different color for parent dirs.
	if (Tree[nIndex].IsParent && nIndex != 0)
		nRow = 1;
    else
		nRow = 0;

	nLevel = Tree[nIndex].Level;

	// Figure out which connecting lines are needed.
	// If this item is the last kid or one it parents
	// is a last kid, then it does not need a connector at that level
	if ( nLevel == 0 )	{
		// Level 0 items never have connectors.
		dwConnectLevel = 0L;
		if (CurrentDrive < 2)
			nRow = CurrentDrive;		// Use floppy drive icons
		else
			nRow = 2;					// Use fixed drive icon

		if (IsCDRomDrive(CurrentDrive))
			nRow = 3;
		nColumn = 2;
	}
	else	{
		// Start at current level - 1
		nTempLevel = nLevel-1;
		// First bit to set ( or not );
		dwMask = (DWORD) pow(2,nLevel-1);
     
		// While were are not at level 0
		while ( nTempLevel >= 0 )	{
			// If not last child at this level then no connection
            // else draw a connection.
			if ( !Tree[nIndex].LastChild) 
				dwConnectLevel |= dwMask;
			else
				dwConnectLevel &= ~dwMask;
			// Move mask bit over 
			dwMask /= 2;

			// Move to neext parent level
			nIndex = Tree[nIndex].Parent;
			// Move up level
			nTempLevel--;
		}
	}
   
	// Call drawing function.
	TreeDraw_OnDrawItem(hwnd,
                       lpDrawItem,
                       nLevel,
                       dwConnectLevel,
					   (LPSTR)szName,
                       nRow,
                       nColumn,
					   &TreeDrawStruct,
					   (DirSizeInfo) ? (LPSTR)Tmp : (LPSTR)NULL);
	return;
}


#pragma argsused
BOOL DirOnCommand(HWND hwnd, UINT id, HWND hwndCtl, UINT code)
{
	int Idx;
	DWORD dwData;

	switch (code)
	{
		case LBN_SELCHANGE:
		case LBN_DBLCLK:
			Idx = SendMessage(hwndCtl, LB_GETCURSEL, 0, 0L);
			dwData = (DWORD) SendMessage(hwndCtl, LB_GETITEMDATA, (WORD)Idx, 0L);
			DirActionItem(hwndCtl, (DWORD)dwData, (WORD)Idx);
			return TRUE;
	}
    return FALSE;
}


#pragma argsused
void DirActionItem(HWND hwnd, DWORD dwData, WORD wIndex)
{
	register int i;
	int Idx;

	if (TreeDraw_IsOpened(&TreeDrawStruct, dwData))	{
    	// If opened, close item
		TreeDraw_CloseItem(&TreeDrawStruct, dwData);
	}
	else	{
    	// If closed, open item
		for (i=0; i<=Count; i++)	{
			if (TreeDraw_IsOpened(&TreeDrawStruct, (long)i))
				TreeDraw_CloseItem(&TreeDrawStruct, (long)i);
		}
		TreeDraw_OpenItem(&TreeDrawStruct, dwData);

		SendMessage(hwnd, WM_SETREDRAW, TRUE, 0L);
		InvalidateRect(hwnd, NULL, FALSE);

		Idx = SendMessage(hwnd, LB_GETCURSEL, 0, 0L);
		if (GetFiles(Tree[Idx].Path, "*.*") == -1)	{
			if (FreeUpMsgBox(MB_YESNO | MB_ICONEXCLAMATION, szAppName,
										   "Directory %s does not exist.  "
										   "Would you like to rescan the directory tree?\r\n\r\n "
										   "Select \"Yes\" to update tree structure.\r\n",
										   strupr(Tree[Idx].Path)) == IDYES)	{
				SendMessage(ghwndMain, WM_COMMAND, IDM_READDIRS, 0L);
				return;
			}
        }
    }
}


void SystemChangeColor(void)
{
	TreeDraw_DrawInit(ghInstance,
					  IDC_FOLDERS,
					  ROWS,
					  COLS,
                      TRUE,
					  &TreeDrawStruct,
					  TRUE);
	FileDraw_DrawInit(ghInstance,
					  IDC_FOLDERS,
					  ROWS,
					  COLS,
					  TRUE,
					  &FilesDrawStruct);
}


#pragma argsused
BOOL CALLBACK _export ScanTreeDlg(HWND hDlg, UINT msg,
									WPARAM wParam, LPARAM lParam)
{
	char OldDir[128], dir[128];
    char List[256], Temp[20];
    register int i;
	HFONT hFont;
	LOGFONT lFont;
    int dummy;

	switch (msg)
	{
		case WM_INITDIALOG:
			CenterDialog(hDlg);
			hTotalFiles = GetDlgItem(hDlg, IDC_TOTALFILES);
			hTotalDirs  = GetDlgItem(hDlg, IDC_TOTALDIRS);
			hCurrentDir = GetDlgItem(hDlg, IDC_CURRENTDIR);
			Quit = FALSE;

			hFont = (HFONT)NULL;
			if ((hFont = (HFONT)SendMessage(hDlg, WM_GETFONT, 0, 0L)) != 0)
            {
				if (GetObject(hFont, sizeof(LOGFONT), (LPSTR)&lFont))
                {
					lFont.lfWeight = FW_NORMAL;
					if ((hFont = CreateFontIndirect((LPLOGFONT)&lFont)) != 0)	{
						SendMessage(hTotalFiles, WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(hTotalDirs, WM_SETFONT, (WPARAM)hFont, FALSE);
						SendMessage(hCurrentDir, WM_SETFONT, (WPARAM)hFont, FALSE);
                    }
				}
			}
			return TRUE;

		case WM_COMMAND:
			switch (LOWORD(wParam))
			{
				case ID_SCAN:
					Initialize((TreeStruct _huge *)Tree);
					ScanTree();
                    SendMessage(hDlg, WM_CLOSE, 0, 0L);
                    break;
				case IDCANCEL:
					Quit = TRUE;
                    PostMessage(hDlg, WM_CLOSE, 0, 0L);
					return TRUE;
			}
			break;
		case WM_CTLCOLOR:
		{
			return (BOOL)Ctl3dCtlColorEx(msg, wParam, lParam);
		}
		case WM_PAINT:
		{
			PostMessage(hDlg, WM_COMMAND, ID_SCAN, 0L);
			return (BOOL)NULL;
		}
		case WM_CLOSE:
            DeleteObject(hFont);
			EndDialog(hDlg, IDCANCEL);
            return TRUE;
	}
	return FALSE;
}


int Compare(TreeStruct _huge *str1, TreeStruct _huge *str2)
{
	return (_fstrcmp((char *)str1->Path, (char *)str2->Path));
}
void SortDir(TreeStruct _huge *Sptr, int Num)
{
	register int i, j, k, rc;
	register TreeStruct _huge *Dptr;
	TreeStruct _huge *Ptr1;
	TreeStruct _huge *Ptr2;
	TreeStruct _huge *TbuffPtr;
	char Tbuff[256];

	Dptr = Sptr;
	TbuffPtr = (TreeStruct _huge *)&Tbuff[0];
	for (j=0; j<=Num; j++)
	{
		k = j;
		Ptr1 = (Dptr + j);
		_fmemcpy(Tbuff, Ptr1, sizeof(TreeStruct));
		for (i=j; i<=Num; i++)
		{
			Ptr1 = (Dptr+i);
			rc = Compare(Ptr1, TbuffPtr);
			if (rc < 0)	{
				_fmemcpy(Tbuff, Ptr1, sizeof(TreeStruct));
				k = i;
			}
		}
		if (j != k)
		{
			Ptr1 = (Dptr + j);
			Ptr2 = (Dptr + k);
			_fmemcpy(Tbuff, Ptr1, sizeof(TreeStruct));
			_fmemcpy(Ptr1, Ptr2, sizeof(TreeStruct));
			_fmemcpy(Ptr2, Tbuff, sizeof(TreeStruct));
		}
	}
}


void ScanTree()
{
	int len;
	struct find_t f;
	char TotalStr[16];
	MSG msg;
    int dummy;

	FlushMsgLoop();

	len = lstrlen(CurPath);
	lstrcpy(Tree[Count].Path, CurPath);
	SetWindowText(hCurrentDir, (LPSTR)Tree[Count].Path);
    // So we can sort correctly
	PathCh(CurPath, lstrlen(CurPath), '/', '\\');
	if (CurPath[lstrlen(CurPath)-1] != '\\')
		lstrcat(CurPath, "\\");
	lstrcat(CurPath,"*.*");

    // Get the directory size
	Tree[Count].DirSize = GetDirSize(CurPath, f);
    Deep++;

	if(!_dos_findfirst(CurPath, 0xffff, &f) && !Quit) {
		if ((f.attrib & _A_SUBDIR) == _A_SUBDIR && f.name[0] != '.') {

        	// Chop it to the original length
			CurPath[len] = '\0';
			if (CurPath[lstrlen(CurPath)-1] != '\\')
				lstrcat(CurPath,"\\");
            // Add the new directory name
			lstrcat(CurPath,f.name);

			lstrcpy(TotalStr, ltoa(TotalFiles, "", 10));
			SetWindowText(hTotalFiles, (LPCSTR)TotalStr);

			Count++;
			Tree[Count].Level = Deep - 1;
			Tree[Count].LastChild = FALSE;
			Tree[Count].IsParent = FALSE;
			Tree[Count].ChildNum = 0;
			lstrcpy(Tree[Count].Name, f.name);
			PathCh(CurPath, lstrlen(CurPath), '\\', '/');
			SetWindowText(hTotalDirs, (LPCSTR)itoa(Count, "", 10));

            // Recurse again
			ScanTree();
            Deep--;
		}

		while(!_dos_findnext(&f)) {
			if ((f.attrib & _A_SUBDIR) == _A_SUBDIR && f.name[0] != '.') {
				CurPath[len] = '\0';
			if (CurPath[lstrlen(CurPath)-1] != '\\')
					lstrcat(CurPath, "\\");
				lstrcat(CurPath, f.name);

				lstrcpy(TotalStr, ltoa(TotalFiles, "", 10));
				SetWindowText(hTotalFiles, (LPCSTR)TotalStr);

                Count++;
				Tree[Count].Level = Deep - 1;
				Tree[Count].LastChild = FALSE;
                Tree[Count].IsParent = FALSE;
				lstrcpy(Tree[Count].Name, f.name);
				PathCh(CurPath, lstrlen(CurPath), '\\', '/');
				SetWindowText(hTotalDirs, (LPCSTR)itoa(Count, "", 10));
                // Recurse some more
				ScanTree();
                Deep--;
			}
		}
	}
    // Cut off the file mask and return
	CurPath[len] = '\0';
}


void Initialize(TreeStruct _huge *Tptr)
{
	char tmp[128];
	register int i;
	struct find_t f;
    TreeStruct _huge *TreePtr;

    TreePtr = Tptr;
	for (i=0; i<MAXPATHS; i++)	{
		_fmemset(TreePtr->Name, '\0', lstrlen(TreePtr->Name));
		_fmemset(TreePtr->Path, '\0', lstrlen(TreePtr->Path));
//		_fmemset(TreePtr->CustomName, '\0', lstrlen(TreePtr->CustomName));
		TreePtr->DirSize = 0L;
		TreePtr->Parent = 0;
        TreePtr->SubCnt = 0;
		TreePtr->Level = 0;
		TreePtr->LastChild = FALSE;
		TreePtr->IsParent = FALSE;
        TreePtr->ChildNum = 0;
        TreePtr++;
	}

    setdisk(CurrentDrive);
	chdir("\\");
	_fmemset(CurPath, '\0', sizeof(CurPath));

	TotalTaggedFiles = 0;
	TotalTaggedBytes = 0L;
    Tree[0].Level = 0;
	Deep = 1;
	TotalFiles = 0;
    Count = 0;

	wsprintf(CurPath, "%c:/", getdisk() + 'A');
	lstrcpy(Tree[0].Path, CurPath);
}

#pragma argsused
unsigned long GetDirSize(char *Path, struct find_t f)
{
	int Attr = _A_ARCH + _A_HIDDEN + _A_SYSTEM + _A_RDONLY;
	int dun;
	unsigned long Cnt = 0L;
    struct find_t fp;

	dun = _dos_findfirst(Path, Attr, &fp);
	while (!dun)	{
		Cnt += fp.size;

		TotalFiles++;
		dun = _dos_findnext(&fp);
	}
	return(Cnt);
}


void InitFont()
{
    FreeUpSetFont(ghwndMain);
}

void FreeUpSetFont(HWND hwnd)
{
	HFONT hFontNew;

	hFontNew = CreateFontIndirect(&logFont);
	DeleteObject(ghFont);
	SendMessage(hwnd, WM_SETFONT, hFontNew, 0L);
	ghFont = hFontNew;
}


BOOL ReadDataFile()
{
	FILE *fp;
	register int i;
	char buff[80];
	char CurDir[128];
    char Fname[14];

	getcwd(CurDir, 128);
    LogPath(StartUpDir);

    wsprintf(Fname, "FUDRIVE._%c", CurDir[0]);
	if ((fp = fopen(Fname, "rb")) == NULL)	{
//		MessageBox(NULL, "Could not open data file!", "Read Error", MB_OK);

		return FALSE;
	}

	fread(&Count, sizeof(int), 1, fp);
	fread(&TotalFiles, sizeof(int), 1, fp);

	for (i=0; i<=Count; i++)	{
/*
		fread(&Tree[i].Name, sizeof(Tree[i].Name), 1, fp);
		fread(&Tree[i].Path, sizeof(Tree[i].Path), 1, fp);
		fread(&Tree[i].DirSize, sizeof(Tree[i].DirSize), 1, fp);
		fread(&Tree[i].Parent, sizeof(Tree[i].Parent), 1, fp);
		fread(&Tree[i].SubCnt, sizeof(Tree[i].SubCnt), 1, fp);
		fread(&Tree[i].Level, sizeof(Tree[i].Level), 1, fp);
		fread(&Tree[i].LastChild, sizeof(Tree[i].LastChild), 1, fp);
		fread(&Tree[i].IsParent, sizeof(Tree[i].IsParent), 1, fp);
*/
		fread(&Tree[i], sizeof(TreeStruct), 1, fp);
	}

	fclose(fp);
	LogPath(CurDir);
	return TRUE;
}
void WriteDataFile()
{
	FILE *fp, *fp2;
	register int i;
	char buff[80];
	char CurDir[128];
	char Fname[14], Fname2[14];

	getcwd(CurDir, 128);
    LogPath(StartUpDir);

	wsprintf(Fname, "FUDRIVE._%c", CurDir[0]);
	if ((fp = fopen(Fname, "wb")) == NULL)	{
		MessageBox(NULL, "Could not open data file!", "Error", MB_OK);
		return;
	}
/*
	wsprintf(Fname2, "$TEST$_%c", CurDir[0]);
	if ((fp2 = fopen(Fname2, "w")) == NULL)	{
		MessageBox(NULL, "Could not open test file!", "Error", MB_OK);
		return;
	}
*/
	FlushMsgLoop();

    LoadWaitCursor();
	fwrite(&Count, sizeof(int), 1, fp);
	fwrite(&TotalFiles, sizeof(int), 1, fp);
//    fprintf(fp2, "%d\n", Count);
	for (i=0; i<=Count; i++)	{
/*
		fwrite(&Tree[i].Name, sizeof(Tree[i].Name), 1, fp);
		fwrite(&Tree[i].Path, sizeof(Tree[i].Path), 1, fp);
		fwrite(&Tree[i].DirSize, sizeof(Tree[i].DirSize), 1, fp);
		fwrite(&Tree[i].Parent, sizeof(Tree[i].Parent), 1, fp);
		fwrite(&Tree[i].SubCnt, sizeof(Tree[i].SubCnt), 1, fp);
		fwrite(&Tree[i].Level, sizeof(Tree[i].Level), 1, fp);
		fwrite(&Tree[i].LastChild, sizeof(Tree[i].LastChild), 1, fp);
		fwrite(&Tree[i].IsParent, sizeof(Tree[i].IsParent), 1, fp);
*/
		fwrite(&Tree[i], sizeof(TreeStruct), 1, fp);
/*
		fprintf(fp2, "%s %s %lu %d %d %d %d %d %d\n",
						Tree[i].Name,
						Tree[i].Path,
						Tree[i].DirSize,
						Tree[i].Parent,
						Tree[i].SubCnt,
						Tree[i].Level,
						Tree[i].LastChild,
						Tree[i].IsParent,
						Tree[i].ChildNum);
*/
	}

    LoadArrowCursor();
	fclose(fp);
//	fclose(fp2);
	LogPath(CurDir);
}


#pragma argsused
BOOL CALLBACK _export AboutDlgProc(HWND hDlg, UINT msg,
								   WPARAM wParam, LPARAM lParam)
{
	switch (msg)
	{
		case WM_INITDIALOG:
        	CenterDialog(hDlg);
			return TRUE;
		case WM_COMMAND:
			switch (wParam)
			{
				case IDB_REGISTER:
				case IDB_REGISTERNOT:
					SendMessage(hDlg, WM_CLOSE, 0, 0L);
					break;
			}
			break;
		case WM_CLOSE:
			EndDialog(hDlg, IDOK);
            break;
	}
	return FALSE;
}

LRESULT CALLBACK _export CenterWndProc(HWND hwnd, UINT msg,
									   WPARAM wParam, LPARAM lParam)
{
	static BOOL FirstCall;

	switch (msg)
	{
		case WM_CREATE:
			FirstCall = TRUE;
			hResizeCur = LoadCursor(ghInstance, MAKEINTRESOURCE(IDC_RESIZE));
            return 0;
		case WM_SETCURSOR:
			if (wParam == (HWND)ghwndSeparator)	{
            	SetCursor(hResizeCur);
			}
			return 0L;
		case WM_NCHITTEST:
		{
			if (!wParam)
				return HTCAPTION;
			return 0L;
        }
		case WM_MOVE:
		{
        	RECT r, rcMain;
			if (FirstCall)
				return 0;

			GetWindowRect(GetParent(hwnd), &rcMain);
            GetWindowRect(hwnd, &r);
			r.top = rcMain.top - rcMain.bottom;
			r.bottom = rcMain.bottom - StatBarHeight; // * 20;

			Center = r.left - rcMain.left -	GetSystemMetrics(SM_CXFRAME);

			SendMessage(ghwndMain, WM_SIZE, SIZENORMAL, 0L);

			return 0L;
		}
		case WM_PAINT:
		{
			HDC hdc;
			PAINTSTRUCT ps;

			hdc = BeginPaint(hwnd, &ps);

			SelectObject(hdc, GetStockObject(GRAY_BRUSH));

			EndPaint(hwnd, &ps);
            FirstCall = FALSE;
			return 0L;
		}
		case WM_DESTROY:
        	DestroyCursor(hResizeCur);
			DestroyWindow(ghwndSeparator);
			UnregisterClass("SeparatorClass", ghInstance);
            return 0L;
    }
	return DefWindowProc(hwnd, msg, wParam, lParam);
}


LRESULT CALLBACK _export StatusBarWndProc(HWND hwnd, UINT msg,
									   WPARAM wParam, LPARAM lParam)
{
	switch (msg)
	{
		case WM_CREATE:
		{
			HDC hdc;
			int TextHeight, BdrHeight;
			TEXTMETRIC tm;

			hdc = GetDC(hwnd);
			SelectObject(hdc, GetStockObject(ANSI_VAR_FONT));
            GetTextMetrics(hdc, &tm);
			TextHeight = tm.tmHeight + tm.tmExternalLeading + tm.tmDescent;
            BdrHeight = GetSystemMetrics(SM_CXFRAME);
			StatBarHeight = TextHeight + (BdrHeight * 2) + 2;
			ReleaseDC(hwnd, hdc);
			return 0L;
		}
		case WM_MOVE:
		{
        	RECT r, rcMain;

			GetClientRect(ghwndMain, &rcMain);
			GetWindowRect(hwnd, &r);
            r.left   = rcMain.left;
			r.top    = rcMain.bottom - StatBarHeight;
            r.right  = rcMain.right;
			r.bottom = rcMain.bottom;

			SetWindowPos(hwnd, NULL, r.left, r.top, r.right, r.bottom, SWP_SHOWWINDOW);
			return 0L;
		}

		case WM_PAINT:
		{
			HDC hdc;
            RECT rc, rcS;
			PAINTSTRUCT ps;
			TEXTMETRIC tm;
			int CxWidth, BdrWidth, TxtHeight, BdrHeight;

			hdc = BeginPaint(hwnd, &ps);

			GetClientRect(hwnd, &rc);
            GetClientRect(ghwndSeparator, &rcS);
            GetTextMetrics(hdc, &tm);

			SelectObject(hdc, GetStockObject(GRAY_BRUSH));

			CxWidth = tm.tmAveCharWidth;
			BdrWidth = GetSystemMetrics(SM_CYFRAME);

			StatusRect(hdc, rc.left, rc.top, rc.right, rc.bottom, 0, SR_RAISED);

			DirTotalsRect.left   = rc.left + CxWidth + 1;
			DirTotalsRect.top    = rc.top + BdrWidth + 1;
			DirTotalsRect.right  = Center - (CxWidth/2) - 1;
			DirTotalsRect.bottom = rc.bottom - BdrWidth - 1;

			StatusRect(hdc, rc.left+CxWidth, rc.top+BdrWidth,
					Center - (CxWidth/2), rc.bottom - BdrWidth, 0, SR_GROUP);

			FileTotalsRect.left   = Center + rcS.right + (CxWidth/2) + 1;
			FileTotalsRect.top    = rc.top + BdrWidth + 1;
			FileTotalsRect.right  = rc.right - CxWidth - 1;
			FileTotalsRect.bottom = rc.bottom - BdrWidth - 1;
			  
			StatusRect(hdc, Center + rcS.right + (CxWidth/2), rc.top+BdrWidth,
					rc.right - CxWidth, rc.bottom - BdrWidth, 0, SR_GROUP);

			DrawDirTotals();
			DrawFileTotals();
			EndPaint(hwnd, &ps);
			return 0L;
		}
		case WM_DESTROY:
			DestroyWindow(ghwndStatusBar);
            UnregisterClass("StatusBarClass", ghInstance);
            return 0L;
    }
	return DefWindowProc(hwnd, msg, wParam, lParam);
}

void DrawDirTotals(void)
{
	HDC hDC;
	char CountBuff[40];
    char SizeBuff[40];
	char TotalBuff[80];
	int TxtCenter, OldBkClr, TxtHeight;
    WORD wExtent;
    RECT r;

	if (Count > 0)
		wsprintf(CountBuff, "%d Directories", Count+1);
    else
		wsprintf(CountBuff, "%d Directory", Count+1);

    lstrcpy(SizeBuff, ul2str(TotalFiles));
	wsprintf(TotalBuff, "%s Total Files", SizeBuff);
   	hDC = GetDC(ghwndStatusBar);

	SelectObject(hDC, GetStockObject(ANSI_VAR_FONT));

    CopyRect(&r, &DirTotalsRect);

	wExtent = LOWORD(GetTextExtent(hDC, TotalBuff, lstrlen(TotalBuff)));
 //   TxtHeight = HIWORD(GetTextExtent(hDC, "xxxx", 4));
	OldBkClr = SetBkColor(hDC, RGB(192, 192, 192));

	TxtCenter = r.top;

	ExtTextOut(hDC, r.right - wExtent - 1, TxtCenter, ETO_CLIPPED | ETO_OPAQUE,
				&r, (LPSTR)TotalBuff, lstrlen(TotalBuff), NULL);
	ExtTextOut(hDC, r.left+1, TxtCenter, ETO_CLIPPED,
				&r, (LPSTR)CountBuff, lstrlen(CountBuff), NULL);
	SetBkColor(hDC, OldBkClr);
	ReleaseDC(ghwndStatusBar, hDC);
}
void DrawFileTotals(void)
{
	HDC hDC;
	char DirSizeBuff[40];
    char TaggedSizeBuff[40];
	char FilesBuff[80];
    char TaggedBuff[80];
	int TxtCenter, OldBkClr, TxtHeight;
    WORD wExtent;
	RECT r;
	register int i;

	TotalTaggedFiles = GetTaggedFileCount();
	TotalTaggedBytes = 0L;

	if (TotalTaggedFiles == 0)
		EnableButtons(FALSE);
	else if (TotalTaggedFiles > 0)
		EnableButtons(TRUE);

	for (i=0; i<FileCount; i++)	{
		if (SendMessage(ghwndFileBox, LB_GETSEL, i, 0L))	{
			TotalTaggedBytes += Files[i].Size;
			Files[i].Tagged = TRUE;
		}
		else	{
			Files[i].Tagged = FALSE;
		}
	}

	lstrcpy(DirSizeBuff, ul2str(DirSize));
	lstrcpy(TaggedSizeBuff, ul2str(TotalTaggedBytes));
	wsprintf(FilesBuff, "%d File(s) %s Bytes", FileCount, DirSizeBuff);
    wsprintf(TaggedBuff, "%d Selected Files %s Bytes", TotalTaggedFiles, TaggedSizeBuff);
   	hDC = GetDC(ghwndStatusBar);

	SelectObject(hDC, GetStockObject(ANSI_VAR_FONT));

    CopyRect(&r, &FileTotalsRect);

	wExtent = LOWORD(GetTextExtent(hDC, TaggedBuff, lstrlen(TaggedBuff)));
//    TxtHeight = HIWORD(GetTextExtent(hDC, "xxxx", 4));
	OldBkClr = SetBkColor(hDC, RGB(192, 192, 192));

	TxtCenter = r.top;

	ExtTextOut(hDC, r.right - wExtent - 1, TxtCenter, ETO_CLIPPED | ETO_OPAQUE,
				&r, (LPSTR)TaggedBuff, lstrlen(TaggedBuff), NULL);
	ExtTextOut(hDC, r.left+1, TxtCenter, ETO_CLIPPED,
				&r, (LPSTR)FilesBuff, lstrlen(FilesBuff), NULL);
	SetBkColor(hDC, OldBkClr);
	ReleaseDC(ghwndStatusBar, hDC);
}
