// =========================================================
// TOOLTIP.CPP
//
// Tooltip classes for OWL 2
//
// Author:  Steve Saxon (Compuserve: 100321,2355)
// Written: 26th June 1994
// =========================================================

#include <owl\owlpch.h>
#pragma hdrstop

#include "tooltip.h"

#define TIP_DELAY	400
#define ID_TIMER	1000

static BOOL bEnableHints = FALSE;	// prevents menus displaying hints!! 

DEFINE_RESPONSE_TABLE1 (TToolTip, TWindow)
  EV_WM_TIMER,
END_RESPONSE_TABLE;

TToolTip::TToolTip () : TWindow (NULL, "")
{
	Attr.Style 		= WS_POPUP | WS_BORDER;
	Attr.ExStyle 	= WS_EX_TOPMOST;

	uiTimer			= NULL;
	bEnabled		= TRUE;

	Create ();
}

TToolTip::~TToolTip ()
{
	KillTipTimer ();
}

void TToolTip::GetWindowClass (WNDCLASS &wc)
{
	TWindow::GetWindowClass (wc);

	// yellow background brush
	wc.hbrBackground = CreateSolidBrush (RGB (255, 255, 0));
}

LPSTR TToolTip::GetClassName ()
{
	return "Steve:Tooltip";
}

void TToolTip::Paint (TDC &dc, BOOL, TRect &)
{
	char	szText[50];
	TRect	client;

	// get the tooltip text
	GetWindowText (szText, sizeof (szText));
	GetClientRect (client);

	// set up the device context
	dc.SetBkMode (TRANSPARENT);
	dc.SelectStockObject (ANSI_VAR_FONT);

	// draw the text
	dc.DrawText (szText, -1, client, DT_CENTER | DT_SINGLELINE | DT_VCENTER);
}

void TToolTip::SetCaption (const char far* title)
{
	TWindow::SetCaption (title);

	// if the caption is missing, hide the tip window
	if (title == NULL || !*title || !bEnabled)
	{
		KillTipTimer ();

		Show (SW_HIDE);
	}
	else
	{
		// work out the extent of the text
		{
			TClientDC	dc (HWindow);

			dc.SelectStockObject (ANSI_VAR_FONT);
			sizeText = dc.GetTextExtent (title, lstrlen (title));

			sizeText.cx	+= 5;
			sizeText.cy	+= 4;

			Show (SW_HIDE);
		}

		// create the timer - this will send a WM_TIMER message
		// after 'TIP_DELAY' milliseconds
		uiTimer 		= SetTimer (ID_TIMER, TIP_DELAY);
	}
}

void TToolTip::PositionTip ()
{
	// position the tip relative to the mouse
	TPoint	ptMouse;
	TSize	scr (GetSystemMetrics (SM_CXSCREEN), GetSystemMetrics (SM_CYSCREEN));

	GetCursorPos (ptMouse);

	ptMouse.x	-= 2;
	ptMouse.y	+= 22;

	TRect	rc (ptMouse, sizeText);

	// check x screen position
	if (rc.left < 0)
	{
		rc.Offset (-rc.left + 2, 0);
	}
	else
	{
		if (rc.right > scr.cx)
		{
			rc.Offset (scr.cx - rc.right - 2, 0);
		}
	}

	// check y screen position
	if (rc.bottom > scr.cy)
	{
		rc.Offset (0, -42);
	}

	SetWindowPos (NULL, rc, SWP_NOZORDER | SWP_NOACTIVATE);
}

void TToolTip::KillTipTimer ()
{
	// destroy the timer
	if (uiTimer)
	{
		KillTimer (ID_TIMER);
		uiTimer = NULL;
	}
}

void TToolTip::ShowNow ()
{
	// position the tip window
	PositionTip ();
	
	// show the tip window
	Show (SW_SHOWNA);
	UpdateWindow ();
}

void TToolTip::EvTimer (UINT)
{
	// timer message received - show the tip window!
	if (uiTimer)
	{
		KillTipTimer ();	// prevent further timer messages

		ShowNow ();
	}
}

// ================================================================ //
// override class for control bar
// ================================================================ //

DEFINE_RESPONSE_TABLE1 (TTipControlBar, TControlBar)
  EV_WM_MOUSEMOVE,
  EV_WM_LBUTTONDOWN,
END_RESPONSE_TABLE;

TTipControlBar::TTipControlBar (TToolTip& tip, TWindow* parent, TTileDirection direction, TFont* font, TModule* module)
	 : 	TControlBar (parent, direction, font, module),
		tooltip (tip)
{
	SetHintMode(TGadgetWindow::EnterHints);
}

void TTipControlBar::EvMouseMove (UINT modKeys, TPoint& point)
{
	if (!Capture && !GadgetFromPoint (point))
	{
		// hide the tip window if not over a gadget
		tooltip.HideTip ();
	}

	bEnableHints = TRUE;

	TControlBar::EvMouseMove (modKeys, point);

	bEnableHints = FALSE;
}

void TTipControlBar::EvLButtonDown (UINT modKeys, TPoint& point)
{
	// hide the tip window if mouse-button pressed
	tooltip.HideTip ();

	TControlBar::EvLButtonDown (modKeys, point);
}


// ================================================================ //
// override class for status bar
// ================================================================ //

TTipStatusBar::TTipStatusBar (TToolTip& tip, TWindow* parent, TGadget::TBorderStyle borderStyle, UINT modeIndicators, TFont *font, TModule* module)
	: 	TStatusBar (parent, borderStyle, modeIndicators, font, module),
		tooltip (tip)
{

}

void TTipStatusBar::SetHintText (const char *lpszText)
{
	// when hint message displayed on the status bar,
	// pick out the tooltip text and display it (with delay)

	if (lpszText != NULL)
	{
		static char	buf[128];

		lstrcpy (buf, lpszText);

		// locate the tooltip text
		for (int n = 0; buf[n] && buf[n] != '\n'; n++) ;

		if (buf[n])
		{
			buf[n++] = '\0';
		}

		// only display hints from gadgets (not menus!)
		if (bEnableHints)
		{
			tooltip.SetCaption (buf + n);
		}

		TStatusBar::SetHintText (buf);
	}
	else
	{
		tooltip.SetCaption (NULL);

		TStatusBar::SetHintText (NULL);
	}
}

