// PV Browser

#include <windows.h>
#include <shellapi.h>
#include <string.h>
#include "browse.h"
#include "filedial.h"

static void diag(char *f, ...)  // diagnostic
{
  char s[256];
  wvsprintf(s, f, &f + 1);
  MessageBox(NULL, s, "Diagnostic Message", MB_OK);
}

extern "C" {
  long far pascal MainWindowProc(HWND, UINT, WPARAM, LPARAM);
  int far pascal  SearchDialogProc(HWND, UINT, WPARAM, LPARAM);
  int far pascal  ListDialogProc(HWND, UINT, WPARAM, LPARAM);
  int pascal      WinMain(HANDLE, HANDLE, LPSTR, int);
}

const COLORREF BLACK           = RGB(255,255,255);
const unsigned BLOCK_SIZE      = 4*1024;
const int      BROWSE_FONTS    = 4;
const int      EOF             = -1;
const int      INITIAL_HEIGHT  = 300;
const int      INITIAL_WIDTH   = 500;
const int      MAX_FILESPECS   = MAX_DIR+1+MAX_FILE;
const int      MAX_SEARCH_TEXT = 79;
const int      SCROLL_RANGE    = 100;
const int      STATUS_HEIGHT   = 24;
const COLORREF WHITE           = RGB(0,0,0);
const int      MAX_FILES       = 10;
const int      MAX_BLOCKS      = MAX_FILES+40;

static char title_bar[12+MAX_FILESPECS+1] = "PV Browse - (no file)";

typedef struct tagWIDEHIGH
{
  long width;
  long height;
} WIDEHIGH;

typedef struct tagPOSITION
{
  long row;
  long column;
  long position;
} POSITION;

typedef struct tagBIGRECT
{
  long left;
  long top;
  long right;
  long bottom;
} BIGRECT;

typedef struct tagSTATUS
{
  long row;
  long column;
  long height;
} STATUS; 

struct block;

struct file
{
  long      height;
  POSITION  corner;
  block    *current_block;
  POSITION  cursor;
  char      filespecs[MAX_FILESPECS+1];
  int       font_number;
  HFILE     handle;
  POSITION  mark;
  BOOL      marked;
  BOOL      match_case;
  file     *next;
  file     *previous;
  WIDEHIGH  text;
  POSITION  beginning;

  void      clip_rectangle(RECT &, BIGRECT &);
  void      invalidate_rectangle(BIGRECT &);
  void      center_cursor(void);
  void      error_message(char *);
  void      invalidate_difference(BIGRECT &, BIGRECT &);
  void      marked_rectangle(BIGRECT &, POSITION &);
  HGLOBAL   marked_text(void);
  long      message(UINT, WPARAM, LPARAM);
  void      move_mark(POSITION &);
  void      remove_mark(POSITION &);
  BOOL      next_row(POSITION &);
  void      previous_row(POSITION &);
  int       read_and_advance(POSITION &);
  int       read_character(DWORD);
  void      zoom(void);
  void      scroll(int, int);
  void      scroll_if_necessary(void);
  void      search_again(void);
  void      set_cursor(void);
  void      set_scroll_bars(void);
  void      update_status(void);
  void      update_title(void);

  static HFONT     font[BROWSE_FONTS];
  static WIDEHIGH  character[BROWSE_FONTS];
  static file     *current;
  static HWND      window;
  static RECT      display;
  static HWND      status_bar;
  static STATUS    display_status;
  static int       number_of_files;
  static file     *title;
  static char      search_text[MAX_SEARCH_TEXT+1];
  static file     *dummy;

  static void      close(void);
  static BOOL      open(char *);
};

HFONT     file::font[BROWSE_FONTS];
WIDEHIGH  file::character[BROWSE_FONTS];
file     *file::current;
file     *file::title;
RECT      file::display;
HWND      file::window;
HWND      file::status_bar;
STATUS    file::display_status;
int       file::number_of_files;
char      file::search_text[MAX_SEARCH_TEXT+1];
file     *file::dummy;

struct block
{
  unsigned           count;
  unsigned char far *data;
  file              *f;
  HGLOBAL            hdata;
  block             *older;
  long               position;
  block             *younger;

                    ~block(void);
  void               insert(void);
  void               remove(void);

  static block       *oldest;
  static int         number_of_blocks;
};

block *block::oldest;
int    block::number_of_blocks;

static HINSTANCE application_instance;

/*-----------------------------------------------------------------------------
This function takes a locked block, inserts it into the age queue as the
youngest block, and unlocks it.
-----------------------------------------------------------------------------*/

void block::insert(void)
{
  if (oldest == NULL)
  {
    oldest = this;
    younger = older = this;
  }
  else
  {
    younger = oldest;
    older = oldest->older;
    younger->older = older->younger = this;
  }
  GlobalUnlock(hdata);
  data = NULL;
}

/*-----------------------------------------------------------------------------
This function removes a block from the age queue.
-----------------------------------------------------------------------------*/

void block::remove(void)
{
  if (oldest == this) oldest = oldest->younger;
  if (oldest == this)
    oldest = NULL;
  else
  {
    younger->older = older;
    older->younger = younger;
  }
}

/*-----------------------------------------------------------------------------
This function removes a block from the age queue and locks it. It returns TRUE
if the operation was successful. If the block had been discarded, it destroys
the block and returns FALSE.
-----------------------------------------------------------------------------*/

static BOOL remove_and_lock(block *b)
{
  b->remove();
  b->data = GlobalLock(b->hdata);
  if (b->data == NULL)
  {
    GlobalUnlock(b->hdata);
    GlobalFree(b->hdata);
    delete b;
    return FALSE;
  }
  return TRUE;
}

/*-----------------------------------------------------------------------------
The destructor removes a block from the age queue, if it is in one, and
releases its global memory block.
-----------------------------------------------------------------------------*/

block::~block()
{
  if (data == NULL)
    remove();
  else
    GlobalUnlock(hdata);
  GlobalFree(hdata);
  number_of_blocks--;
}

/*-----------------------------------------------------------------------------
This function gets a new block, locks it and reads a block into it, if
possible. If that is not possible, it destroys the oldest block in the age
queue and tries again. The process continues until a block is succesfully
obtained or until the age queue is emptied, in which case it returns NULL.
-----------------------------------------------------------------------------*/

static block *new_block(file *f, long position)
{
  while (TRUE)
  {
    block *b;
    if (block::number_of_blocks < MAX_BLOCKS && (b = new block) != NULL)
    {
      block::number_of_blocks++;
      b->hdata = GlobalAlloc(GMEM_MOVEABLE | GMEM_DISCARDABLE, BLOCK_SIZE);
      if (b->hdata != NULL)
      {
        b->data = GlobalLock(b->hdata);
        b->f = f;
        b->position = position;
        b->count = _llseek(f->handle, position, 0) == position ?
          _lread(f->handle, b->data, BLOCK_SIZE) : 0;
        return b;
      }
      b->data = NULL;
      delete b;
    }
    if (block::oldest == NULL) return NULL;
    delete block::oldest;
  }
}

/*-----------------------------------------------------------------------------
This function deletes all blocks in the age queue for a particular file, or
for all files if f == NULL.
-----------------------------------------------------------------------------*/

static void delete_all_blocks(file *f)
{
  while (block::oldest != NULL && (f == NULL || block::oldest->f == f))
    delete block::oldest;
  if (block::oldest != NULL)
  {
    block *b = block::oldest->older;
    while (b != block::oldest)
    {
      block *next = b->older;
      if (b->f == f) delete b;
      b = next;
    }
  }
}

/*-----------------------------------------------------------------------------
This function opens a file, makes it the current file, and returns TRUE if the
operation was successful.
-----------------------------------------------------------------------------*/

BOOL file::open(char *specs)
{
  file *f = new file;
  if (f == NULL) return FALSE;
  memset(f, 0, sizeof(file));
  f->height = -1L;
  f->handle = _lopen(specs, 0);
  if (f->handle < 0)
  {
    delete f;
    return FALSE;
  }
  f->current_block = new_block(f, 0L);
  if (f->current_block == NULL)
  {
    _lclose(f->handle);
    delete f;
    return FALSE;
  }
  if (number_of_files == 0)
    f->next = f->previous = f;
  else
  {
    f->next = current->next;
    f->previous = current;
    current->next = current->next->previous = f;
  }
  current = f;
  f->font_number = 2;
  lstrcpy(f->filespecs, specs);
  number_of_files++;
  return TRUE;
}

/*-----------------------------------------------------------------------------
This function closes the current file and makes the next file the current
file.
-----------------------------------------------------------------------------*/

void file::close(void)
{
  _lclose(current->handle);
  delete current->current_block;
  delete_all_blocks(current);
  file *f = current;
  current = f->next;
  if (f == current)
    current = dummy;
  else
  {
    f->previous->next = current;
    current->previous = f->previous;
  }
  delete f;
  number_of_files--;
}


/*-----------------------------------------------------------------------------
This function destroys and remakes the cursor in a new size.
-----------------------------------------------------------------------------*/

static void remake_cursor(int font_number)
{
  DestroyCaret();
  CreateCaret(file::window, NULL, 2, file::character[font_number].height);
  ShowCaret(file::window);
}

/*-----------------------------------------------------------------------------
This function invalidates the part of rectangle r1 that is not inside
rectangle r2.
-----------------------------------------------------------------------------*/

void file::invalidate_difference(BIGRECT &r1, BIGRECT &r2)
{
  BIGRECT r, i;
  r = r1;
  if (r.left < r2.left)
  {
    if (r.right <= r.left)
    {
      invalidate_rectangle(r);
      return;
    }
    i.top = r.top;
    i.bottom = r.bottom;
    i.left = r.left;
    r.left = i.right = r2.left;
    invalidate_rectangle(i);
  }
  if (r.right > r2.right)
  {
    if (r.left >= r2.right)
    {
      invalidate_rectangle(r);
      return;
    }
    i.top = r.top;
    i.bottom = r.bottom;
    i.right = r.right;
    r.right = i.left = r2.right;
    invalidate_rectangle(i);
  }
  if (r.top < r2.top)
  {
    if (r.bottom <= r.top)
    {
      invalidate_rectangle(r);
      return;
    }
    i.left = r.left;
    i.right = r.right;
    i.top = r.top;
    r.top = i.bottom = r2.top;
    invalidate_rectangle(i);
  }
  if (r.bottom > r2.bottom)
  {
    if (r.top >= r2.bottom)
    {
      invalidate_rectangle(r);
      return;
    }
    i.left = r.left;
    i.right = r.right;
    i.bottom = r.bottom;
    r.bottom = i.top = r2.bottom;
    invalidate_rectangle(i);
  }
}

/*-----------------------------------------------------------------------------
This function handles all messages to the main window by passing them to
file::message().
-----------------------------------------------------------------------------*/

long far pascal MainWindowProc(HWND h, UINT msg, WPARAM wparm, LPARAM lparm)
{
  file::window = h;
  return file::current->message(msg, wparm, lparm);
}

/*-----------------------------------------------------------------------------
This function handles all messages to the List dialog box.
-----------------------------------------------------------------------------*/

#pragma argsused
int far pascal ListDialogProc(HWND h, UINT msg, WPARAM wparm, LPARAM lparm)
{
  switch (msg)
  {
    case WM_INITDIALOG:
    {
      file *f = file::current;
      do
      {
        char *s = f->filespecs + strlen(f->filespecs);
        while (s > f->filespecs && s[-1] != '\\' && s[-1] != ':') s--;
        SendDlgItemMessage(h, LIST_BOX, LB_ADDSTRING, 0, (LPARAM)(LPCSTR) s);
        SendDlgItemMessage(h, LIST_BOX, LB_SETCURSEL, 0, 0L);
        f = f->next;
      } while (f != file::current);
      return 1;
    }
    case WM_COMMAND:
      switch (wparm)
      {
        case LIST_OK:
        {
          int i = SendDlgItemMessage(h, LIST_BOX, LB_GETCURSEL, 0, 0L);
          if (i == LB_ERR) i = 0;
          file *f = file::current;
          while (i-- > 0) f = f->next;
          EndDialog(h, (int) f);
          return 1;
        }
        case LIST_BOX:
          if (HIWORD(lparm) == LBN_DBLCLK)
          {
            int i = SendDlgItemMessage(h, LIST_BOX, LB_GETCURSEL, 0, 0L);
            if (i == LB_ERR) i = 0;
            file *f = file::current;
            while (i-- > 0) f = f->next;
            EndDialog(h, (int) f);
            return 1;
          }
        }
      break;
    case WM_CLOSE:
      EndDialog(h, (int) file::current);
      return 0;
  }
  return 0;
}

/*-----------------------------------------------------------------------------
This function handles all messages to the Search dialog box.
-----------------------------------------------------------------------------*/

#pragma argsused
int far pascal SearchDialogProc(HWND h, UINT msg, WPARAM wparm, LPARAM lparm)
{
  switch (msg)
  {
    case WM_INITDIALOG:
    {
      SetWindowText(h, file::current->filespecs);
      SendDlgItemMessage(h, SEARCH_TEXT, EM_LIMITTEXT, MAX_SEARCH_TEXT, 0L);
      SetDlgItemText(h, SEARCH_TEXT, file::current->search_text);
      SendDlgItemMessage(h, SEARCH_MATCHCASE, BM_SETCHECK,
        file::current->match_case ? 1 : 0, 0L);
      return 1;
    }
    case WM_COMMAND:
      switch (wparm)
      {
        case SEARCH_SEARCH:
        {
          GetDlgItemText(h, SEARCH_TEXT, file::current->search_text,
            MAX_SEARCH_TEXT + 1);
          file::current->match_case =
            SendDlgItemMessage(h, SEARCH_MATCHCASE, BM_GETCHECK, 0, 0L);
          EndDialog(h, 1);
          return 1;
        }
        case SEARCH_CANCEL:
          EndDialog(h, 0);
          return 1;
        }
      break;
    case WM_CLOSE:
      EndDialog(h, 0);
      return 0;
  }
  return 0;
}

/*-----------------------------------------------------------------------------
This function constructs or reconstructs a font.
-----------------------------------------------------------------------------*/

static HFONT construct_font(int height)
{
  LOGFONT f;
  memset(&f, 0, sizeof(LOGFONT));
  f.lfHeight = height;
  f.lfWeight = FW_MEDIUM;
  f.lfPitchAndFamily = FIXED_PITCH | FF_ROMAN;
  f.lfQuality = PROOF_QUALITY;
  return CreateFontIndirect(&f);
}

/*-----------------------------------------------------------------------------
This function retrieves the character height and width of the currently
selelcted font in the given display context.
-----------------------------------------------------------------------------*/

static void character_height_and_width(HDC dc, long &height, long &width)
{
  TEXTMETRIC tm;
  GetTextMetrics(dc, &tm);
  width = tm.tmAveCharWidth;
  height = tm.tmHeight + tm.tmExternalLeading;
}

/*-----------------------------------------------------------------------------
This function converts a big rectangle with row and column coordinates into
a rectangle in the client area, with pixel coordinates.
-----------------------------------------------------------------------------*/

void file::clip_rectangle(RECT &r, BIGRECT &b)
{
  long left, top, right, bottom; 
  left = b.left * character[font_number].width;
  top = b.top * character[font_number].height;
  right = b.right * character[font_number].width;
  bottom = b.bottom * character[font_number].height;
  r.left = left < 0 ? 0 : left > display.right ? display.right : (int) left;
  r.top = top < 0 ? 0 : top > display.bottom ? display.bottom : (int) top;
  r.right =
    right < 0 ? 0 : right > display.right ? display.right : (int) right;
  r.bottom =
    bottom < 0 ? 0 : bottom > display.bottom ? display.bottom : (int) bottom;
}


/*-----------------------------------------------------------------------------
This function invalidates the entire window and reconstitutes it so the cursor
will be approximately at the center of the window.
-----------------------------------------------------------------------------*/

void file::center_cursor(void)
{
  corner = cursor;
  unsigned x = (display.bottom / character[font_number].height) / 2;
  while (corner.row > 0 && x > 0)
  {
    previous_row(corner);
    x--;
  }
  corner.column -= (display.right / character[font_number].width) / 2;
  if (corner.column < 0) corner.column = 0;
  InvalidateRect(window, NULL, TRUE);
  set_cursor();
  set_scroll_bars();
}

/*-----------------------------------------------------------------------------
This function pops up a box containing the specified message, together with an
OK button and the filespecs.
-----------------------------------------------------------------------------*/

void file::error_message(char *s)
{
  MessageBox(window, s, filespecs, MB_OK);
}

/*-----------------------------------------------------------------------------
This function invalidates the part of the client area covered by a big
rectangle.
-----------------------------------------------------------------------------*/

void file::invalidate_rectangle(BIGRECT &rr)
{
  RECT r;
  clip_rectangle(r, rr);
  InvalidateRect(window, &r, TRUE);
} 

/*-----------------------------------------------------------------------------
This function determines the relative row and column numbers of the marked
area between the mark position and the given cursor position.
-----------------------------------------------------------------------------*/

void file::marked_rectangle(BIGRECT &rr, POSITION &cur)
{
  rr.left = mark.column - corner.column;
  rr.top = mark.row - corner.row;
  rr.right = cur.column - corner.column;
  rr.bottom = cur.row - corner.row;
  if (rr.left > rr.right)
  {
    long x = rr.left;
    rr.left = rr.right;
    rr.right = x;
  }
  if (rr.top > rr.bottom)
  {
    long x = rr.top;
    rr.top = rr.bottom;
    rr.bottom = x;
  }
  rr.bottom++;
}

/*-----------------------------------------------------------------------------
This function allocates a global memory block, extracts the marked text,
inserts \r\n at the end of each line, puts the result into the memory block,
and returns a memory block handle, or NULL if the memory allocation failed.
-----------------------------------------------------------------------------*/

HGLOBAL file::marked_text(void)
{
  POSITION p;
  p.position = mark.position;
  p.row = mark.row;
  p.column = 0;
  long bottom = cursor.row;
  if (bottom < p.row)
  {
    p.position = cursor.position;
    p.row = cursor.row;
    bottom = mark.row;
  }
  long left = mark.column;
  long right = cursor.column;
  if (right < left)
  {
    right = mark.column;
    left = cursor.column;
  }
  DWORD size = (right - left + 2) * (bottom - p.row + 1) + 1;
  HGLOBAL h = size <= 10*1024 ? GlobalAlloc(GMEM_MOVEABLE, size) : NULL;
  if (h != NULL)
  {
    char far *s = GlobalLock(h);
    while (p.row <= bottom)
    {
      long col = p.column;
      int c = read_and_advance(p);
      if (c == EOF || c == '\n')
      {
        if (col < left) col = left;
        while (col < right)
        {
          *s++ = ' ';
          col++;
        }
        *s++ = '\r';
        *s++ = '\n';
      }
      else if (left <= col && col < right)
        *s++ = c;
      if (c == EOF) break;
    }
    *s = 0;
    GlobalUnlock(h);
  }
  return h;
}

/*-----------------------------------------------------------------------------
This function handles all messages received by the main window.
-----------------------------------------------------------------------------*/

long file::message(UINT msg, WPARAM wparm, LPARAM lparm)
{
  static char new_filespecs[MAX_FILESPECS+1];
  static HWND timer = NULL;
  static DWORD click_timeout = 0;

  switch (msg)
  {
    case WM_CREATE:
      {
        GetClientRect(window, &display);
        display.bottom -= STATUS_HEIGHT+2;
      }
      {
        unsigned i;
        for (i = 0; i < BROWSE_FONTS; i++)
        {
          font[i] = construct_font(4 << i);
          HDC dc = GetDC(window);
          HFONT old_font = SelectObject(dc, font[i]);
          character_height_and_width(dc, character[i].height,
            character[i].width);
          SelectObject(dc, old_font);
          ReleaseDC(window, dc);
        }
      }
      update_title();
      SetScrollRange(window, SB_VERT, 0, SCROLL_RANGE-1, FALSE);
      SetScrollRange(window, SB_HORZ, 0, SCROLL_RANGE-1, FALSE);
      status_bar = CreateWindow("STATIC", "row 1, column 1",
        WS_CHILD | WS_VISIBLE | SS_CENTER,
        0, display.bottom+2, display.right, STATUS_HEIGHT, window,
        -1, application_instance, NULL);
      display_status.height = -1;
      DragAcceptFiles(window, TRUE);
      return 0;
    case WM_SIZE:
      display.right = LOWORD(lparm);
      display.bottom = HIWORD(lparm) - (STATUS_HEIGHT+2);
      MoveWindow(status_bar, 0, display.bottom+2, display.right, STATUS_HEIGHT,
        TRUE);
      return 0;
    case WM_VSCROLL:
    {
      if (number_of_files == 0) break;
      switch (wparm)
      {
        case SB_LINEUP:
          if (corner.row > 0)
          {
            previous_row(corner);
            scroll(0, character[font_number].height);
            set_cursor();
            set_scroll_bars();
          }
          break;
        case SB_LINEDOWN:
        {
          if (next_row(corner))
          {
            scroll(0, -character[font_number].height);
            set_cursor();
            set_scroll_bars();
          }
          break;
        }
        case SB_PAGEUP:
          if (corner.row > 0)
          {
            int count = display.bottom / character[font_number].height - 1;
            if (count <= 0) count = 1;
            if (corner.row < count) count = corner.row;
            for (int i = 0; i < count; i++)
              previous_row(corner);
            scroll(0, character[font_number].height * count);
            set_cursor();
            set_scroll_bars();
          }
          break;
        case SB_PAGEDOWN:
        {
          int count = display.bottom / character[font_number].height - 1;
          if (count <= 0) count = 1;
          int i;
          for (i = 0; i < count; i++)
            if (!next_row(corner))
              break;
          if (i != 0)
          {
            scroll(0, -character[font_number].height * i);
            set_scroll_bars();
            set_cursor();
          }
          break;
        }
        case SB_THUMBPOSITION:
        {
          long row = LOWORD(lparm) * text.height / SCROLL_RANGE;
          while (corner.row > row)
            previous_row(corner);
          while (corner.row < row)
            next_row(corner);
          InvalidateRect(window, NULL, TRUE);
          set_scroll_bars();
          set_cursor();
          break;
        }
      }
      return 0;
    }
    case WM_HSCROLL:
    {
      if (number_of_files == 0) break;
      switch (wparm)
      {
        case SB_LINEUP:
          if (corner.column > 0)
          {
            corner.column--;
            scroll(character[font_number].width, 0);
            set_cursor();
            set_scroll_bars();
          }
          break;
        case SB_LINEDOWN:
          if (corner.column < text.width)
          {
            corner.column++;
            scroll(-character[font_number].width, 0);
            set_cursor();
            set_scroll_bars();
          }
          break;
        case SB_PAGEUP:
          if (corner.column > 0)
          {
            int count = display.right / character[font_number].width - 1;
            if (count < 0) count = 1;
            if (count > corner.column) count = corner.column;
            corner.column -= count;
            scroll(character[font_number].width * count, 0);
            set_cursor();
            set_scroll_bars();
          }
          break;
        case SB_PAGEDOWN:
          if (corner.column + 1 < text.width)
          {
            int count = display.right / character[font_number].width - 1;
            if (count < 0) count = 1;
            if (count > text.width - 1 - corner.column)
              count = text.width - 1 - corner.column;
            corner.column += count;
            scroll(-character[font_number].width * count, 0);
            set_cursor();
            set_scroll_bars();
          }
          break;
        case SB_THUMBPOSITION:
          corner.column = LOWORD(lparm) * text.width / SCROLL_RANGE;
          InvalidateRect(window, NULL, TRUE);
          set_scroll_bars();
          set_cursor();
          break;
      }
      return 0;
    }
    case WM_KEYDOWN:
    {
      if (number_of_files == 0) break;
      BOOL control = GetKeyState(VK_CONTROL) & 0x8000;
      BOOL shift = GetKeyState(VK_SHIFT) & 0x8000;
      POSITION old_cursor = cursor;
      switch (wparm)
      {
        case VK_LEFT:
          if (cursor.column == 0) return 0;
          cursor.column--;
          break;
        case VK_RIGHT:
          if (cursor.column >= text.width) return 0;
          cursor.column++;
          break;
        case VK_HOME:
          cursor.column = 0;
          if (control)
          {
            cursor.row = 0;
            cursor.position = 0;
          }
          break;
        case VK_END:
          if (control)
          {
            while (next_row(cursor));
          }
          cursor.column = text.width;
          break;
        case VK_UP:
          if (cursor.row == 0) return 0;
          previous_row(cursor);
          break;
        case VK_DOWN:
          next_row(cursor);
          break;
        case VK_PRIOR:
          if (cursor.row == 0) return 0;
          {
            int count = display.bottom / character[font_number].height - 1;
            if (count <= 0) count = 1;
            while (count > 0 && cursor.row > 0)
            {
              previous_row(cursor);
              count--;
            }
          }
          break;
        case VK_NEXT:
        {
          int count = display.bottom / character[font_number].height - 2;
          if (!next_row(cursor)) return 0;
          while (count > 0 && next_row(cursor)) count--;
          break;
        }
        default:
          return DefWindowProc(window, msg, wparm, lparm);
      }
      if (shift)
        move_mark(old_cursor);
      else
        remove_mark(old_cursor);
      scroll_if_necessary();
      return 0;
    }
    case WM_LBUTTONDOWN:
    {
      if (number_of_files == 0) break;
      int row = HIWORD(lparm) / character[font_number].height;
      int column = (LOWORD(lparm) + character[font_number].width/2) /
        character[font_number].width;
      remove_mark(cursor);
      if (column >= display.right / character[font_number].width)
        column--;
      if (row >= display.bottom / character[font_number].height)
        row--;
      cursor = corner;
      while (cursor.row < row + corner.row && cursor.row + 1 < text.height)
        next_row(cursor);
      while (cursor.row > row + corner.row && cursor.row > 0)
        previous_row(cursor);
      cursor.column = column + corner.column;
      if (cursor.column > text.width)
        cursor.column = text.width;
      set_cursor();
      return 0;
    }
    case WM_MOUSEMOVE:
      if (number_of_files != 0 && wparm & MK_LBUTTON &&
        GetTickCount() >= click_timeout && text.height != 0)
      {
        long row = HIWORD(lparm) / character[font_number].height + corner.row;
        if (row > text.height - 1)
          row = text.height - 1;
        long column = (LOWORD(lparm) + character[font_number].width/2) /
          character[font_number].width + corner.column;
        if (column > text.width)
          column = text.width;
        if (row != cursor.row || column != cursor.column)
        {
          POSITION old_cursor = cursor;
          while (cursor.row > row)
            previous_row(cursor);
          while (cursor.row < row)
            next_row(cursor);
          cursor.column = column;
          move_mark(old_cursor);
          scroll_if_necessary();
        }
        return 0;
      }
      break;
    case WM_PAINT:
      if (number_of_files != 0)
      {
        PAINTSTRUCT paint;
        HDC dc = BeginPaint(window, &paint);
        HFONT old_font = SelectObject(dc, font[font_number]);
        POSITION p;
        p.row = corner.row;
        p.column = 0;
        p.position = corner.position;
        int y = 0;
        int c = read_and_advance(p);
        SetTextColor(dc, WHITE);
        SetBkColor(dc, BLACK);
        while (y < 2*display.bottom && c != EOF)
        {
          long count = corner.column;
          while (count-- != 0 && c != '\n' && c != EOF)
            c = read_and_advance(p);
          int x = 0;
          while (c != '\n' && c != EOF)
          {
            char buffer[80];
            int j = 0;
            while (c != '\n' && c != EOF && j < sizeof(buffer))
            {
              buffer[j++] = c;
              c = read_and_advance(p);
            }
            if (y /* + character[font_number].height */ < display.bottom &&
              x < display.right)
            {
              TextOut(dc, x, y, buffer, j);
            }
            x += character[font_number].width * j;
          }
          y += character[font_number].height;
          if (c == '\n') c = read_and_advance(p);
        }
        // apply mark, if any
        if (marked)
        {
          BIGRECT rr;
          marked_rectangle(rr, cursor);
          RECT r;
          clip_rectangle(r, rr);
          InvertRect(dc, &r);
        }
        // draw line between text and status bar
        {
          HPEN old_pen = SelectObject(dc, GetStockObject(BLACK_PEN));
          MoveTo(dc, 0, display.bottom+1);
          LineTo(dc, display.right, display.bottom+1);
          SelectObject(dc, old_pen);
        }
        SelectObject(dc, old_font);
        EndPaint(window, &paint);
        // update cursor and status bar if necessary
        set_cursor();
        // change title bar if necessary
        update_title();
        return 0;
      }
      break;
    case WM_INITMENUPOPUP:
      if (HIWORD(lparm) == 0)
      {
        switch (LOWORD(lparm))
        {
          case 0:
            EnableMenuItem((HMENU) wparm, MENU_OPEN, MF_BYCOMMAND |
                (number_of_files < MAX_FILES ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_CLOSE, MF_BYCOMMAND |
              (number_of_files != 0 ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_CLOSEALL, MF_BYCOMMAND |
              (number_of_files != 0 ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_LIST, MF_BYCOMMAND |
                (number_of_files > 1 ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_NEXT, MF_BYCOMMAND |
                (number_of_files > 1 ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_PREVIOUS, MF_BYCOMMAND |
                (number_of_files > 1 ? MF_ENABLED : MF_GRAYED));
            break;
          case 1:
            EnableMenuItem((HMENU) wparm, MENU_BEGINNING, MF_BYCOMMAND |
              (number_of_files != 0 ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_END, MF_BYCOMMAND |
              (number_of_files != 0 &&
                !(marked && mark.column != cursor.column) &&
                beginning.column != cursor.column ?
                MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_CLIPBOARD, MF_BYCOMMAND |
              (number_of_files != 0 &&
              marked && mark.column != cursor.column ?
              MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_PRINTER, MF_BYCOMMAND |
              (number_of_files != 0 &&
              marked && mark.column != cursor.column ?
              MF_ENABLED : MF_GRAYED));
            break;
          case 2:
            EnableMenuItem((HMENU) wparm, MENU_SEARCH, MF_BYCOMMAND |
              (number_of_files != 0 ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_AGAIN, MF_BYCOMMAND |
              (search_text[0] != 0 ? MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_ZOOMIN, MF_BYCOMMAND |
              (number_of_files != 0 && font_number < BROWSE_FONTS - 1 ?
                MF_ENABLED : MF_GRAYED));
            EnableMenuItem((HMENU) wparm, MENU_ZOOMOUT, MF_BYCOMMAND |
              (number_of_files != 0 && font_number > 0 ? MF_ENABLED :
                MF_GRAYED));
          return 0;
        }
      }
      break;
    case WM_DROPFILES:
    {
      BOOL too_many_files = FALSE;
      int n = DragQueryFile((HANDLE) wparm, 0xFFFF, NULL, 0);
      int i;
      char specs[MAX_FILESPECS+1];
      for (i = 0; i < n; i++)
      {
        DragQueryFile((HANDLE) wparm, i, specs, sizeof(specs));
        if (number_of_files < MAX_FILES)
        {
          file *f = this;
          AnsiUpper(new_filespecs);
          for (int j = 0; j < number_of_files; j++)
          {
            if (strcmp(specs, f->filespecs) == 0)
            {
              if (f != this)
              {
                current = f;
                remake_cursor(f->font_number);
                InvalidateRect(window, NULL, TRUE);
              }
              break;
            }
            f = f->next;
          }
          if (j == number_of_files) file::open(specs);
        }
        else
          too_many_files = TRUE;
      }
      DragFinish((HANDLE) wparm);
      if (number_of_files == MAX_FILES)
        DragAcceptFiles(window, FALSE);
      update_title();
      if (too_many_files)
        error_message("Too many files");
      return 0;
    }
    case WM_COMMAND:
      switch(wparm)
      {
        case MENU_SEARCH:
        {
          DLGPROC proc =
            (DLGPROC) MakeProcInstance((FARPROC) SearchDialogProc,
            application_instance);
          BOOL cancelled = !DialogBox(application_instance, "BROWSE_SEARCH",
            window, proc);
          FreeProcInstance((FARPROC) proc);
          if (cancelled) return 0;
          // else fall through to MENU_AGAIN
        }
        case MENU_AGAIN:
          search_again();
          return 0;
        case MENU_LIST:
        {
          DLGPROC proc =
            (DLGPROC) MakeProcInstance((FARPROC) ListDialogProc,
            application_instance);
          file *f = (file *) DialogBox(application_instance, "BROWSE_LIST",
            window, proc);
          FreeProcInstance((FARPROC) proc);
          if (f != this)
          {
            current = f;
            remake_cursor(f->font_number);
            InvalidateRect(window, NULL, TRUE);
          }
          click_timeout = GetTickCount() + 100;
          return 0;
        }
        case MENU_ZOOMIN:
          font_number++;
          zoom();
          return 0;
        case MENU_ZOOMOUT:
          font_number--;
          zoom();
          return 0;
        case MENU_OPEN:
        {
          if (number_of_files == 0)
          {
            GetWindowsDirectory(new_filespecs, MAX_DIR);
            new_filespecs[3] = 0;
          }
          else
            lstrcpy(new_filespecs, filespecs);
          unsigned n = strlen(new_filespecs);
          while (n > 0 && new_filespecs[n-1] != '\\' &&
            new_filespecs[n-1] != ':')
          {
            n--;
          }
          new_filespecs[n] = 0;
          if (file_dialog(window, "PV Browser", new_filespecs,
            sizeof(new_filespecs), "All Files (*.*)\0*.*\0", "",
            FD_OPEN + FD_MUSTEXIST))
          {
            unsigned i;
            file *f = this;
            AnsiUpper(new_filespecs);
            for (i = 0; i < number_of_files; i++)
            {
              if (strcmp(new_filespecs, f->filespecs) == 0)
              {
                if (f != this)
                {
                  current = f;
                  remake_cursor(f->font_number);
                  InvalidateRect(window, NULL, TRUE);
                }
                return 0;
              }
              f = f->next;
            }
            if (number_of_files == MAX_FILES - 1)
              DragAcceptFiles(window, FALSE);
            timer = SetTimer(window, NULL, 200, NULL);
          }
          else
            new_filespecs[0] = 0;
          return 0;
        }
        case MENU_NEXT:
          current = next;
          remake_cursor(next->font_number);
          InvalidateRect(window, NULL, TRUE);
          return 0;
        case MENU_PREVIOUS:
          current = previous;
          remake_cursor(previous->font_number);
          InvalidateRect(window, NULL, TRUE);
          return 0;
        case MENU_CLOSE:
          file::close();
          if (number_of_files == MAX_FILES - 1)
            DragAcceptFiles(window, TRUE);
          InvalidateRect(window, NULL, TRUE);
          if (number_of_files == 0) update_title();
          return 0;
        case MENU_CLOSEALL:
        {
          int n = number_of_files;
          while (number_of_files != 0) file::close();
          if (n == MAX_FILES) DragAcceptFiles(window, TRUE);
          InvalidateRect(window, NULL, TRUE);
          update_title();
          return 0;
        }
        case MENU_EXIT:
          DestroyWindow(window);
          return 0;
        case MENU_CLIPBOARD:
        {
          HGLOBAL h = marked_text();
          if (h != NULL)
          {
            if (OpenClipboard(window))
            {
              EmptyClipboard();
              SetClipboardData(CF_TEXT, h);
              CloseClipboard();
            }
            else
            {
              error_message("Clipboard access denied");
              GlobalFree(h);
            }
          }
          else
            error_message("Marked block is too large");
          return 0;
        }
        case MENU_PRINTER:
        {
          char info[80];
          char *device = info;
          char *driver = NULL;
          char *port = NULL; 
          GetProfileString("windows", "device", "", info, sizeof(info));
          {
            char *s = info;
            while (*s != 0)
            {
              if (*s == ',')
              {
                *s++ = 0;
                while (*s == ' ')
                  s++;
                if (driver == NULL)
                  driver = s;
                else
                {
                  port = s;
                  break;
                }
              }
              else
                s++;
            }
          }
          HDC dc = CreateDC(driver, device, port, NULL);
          if (dc != NULL)
          {

            long character_height;
            long character_width;
            DOCINFO di;
            HFONT printer_font;
            // get information about default font
            character_height_and_width(dc, character_height, character_width);
            // scale height according to display font
            switch (font_number)
            {
              case 0:
                character_height >>= 2;
                break;
              case 1:
                character_height >>= 1;
                break;
              case 3:
                character_height <<= 1;
                break;
            }
            if (character_height < 4) character_height = 4;
            HFONT old_font =
              SelectObject(dc, construct_font(character_height));
            di.cbSize = sizeof(DOCINFO);
            di.lpszDocName = title_bar;
            di.lpszOutput = NULL;
            character_height_and_width(dc, character_height, character_width);
            StartDoc(dc, &di);
            StartPage(dc);
            POSITION p;
            p.position = mark.position;
            p.row = mark.row;
            p.column = 0;
            long bottom = cursor.row;
            if (bottom < p.row)
            {
              p.position = cursor.position;
              p.row = cursor.row;
              bottom = mark.row;
            }
            long left = mark.column;
            long right = cursor.column;
            if (right < left)
            {
              right = mark.column;
              left = cursor.column;
            }
            int y = 0;
            while (p.row <= bottom)
            {
              char buffer[80];
              int i = 0;
              int x = 0;
              int c = read_and_advance(p);
              while (c != EOF && c != '\n')
              {
                if (left < p.column && p.column <= right)
                {
                  buffer[i++] = c;
                  if (i == sizeof(buffer))
                  {
                    TextOut(dc, x, y, buffer, sizeof(buffer));
                    i = 0;
                    x += character_width * sizeof(buffer);
                  }
                }
                c = read_and_advance(p);
              }
              if (i > 0) TextOut(dc, x, y, buffer, i);
              y += character_height;
              if (c == EOF) break;
            }
            EndPage(dc);
            EndDoc(dc);
            SelectObject(dc, old_font);
            DeleteDC(dc);
          }
          else
            error_message("Printer error");
          return 0;
        }
        case MENU_BEGINNING:
          remove_mark(cursor);
          beginning = cursor;
          return 0;
        case MENU_END:
        {
          BIGRECT new_mark;
          mark = beginning;
          marked = TRUE;
          marked_rectangle(new_mark, cursor);
          invalidate_rectangle(new_mark);
          return 0;
        }
        case MENU_CONTENTS:
          WinHelp(window, "BROWSE.HLP", HELP_INDEX, 0);
          return 0;
        case MENU_HELPSEARCH:
          WinHelp(window, "BROWSE.HLP", HELP_PARTIALKEY,
            (DWORD) (char far *) "");
          return 0;
        case MENU_ABOUT:
          error_message(
            "PV Browser 1.0.\n"
            "Not copyrighted, no restrictions on use.\n"
            "    from\n"
            "Plain Vanilla Corporation\n"
            "P.O. Box 4493\n"
            "San Diego CA 92164\n"
            "June 4, 1994");
          return 0;
      }
      break;
    case WM_TIMER:
      if (new_filespecs[0] != 0)
      {
        file::open(new_filespecs);
        new_filespecs[0] = 0;
      }
      else
      {
        KillTimer(window, timer);
        timer = NULL;
        remake_cursor(font_number);
        InvalidateRect(window, NULL, TRUE);
      }
      return 0;
    case WM_KILLFOCUS:
      DestroyCaret();
      return 0;
    case WM_SETFOCUS:
      if (number_of_files == 0)
      {
        CreateCaret(window, NULL, 2, character[2].height);
        ShowCaret(window);
        SetCaretPos(-character[2].width, -character[2].height);
      }
      else
      {
        //DestroyCaret();
        CreateCaret(window, NULL, 2, character[font_number].height);
        ShowCaret(window);
        set_cursor();
      }
      InvalidateRect(window, NULL, TRUE);
      return 0;
    case WM_CLOSE:
    case WM_DESTROY:
      DragAcceptFiles(window, FALSE);
      DestroyCaret();
      {
        unsigned i;
        for (i = 0; i < BROWSE_FONTS; i++)
          DeleteObject(font[i]);
      }
      delete_all_blocks(NULL);
      if (timer != NULL) KillTimer(window, timer);
      PostQuitMessage(0);
      return 0;
  }
  return DefWindowProc(window, msg, wparm, lparm);
}

/*-----------------------------------------------------------------------------
This function moves the mark from the old position to its current
position, taking care to invalidate only the parts of the client area that
have actually changed in the process.
-----------------------------------------------------------------------------*/

void file::move_mark(POSITION &old_cursor)
{
  BIGRECT new_mark;
  marked_rectangle(new_mark, cursor);
  if (marked)
  {
    BIGRECT old_mark;
    marked_rectangle(old_mark, old_cursor);
    invalidate_difference(old_mark, new_mark);
    invalidate_difference(new_mark, old_mark);
  }
  else
  {
    mark = old_cursor;
    marked = TRUE;
    invalidate_rectangle(new_mark);
  }
}

/*-----------------------------------------------------------------------------
This function starts at the beginning of a row and finds the position of the
beginning of the next following row. If there is no next following row, it
restores the position and returns FALSE;
-----------------------------------------------------------------------------*/

BOOL file::next_row(POSITION &p)
{
  if (read_character(p.position) != EOF)
  {
    int c;
    POSITION pp;
    pp.row = p.row;
    pp.position = p.position;
    pp.column = 0;
    while (TRUE)
    {
      switch (read_and_advance(pp))
      {
        case '\n':
          if (read_character(pp.position) == EOF) return FALSE;
          p.row = pp.row;
          p.position = pp.position;
          return TRUE;
        case EOF:
          return FALSE;
      }
    }
  }
  return FALSE;
}

/*-----------------------------------------------------------------------------
This function starts at the beginning of a row and finds the position of the
beginning of the next preceding row.
-----------------------------------------------------------------------------*/

void file::previous_row(POSITION &p)
{
  if (--p.row == 0)
    p.position = 0;
  else
  {
    p.position--;
    while (read_character(--p.position) != '\n');
    p.position++;
  }
}

/*-----------------------------------------------------------------------------
This function reads a character from a file and then advances the position,
row and column markers to the next following position.
-----------------------------------------------------------------------------*/

int file::read_and_advance(POSITION &position)
{
  int c;
  do c = read_character(position.position++); while (c == '\r');
  if (c == EOF)
  {
    if (position.column != 0 && ++position.row > text.height)
      text.height = position.row;
    position.position--;
    height = text.height;
  }
  else if (c == '\n')
  {
    position.column = 0;
    if (++position.row > text.height)
      text.height = position.row;
  }
  else
  {
    if (++position.column > text.width)
      text.width = position.column;
  }
  return c;
}

/*-----------------------------------------------------------------------------
This function reads a character from a file.
-----------------------------------------------------------------------------*/

int file::read_character(DWORD pos)
{
  block *b = current_block;
  DWORD index = pos - b->position;
  if (index < (DWORD) BLOCK_SIZE)
    return (unsigned) index < b->count ? b->data[(unsigned) index] : EOF;
  b->insert();
  block *bb;
  for (bb = b->older; bb != b; bb = bb->older)
  {
    index = pos - bb->position;
    if (index < (DWORD) BLOCK_SIZE)
    {
      if (!remove_and_lock(bb)) break;
      current_block = bb;
      return (unsigned) index < bb->count ? bb->data[(unsigned) index] : EOF;
    }
  }
  bb = new_block(this, pos & (- (long) BLOCK_SIZE));
  if (bb == NULL) diag("insufficient memory");
  current_block = bb;
  index = pos - bb->position;
  return (unsigned) index < bb->count ? bb->data[(unsigned) index] : EOF;
}

/*-----------------------------------------------------------------------------
This function removes the mark.
-----------------------------------------------------------------------------*/

void file::remove_mark(POSITION &cur)
{
  if (marked)
  {
    BIGRECT rr;
    marked_rectangle(rr, cur);
    invalidate_rectangle(rr);
    marked = FALSE;
  }
}

/*-----------------------------------------------------------------------------
This function scrolls the window.
-----------------------------------------------------------------------------*/

void file::scroll(int dx, int dy)
{
  ScrollWindow(window, dx, dy, &display, &display);
}

/*-----------------------------------------------------------------------------
This function scrolls the window, if necessary, so the cursor will remain
visible.
-----------------------------------------------------------------------------*/

void file::scroll_if_necessary(void)
{
  long count = (int) cursor.row - (int) corner.row;
  if (count < 0)
  {
    if (count > -display.bottom / character[font_number].height)
      scroll(0, -count * character[font_number].height);
    else
      InvalidateRect(window, NULL, TRUE);
    do previous_row(corner); while (++count != 0);
  }
  else
  {
    count -= (display.bottom / character[font_number].height - 1);
    if (count > 0)
    {
      if (count < display.bottom / character[font_number].height)
        scroll(0, -count * character[font_number].height);
      else
        InvalidateRect(window, NULL, TRUE);
      do next_row(corner); while (--count != 0);
    }
  }
  count = (int) cursor.column - (int) corner.column;
  if (count < 0)
  {
    if (count > -display.right / character[font_number].width)
      scroll(-count * character[font_number].width, 0);
    else
      InvalidateRect(window, NULL, TRUE);
    corner.column = cursor.column;
  }
  else
  {
    count -= (display.right / character[font_number].width - 1);
    if (count > 0)
    {
      if (count < display.right / character[font_number].width)
        scroll(-count * character[font_number].width, 0);
      else
        InvalidateRect(window, NULL, TRUE);
      corner.column += count;
    }
  }
  set_cursor();
  set_scroll_bars();
}

/*-----------------------------------------------------------------------------
This function searches for the specified text, starting at the cursor
position, and repositions the cursor to the end of the text if it it is found.
-----------------------------------------------------------------------------*/

void file::search_again(void)
{
  if (search_text[0] != 0)
  {
    POSITION p = cursor;
    long beginning_of_line = cursor.position;
    char buffer[MAX_SEARCH_TEXT];
    int c = read_character(p.position);
    unsigned length = strlen(search_text);
    p.column = 0;
    while (p.row == cursor.row && p.column < cursor.column)
    {
      c = read_and_advance(p);
      if (p.column == 0) beginning_of_line = p.position;
    }
    {
      for (unsigned i = 0; i < length; i++)
      {
        buffer[i] = c = read_and_advance(p);
        if (p.column == 0) beginning_of_line = p.position;
      }
    }
    while (c != EOF)
    {
      if (match_case &&
        memcmp(search_text, buffer, length) == 0 ||
        !match_case &&
        memicmp(search_text, buffer, length) == 0)
      {
        cursor.row = p.row;
        cursor.column = p.column;
        cursor.position = beginning_of_line;
        if (cursor.row < corner.row ||
          cursor.row >= corner.row +
            display.bottom / character[font_number].height ||
          cursor.column < corner.column ||
          cursor.column >= corner.column +
            display.right / character[font_number].width)
        {
          center_cursor();
        }
        else
          set_cursor();
        set_scroll_bars();
        return;
      }
      for (unsigned i = 0; i < length-1; i++)
        buffer[i] = buffer[i+1];
      buffer[i] = c = read_and_advance(p);
      if (p.column == 0) beginning_of_line = p.position;
    }
    error_message("Search string not found");
    set_scroll_bars();
  }
}

/*-----------------------------------------------------------------------------
This function moves the cursor to the position specified in the file
structure.
-----------------------------------------------------------------------------*/

void file::set_cursor(void)
{
  long row = (cursor.row - corner.row) * character[font_number].height;
  long column = (cursor.column - corner.column) * character[font_number].width;
  SetCaretPos(column < 0 ? -character[font_number].width :
    column > display.right ? display.right + 1 : (int) column,
    row < 0 ? -character[font_number].height :
    row > display.bottom - character[font_number].height ?
    display.bottom + 2 + STATUS_HEIGHT : (int) row);
  update_status();
}

/*-----------------------------------------------------------------------------
This function sets the scroll bar runners.
-----------------------------------------------------------------------------*/

void file::set_scroll_bars(void)
{
  SetScrollPos(window, SB_HORZ, text.width > 0 ?
    (corner.column * SCROLL_RANGE) / text.width : 0, TRUE);
  SetScrollPos(window, SB_VERT, text.height > 0 ?
      (corner.row * SCROLL_RANGE) / text.height : 0, TRUE);
}

/*-----------------------------------------------------------------------------
This function updates the status bar if the information on it has changed.
-----------------------------------------------------------------------------*/

void file::update_status(void)
{
  if (cursor.row != display_status.row ||
    cursor.column != display_status.column ||
    height != display_status.height)
  {
    char caption[100];
    DestroyWindow(status_bar);
    wsprintf(caption, "cursor in row %ld, column %ld", cursor.row+1,
      cursor.column+1);
    if (height >= 0)
      wsprintf(caption + strlen(caption), "  ***  height = %ld, width = %ld",
        height, text.width);
    status_bar = CreateWindow("STATIC", caption,
      WS_CHILD | WS_VISIBLE | SS_CENTER,
      0, display.bottom+2, display.right, STATUS_HEIGHT, window,
      -1, application_instance, NULL);
    display_status.row = cursor.row;
    display_status.column = cursor.column;
    display_status.height = height;
  }
}

void file::update_title(void)
{
  if (title != current)
  {
    lstrcpy(title_bar + 12, current->filespecs);
    SetWindowText(window, title_bar);
    title = current;
  }
}

/*-----------------------------------------------------------------------------
This function reconstructs the window after a zoom operation. The window is
positioned so the cursor is as near the center of the screen as possible.
-----------------------------------------------------------------------------*/

void file::zoom(void)
{
  remake_cursor(font_number);
  center_cursor();
}

/*-----------------------------------------------------------------------------
The WinMain function accepts a file specification from the command line. If
there is no file specification on the command line, it prompts the user for
one.
-----------------------------------------------------------------------------*/

int pascal WinMain(HANDLE inst, HANDLE prev_inst, LPSTR comline, int show)
{
  application_instance = inst;
  file::current = file::dummy = new file;
  memset(file::current, 0, sizeof(file));
  lstrcpy(file::current->filespecs, "(no file)");
  file::current->font_number = 2;
  {
    char filespecs[MAX_FILESPECS+1];
    int i = 0;
    int c;
    while ((c = (*comline++ & 0xFF)) != 0)
    {
      if (c != ' ' && c != '\t' && i < MAX_FILESPECS)
        filespecs[i++] = c;
    }
    if (i > 0)
    {
      filespecs[i] = 0;
      AnsiUpper(filespecs);
      file::open(filespecs);
    }
  }
  if (prev_inst == NULL)
  {
    WNDCLASS window;
    window.cbClsExtra = 0;
    window.hbrBackground = GetStockObject(WHITE_BRUSH);
    window.hCursor = LoadCursor(NULL, IDC_ARROW);
    window.hIcon = LoadIcon(inst, "BROWSE_ICON");
    window.hInstance = inst;
    window.lpfnWndProc = MainWindowProc;
    window.lpszClassName = "PVBROWSER";
    window.lpszMenuName = "BROWSE_MENU";
    window.style = CS_HREDRAW | CS_VREDRAW;
    RegisterClass(&window);
  }
  {
    HWND handle = CreateWindow
    (
      "PVBROWSER",               // class name
      title_bar,                 // caption
      WS_OVERLAPPED
      | WS_CAPTION
      | WS_THICKFRAME
      | WS_MINIMIZEBOX
      | WS_MAXIMIZEBOX
      | WS_SYSMENU,
      10,                   // x position
      10,                   // y position
      INITIAL_WIDTH,        // x size
      INITIAL_HEIGHT,       // y size
      NULL,                 // parent window
      NULL,                 // menu
      inst,                 // program instance
      NULL                  // parameters
    );
    ShowWindow(handle, show);
    HANDLE accelerators = LoadAccelerators(inst, "BROWSE_ACCEL");
    MSG message;
    while (GetMessage(&message, 0, 0, 0))
    {
      if (!TranslateAccelerator(handle, accelerators, &message))
      {
        TranslateMessage(&message);
        DispatchMessage(&message);
      }
    }
    return message.wParam;
  }
}
