; general lisp functions 
; Copyright 1994 Apteryx Lisp Ltd

(setq *is-apteryx* (boundp '*apteryx-if-bound*))

(defun get-no-fail (sym prop)
  (let ( (value (get sym prop)) )
    (if (not value)
      (error "Failure to retrieve property" (list sym prop)) )
    value) )

(defmacro pr (name &optional where)
  `(progn
     ,@(if where
       `((format t "~A: " ',where)) )
     (format t "~S = ~S~%" ',name ,name)
     ,name) )

(defmacro push (list el)
  `(setq ,list (cons ,el ,list)) )

(defmacro pop (list)
  `(setq ,list (cdr ,list)) )

(defun flatten (list)
  (let ( (out nil) )
    (dolist (elt list)
      (if (listp elt)
        (setq out (append (reverse (flatten elt)) out))
        (setq out (cons elt out)) ) )
    (reverse out) ) )

; (flatten '(a ((b c)) (d e) (f (g h)) (i) () j k))

(defun quoted (x)
  (list 'quote x))

(defconstant decimal-digits (vector "0" "1" "2" "3" "4" "5" "6" "7" "8" "9"))

(defun ordinal (n)
  (strcat (prin1-to-string n)
    (let ( (n100 (rem n 100)) )
      (if (and (> n100 10) (< n100 20))
        "th"
        (case (rem n 10)
          (1 "st")
          (2 "nd")
          (3 "rd")
          (t "th") ) ) ) ) )
;(ordinal 31)

(defmacro with-open-file (name stream direc &rest exprs)
  `(let ((,stream (open ,name :direction ,direc)))
     (if ,stream
       (unwind-protect
         (progn ,@exprs)
         (close ,stream) )
       (error "Failure to open file" name) ) ) )

(defun print-spaces (n)
  (dotimes (i n) (princ " ")) )

(defun and-fun (&rest args)
  (eval (cons 'and args)) )

(defmacro addf (place increment)
  `(setf ,place (+ ,place ,increment)) )

(defmacro subf (place increment)
  `(setf ,place (- ,place ,increment)) )

(defun lines-of-file (filename)
  (let ( (list nil) )
    (with-open-file filename file :input
      (while (not (eofp file))
        (let ( (line (read-line file)) )
          (if (stringp line)
            (setq list (cons line list)) ) ) ) )
    (reverse list) ) )

;;; sorting

(defun split-list (list)
  (let ( (list1 nil) (list2 nil) (list3 nil))
    (dolist (elt (reverse list))
      (setq list3 (cons elt list1))
      (setq list1 list2)
      (setq list2 list3) )
    (cons list2 list1) ) )

; (split-list '(1 2 3 4 5 6 7 8))

(defun merged (list1 list2 less-than)
  (let ( (result nil) (rem-list1 list1) (rem-list2 list2) next-elt)
    (while (or rem-list1 rem-list2)
      (if (or (null rem-list2)
            (and rem-list1
              (funcall less-than (car rem-list1) (car rem-list2)) ) )
        (progn
          (setq next-elt (car rem-list1))
          (setq rem-list1 (cdr rem-list1)) )
        (progn
          (setq next-elt (car rem-list2))
          (setq rem-list2 (cdr rem-list2)) ) )
      (setq result (cons next-elt result)) )
    (reverse result) ) )

(merged '(1 3 5) '(2 6 8) #'<)

(defun merge-sort (list less-than)
  (if (<= (length list) 1)
    list
    (let* ( (halves (split-list list))
            (list1-sorted (merge-sort (car halves) less-than))
            (list2-sorted (merge-sort (cdr halves) less-than)) )
      (merged list1-sorted list2-sorted less-than) ) ) )

(defun sort (list less-than)
  (merge-sort list less-than) )
      
; (sort '(5 7 1 5 10 20 300 -5 71 3 8 9) #'<)

; fake progv - main difference is that there is no
; distinction between dynamic and global value.

(defmacro progv (symbols values &rest stmts)
  (let ( (unbound-value (gensym))
         (symbols2 (gensym))
         (rest-values (gensym))
         (old-value (gensym))
         (old-values (gensym))
         (result (gensym)) )
    `(let* ( (,unbound-value (gensym))
             (,symbols2 ,symbols)
             (,rest-values ,values)
             ,old-value ,result
             (,old-values (mapcar #'(lambda (sym)
                                      (if (boundp sym)
                                        (symbol-value sym)
                                        ,unbound-value) )
                            ,symbols2) ) )
       (unwind-protect
         (progn
           (dolist (sym ,symbols2)
             (if ,rest-values
               (progn
                 (set sym (car ,rest-values))
                 (setq ,rest-values  (cdr ,rest-values)) )
               (makunbound sym) ) )
           (setq ,result (progn ,@stmts)) )
         (dolist (sym ,symbols2)
           (setq ,old-value (car ,old-values))
           (setq ,old-values (cdr ,old-values))
           (if (eq ,old-value ,unbound-value)
             (makunbound sym)
             (set sym ,old-value) ) )
         ,result) ) ) )

;(defun show-a () (format t "a = ~A~%" a))
;(defun show-b () (format t "b = ~A~%" b))
;(setq a 10) (setq b 11)
;(let ( (a 32) )
;  (progv '(a b) '(39 34)
;    (show-a) (show-b) )
;  )
;(show-a)

;;; A simple trace facility

; Usage   (trace fun1 fun2)   trace functions
;         (trace)             see list of traced functions
;         (untrace fun1 fun2) stop tracing functions
;         (untrace)           untrace all traced functions


;(defun square (x) (* x x))

;(square 100)

;(trace square) (untrace square) (trace) (untrace)

(setq *traced-functions* nil)

(defun trace1 (fun)
  (if (not (get fun 'original-function))
    (let ( (fun-value (symbol-function fun)) )
      (setf (symbol-function fun)
        #'(lambda (&rest args)
            (format t "Applying fun #'~A to args ~A~%" fun args)
            (let ( (result (apply fun-value args)) )
              (format t "Applied fun #'~A to args ~A => ~A~%"
                fun args result)
              result) ) )
      (setf (get fun 'original-function) fun-value)
      (setq *traced-functions* (cons fun *traced-functions*))
      fun) ) )

(defun untrace1 (fun)
  (let ( (original-fun (get fun 'original-function)) )
    (when original-fun
      (setf (symbol-function fun) original-fun)
      (remprop fun 'original-function)
      (setq *traced-functions* (remove fun *traced-functions*)) ) )
  fun)

(defmacro trace (&rest funs)
  `(if ',funs
     (mapcar #'trace1 ',funs)
     (format t "Traced functions: ~A~%" *traced-functions*) ) )

(defmacro untrace (&rest funs)
  `(if ',funs
     (mapcar #'untrace1 ',funs)
     (mapcar #'untrace1 *traced-functions*) ) )

;;; position


(defun position (ob list &key (test #'eql))
  (let ( (rest list) (found nil) (pos 0))
    (while (and (not found) (consp rest) )
      (if (funcall test (car rest) ob)
        (setq found t)
        (progn
          (setq rest (cdr rest))
          (setq pos (1+ pos)) ) ) )
    (if found
      pos
      nil) ) )  

(defun mapcan (fun list1 &rest lists)
  (apply #'nconc (apply #' mapcar (cons fun (cons list1 lists)))) )

;;; setf macros

(defmacro appendf (place &rest lists)
  `(setf ,place (append ,place ,@lists)) )

(defmacro incf (place)
  `(setf ,place (1+ ,place)) )

(defmacro addf (place n)
  `(setf ,place (+ ,place ,n)) )

(defmacro decf (place)
  `(setf ,place (1- ,place)) )

(defmacro subf (place n)
  `(setf ,place (- ,place ,n)) )

(defmacro pushf (place x)
  `(setf ,place (cons ,x ,place)) )


