#include <windows.h>
#include <stdlib.h>
#include <string.h>
#include <mem.h>

#include "generic.h"  
#include "otapi.h"
#include "sampobj.h"
#include "ids.h"

BOOL CALLBACK _export MainWindowInspectProc( void *self, SInspectStruct *pstSInspect );

struct stObjListNode
{
   CSampleObj *pObj;
   stObjListNode  *pNext;
};

enum enToolState
{
   eIdle, eDown, eUp, eDrag,
};

void PaintObjects( HDC hDC );
BOOL AppendObj( CSampleObj *pObj );
CSampleObj *FindObj( POINT &pt );

stObjListNode *pObjList = NULL;
enToolState stateObjTool = eIdle;
CSampleObj *pSampleObj = NULL;

HINSTANCE hInst;        
HWND hwndMain;
COLORREF crBkgnd = GetSysColor( COLOR_WINDOW );

//------------------------------------------------------------------------------
int PASCAL WinMain( HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR, int iCmdShow )
{
   MSG msg;           

   if( !hPrevInstance && !InitApplication( hInstance ) )
   {
      return FALSE;  
   }
   if( !InitInstance( hInstance, iCmdShow ) )
   {
      return FALSE;
   }

   while( GetMessage( &msg, NULL, NULL, NULL ) )
   {
      //!!
      //!! MUST call IsInspectMessage() so dlg mgr will handle inspector msgs
      //!!
      if( !OTIsInspectMessage( &msg ) )
      {
         TranslateMessage( &msg );
         DispatchMessage( &msg ); 
      }
   }
   return msg.wParam;
}


//------------------------------------------------------------------------------
BOOL InitApplication( HINSTANCE hInstance )
{
   WNDCLASS  wc;

   wc.style = NULL;                   
   wc.lpfnWndProc = (WNDPROC)MainWndProc;
                                      
   wc.cbClsExtra = 0;                 
   wc.cbWndExtra = 0;                 
   wc.hInstance = hInstance;          
   wc.hIcon = LoadIcon( hInstance, MAKEINTRESOURCE(ICN_APP) );
   wc.hCursor = NULL;
   wc.hbrBackground = GetStockObject( WHITE_BRUSH ); 
   wc.lpszMenuName =  "GenericMenu";  
   wc.lpszClassName = "ObjTweakerSample";

   return RegisterClass( &wc );
}


//------------------------------------------------------------------------------
BOOL InitInstance( HINSTANCE hInstance, int iCmdShow )
{
   hInst = hInstance;

   hwndMain = CreateWindow( "ObjTweakerSample",
                            "Object Tweaker Sample Application", 
                            WS_OVERLAPPEDWINDOW,          
                            CW_USEDEFAULT, CW_USEDEFAULT,                
                            CW_USEDEFAULT, CW_USEDEFAULT,                
                            NULL, NULL, hInstance, NULL );

   if( !IsWindow( hwndMain ) )
   {
      return FALSE;
   }

   ShowWindow( hwndMain, iCmdShow );
   UpdateWindow( hwndMain );

   return TRUE;
}

//------------------------------------------------------------------------------
LONG FAR PASCAL _export MainWndProc( HWND hwnd, UINT iMessage, WPARAM wParam, LPARAM lParam )
{
   char szBuf[80];
   switch( iMessage )
   {
      case WM_COMMAND:   
         switch( wParam )
         {
            case IDM_ABOUT:
               DialogBox( hInst, "AboutBox", hwnd, (DLGPROC)About );
               return 0L;
               
            case IDM_CREATEOBJ:
            {
               RECT rcClient;
               GetClientRect( hwnd, &rcClient );
               MapWindowPoints( hwnd, HWND_DESKTOP, (POINT *)&rcClient, 2 );

               ClipCursor( &rcClient );
               SetCursor( LoadCursor( NULL, IDC_CROSS ) );

               stateObjTool = eUp;

               break;
            }
            
            case IDM_TESTINSPECT:
            {
               // Object for inspector
               static SInspectObjList stSampleObjs;
               stSampleObjs.pObj = (void *)pSampleObj;
               stSampleObjs.next = NULL;
            
               if( pSampleObj )
               {
                  // Inspect the SELECTED OBJECT.  Inspector will use CTL3D for
                  // 3D effects and use a 3D, hierarchical list.
                  OTInspect( CSampleObj::SInspectProc, &stSampleObjs, PPF_CTL3D );
               }
               else
               {
                  // Inspect the MAIN WINDOW.  Inspector will use CTL3D for 3D 
                  // effects and use Chicago style tabbed sheets.
                  OTInspect( MainWindowInspectProc, &stSampleObjs, PPF_TABS | PPF_CTL3D );
               }
               break;
            }
         }
         break;

      case WM_LBUTTONDOWN:
         switch( stateObjTool )
         {
            case eUp:
            {
               ClipCursor( NULL );
               SetCursor( LoadCursor( NULL, IDC_ARROW ) );

               stateObjTool = eIdle;

               RECT rcPos = { LOWORD(lParam), HIWORD(lParam), 
                              LOWORD(lParam)+150, HIWORD(lParam)+100 };
               if( pSampleObj )
               {
                  // Remove previous selection
                  pSampleObj->Invalidate();
               }
                              
               pSampleObj = new CSampleObj( hwnd, &rcPos );
               pSampleObj->Invalidate();
               AppendObj( pSampleObj );
               
               // Replace current inspect object with this one (only if
               // the inspector is currently open).
               static SInspectObjList stSampleObjs;
               stSampleObjs.pObj = (void *)pSampleObj;
               stSampleObjs.next = NULL;
            
               OTInspect( CSampleObj::SInspectProc, &stSampleObjs, PPF_CTL3D, FALSE );               

               break;
            }

            case eIdle:
            {
               static SInspectObjList stSampleObjs;

               if( pSampleObj )
               {
                  // Remove previous selection
                  pSampleObj->Invalidate();
               }

               POINT pt = MAKEPOINT( lParam );
               CSampleObj *pObj = FindObj( pt );

               if( pObj )
               {               
               
                  pSampleObj = pObj;
                  pSampleObj->Invalidate();
                  
                  // Replace current inspect object with this one (only if
                  // the inspector is currently open).
                  stSampleObjs.pObj = (void *)pSampleObj;
                  stSampleObjs.next = NULL;

                  OTInspect( CSampleObj::SInspectProc, &stSampleObjs, PPF_CTL3D, FALSE );                                 
               }
               else
               {
                  pSampleObj = NULL;

                  // Inspect the Main Window.
                  stSampleObjs.pObj = NULL;
                  stSampleObjs.next = NULL;
               
                  OTInspect( MainWindowInspectProc, &stSampleObjs, PPF_CTL3D, FALSE );                                 
               }
               break;
            }
         }
         break;

      case WM_MOUSEMOVE:
      {
         if( stateObjTool == eUp )
         {
            SetCursor( LoadCursor( NULL, IDC_CROSS ) );
         }
         else
         {
            SetCursor( LoadCursor( NULL, IDC_ARROW ) ); 
         }
         return 0L;
      }
      
      case WM_PAINT:   
      {
         PAINTSTRUCT stPaintStruct;
         HDC hDC = BeginPaint( hwnd, &stPaintStruct );
         
         PaintObjects( hDC );
         if( pSampleObj )
         {
            RECT rcExtent;
            pSampleObj->GetExtent( rcExtent );
            
            RECT rcHandle = { rcExtent.left, rcExtent.top,
                              rcExtent.left+10, rcExtent.top+10 };
            InvertRect( hDC, &rcHandle );
            SetRect( &rcHandle, rcExtent.left, rcExtent.bottom-10,
                                rcExtent.left+10, rcExtent.bottom );
            InvertRect( hDC, &rcHandle );
            SetRect( &rcHandle, rcExtent.right-10, rcExtent.top,
                                rcExtent.right, rcExtent.top+10 );
            InvertRect( hDC, &rcHandle );
            SetRect( &rcHandle, rcExtent.right-10, rcExtent.bottom-10,
                                rcExtent.right, rcExtent.bottom );
            InvertRect( hDC, &rcHandle );            
         }
         
         
         EndPaint( hwnd, &stPaintStruct );

         return 0L;
      }

      case WM_ERASEBKGND:
      {
         //
         // Erase the background using brush with bkgnd color.
         //
         HBRUSH hbrBkgnd = CreateSolidBrush( crBkgnd );
         RECT rcBk;
         GetClientRect( hwnd, &rcBk );
         FillRect( (HDC)wParam, &rcBk, hbrBkgnd );
         DeleteObject( hbrBkgnd );

         return 1L;
      }

      case WM_DESTROY:
         PostQuitMessage( 0 );
         return 0L;
    }
    return DefWindowProc( hwnd, iMessage, wParam, lParam );
}


//------------------------------------------------------------------------------
BOOL FAR PASCAL _export About( HWND hDlg, UINT iMessage, WPARAM wParam, LPARAM )
{
   switch( iMessage )
   {
      case WM_INITDIALOG:
         return TRUE;

      case WM_CTLCOLOR:
         SetTextColor( (HDC)wParam, RGB(0,0,128) );
         SetBkColor( (HDC)wParam, RGB(192,192,192) );         
         return (HBRUSH)GetStockObject( LTGRAY_BRUSH );
         
      case WM_COMMAND:           
         if( wParam == IDOK || wParam == IDCANCEL ) 
         {
            EndDialog( hDlg, TRUE );
            return TRUE;
         }
         break;
   }
   return FALSE;
}

//------------------------------------------------------------------------------
//
// Append an object to the circular linked list.
//
BOOL AppendObj( CSampleObj *pObj )
{
   if( !pObj )
   {
      return FALSE;
   }

   stObjListNode *pNode = new stObjListNode;
   pNode->pObj = pObj;

   if( !pObjList )
   {
      pNode->pNext = pNode;
      pObjList = pNode;
   }
   else
   {
      pNode->pNext = pObjList->pNext;
      pObjList->pNext = pNode;
      pObjList = pNode;
   }   

   return TRUE;
}

//------------------------------------------------------------------------------
//
// Return the object containing the specified point.
//
CSampleObj *FindObj( POINT &pt )
{
   stObjListNode *pCsr = pObjList;

   if( pCsr )
   {
      RECT rcExtent;
      do
      {
         pCsr = pCsr->pNext;

         pCsr->pObj->GetExtent( rcExtent );
         if( PtInRect( &rcExtent, pt ) )
         {
            return pCsr->pObj;
         }
      }while( pCsr != pObjList );
   }
   return NULL;
}

//------------------------------------------------------------------------------
//
// Paint each object in the list.
//
void PaintObjects( HDC hDC )
{
   stObjListNode *pCsr = pObjList;

   if( pCsr )
   {
      RECT rcExtent;
      do
      {
         pCsr = pCsr->pNext;
         pCsr->pObj->Paint( hDC );
      }while( pCsr != pObjList );
   }
}

//------------------------------------------------------------------------------
//------------------------------------------------------------------------------
//
// This function represents an interface to the property inspector for Main Window.
//
//------------------------------------------------------------------------------
BOOL CALLBACK _export MainWindowInspectProc( void *, SInspectStruct *pstSInspect )
{
   BOOL bRet = TRUE;

   switch( pstSInspect->uMessage )
   {
      //
      // Set a name string representing this object
      //
      case PI_GETOBJECTNAME:
      {
         strcpy( (char *)pstSInspect->pValue, STR_MAINWINDOW );
      }

      //
      // Supply the object's property info
      //
      case PI_GETPROPERTIES:
      {
         static stPropInfo pstTestProps[MAX_PROPERTIES];

         // Properties info
         stPropInfo pstProps[] =
         {
            { PID_LABELTEXT,   PROPID_NAME,      ID_PROP_LABELTEXT, -1, hInst, NULL,  NULL },
            { PID_COLOR_BACK,  PROPID_COLORREF,  ID_PROP_COLORS,    -1, hInst, NULL,  NULL },                  
         };
         
         memset( pstTestProps, 0, sizeof(pstTestProps) );
         memcpy( pstTestProps, pstProps, sizeof(pstProps) );
         
         // Number of properties
         pstSInspect->uProperty = sizeof(pstProps) / sizeof(stPropInfo);

         // Property info
         pstSInspect->pValue = (void *)pstTestProps;
         
         break;
      }


      //
      // Supply a specified property value
      //
      case PI_GETVALUE:
      {
         switch( pstSInspect->uProperty )
         {
            // Window Text
            case PID_LABELTEXT:
            {
               static char szTitle[MAX_STR_TEXT];
               GetWindowText( hwndMain, szTitle, sizeof( szTitle ) );
               pstSInspect->pszValue = szTitle;
               break;
            }
               
            // Background Color
            case PID_COLOR_BACK:
               pstSInspect->dwValue = (DWORD)crBkgnd;
               break;

            default:
               bRet = FALSE;
         }
         break;
      }

      //
      // Set a specified property value
      //
      case PI_SETVALUE:
      {
         switch( pstSInspect->uProperty )
         {
            // Window Text
            case PID_LABELTEXT:
            {
               SetWindowText( hwndMain, pstSInspect->pszValue );
               break;
            }

            // Background Color
            case PID_COLOR_BACK:
               crBkgnd = (COLORREF)pstSInspect->dwValue;
               InvalidateRect( hwndMain, NULL, TRUE );
               break;

            default:
               bRet = FALSE;
         }
         break;
      }

      default:
         bRet = FALSE;

   }
   return bRet;
}
