/*
 *                          The AVRIL System
 *
 *  AVRIL stands for A Virtual Reality Interface Library.  It's a library
 *  of routines for doing fast, polygon-based rendering on a variety of
 *  platforms.  It will also provide support for device i/o, simulation,
 *  user interface, interaction detection and more.
 *
 *  It's designed to be easy to use, portable, and fast.
 *
 */

/* Copyright 1994 by Bernie Roehl */

/* You may use this code for your own non-commercial projects without
   paying any fees or royalties.  "Non-commercial", in this context,
   means that the software you write is given away for free to anyone
   who wants it.
   
   Commercial use, including shareware, requires a licensing
   fee and a specific written agreement with the author.

   All programs created using this software (both commercial and
   non-commercial) must acknowledge the use of the AVRIL library,
   both in the documentation and in a banner screen at the start or
   end of the program.

   For more information, contact Bernie Roehl (broehl@uwaterloo.ca).

*/

/* This is version 0.9b -- April, 1994 */

/* Three data types are used in AVRIL:

        Scalar  -- a measure of virtual distance
        Factor  -- a multiplication factor, usually in the range -1 to +1
        Angle   -- measured in 65536th's of a degree

   In the floating-point version of the code, they're all floats.  In
   the fixed-point version, they're all stored in a 32-bit word; they
   use differing numbers of bits for the integer and fractional parts of
   their values.
 */

#define VRL_USE_FIXED_POINT 1

#define VRL_VERSION 1

#ifdef VRL_USE_FIXED_POINT
typedef long Scalar;  /* 32.0 */
typedef long Factor;  /* 3.29 */
#define UNITY 536870912L  /* 2**29 */
typedef long Angle;   /* 16.16 */
#define ANGLECONVERSION 65536  /* 2**16 */
Scalar vrl_ScalarDivide(Scalar a, Scalar b);
Scalar vrl_ScalarMultDiv(Scalar a, Scalar b, Scalar c);
Scalar vrl_FactorMultiply(Scalar a, Factor b);
#define vrl_ScalarRound(a) (a)

#else  /* floating-point */
typedef float Scalar;
typedef float Factor;
#define UNITY 1.0
typedef float Angle;
#define ANGLECONVERSION 1
#define vrl_ScalarDivide(a, b) ((a) / (b))
#define vrl_ScalarMultDiv(a, b, c) (((a) * (b)) / (c))
#define vrl_FactorMultiply(a, b) ((a) * (b))
Scalar vrl_ScalarRound(Scalar a);
#endif

/* Conversion routines.  You should always use these, to ensure your code
   will work in both the floating-point and fixed-point implementations.
 */

#define float2scalar(f) ((Scalar) (f))
#define scalar2float(s) ((float) (s))
#define float2factor(f) ((Factor) ((f) * UNITY))
#define factor2float(a) (((float) (a)) / UNITY)
#define float2angle(f) ((Angle) ((f) * ANGLECONVERSION))
#define angle2float(a) (((float) (a)) / ANGLECONVERSION)

/* Math functions */

void vrl_Mathinit(void);     /* initializes math routines */
Factor vrl_Sine(Angle angle);
Factor vrl_Cosine(Angle angle);

/* Vector functions */

typedef Scalar Vector[3];

#define X 0
#define Y 1
#define Z 2

void vrl_VectorCreate(Vector result, Scalar x, Scalar y, Scalar z);
void vrl_VectorCopy(Vector destination, Vector source);
void vrl_VectorAdd(Vector result, Vector v1, Vector v2);
void vrl_VectorSub(Vector result, Vector v1, Vector v2);
Factor vrl_VectorDotproduct(Vector v1, Vector v2);
Scalar vrl_VectorCrossproduct(Vector result, Vector v1, Vector v2);
Scalar vrl_VectorMagnitude(Vector v);
void vrl_VectorNormalize(Vector v);

/* note that normalized vectors actually have Factors as elements */

/* Matrix functions */

typedef Scalar Matrix[4][3];

/* The rotational elements of a matrix are actually Factors, not Scalars;
   only the translational elements are Scalars. */

void vrl_MatrixIdentity(Matrix m);
void vrl_MatrixCopy(Matrix result, Matrix m);
void vrl_MatrixMultiply(Matrix result, Matrix m1, Matrix m2);
void vrl_MatrixInverse(Matrix result, Matrix m);
void vrl_MatrixRotX(Matrix m, Angle angle);
void vrl_MatrixRotY(Matrix m, Angle angle);
void vrl_MatrixRotZ(Matrix m, Angle angle);
void vrl_MatrixRotVector(Matrix m, Angle angle, Vector vector);
void vrl_MatrixResetRotations(Matrix m);
void vrl_MatrixGetBasis(Vector v, Matrix m, int axis);
void vrl_MatrixTranslate(Matrix result, Scalar x, Scalar y, Scalar z);
void vrl_MatrixSetTranslation(Matrix result, Scalar x, Scalar y, Scalar z);

#define vrl_MatrixGetTranslation(v, m) vrl_VectorCopy((v), (m)[3])

/* Transformation functions */

void vrl_Transform(Vector result, Matrix m, Vector v);
Scalar vrl_TransformX(Matrix m, Vector v);
Scalar vrl_TransformY(Matrix m, Vector v);
Scalar vrl_TransformZ(Matrix m, Vector v);

/* Data types */

typedef struct _vrl_world vrl_World;
typedef struct _vrl_object vrl_Object;
typedef struct _vrl_shape vrl_Shape;
typedef struct _vrl_rep vrl_Rep;
typedef struct _vrl_facet vrl_Facet;
typedef struct _vrl_surface vrl_Surface;
typedef struct _vrl_light vrl_Light;
typedef struct _vrl_camera vrl_Camera;
typedef struct { unsigned char red, green, blue; } vrl_Color;
typedef vrl_Color vrl_Palette[256];
typedef struct _vrl_listnode vrl_List;

/* Worlds */

struct _vrl_world
	{
	vrl_Object *objects;      /* tree of objects */
	vrl_Light *lights;        /* linked list of lights */
	vrl_Camera *camera;       /* current camera */
	Factor ambient;           /* ambient light level */
	Scalar scale;             /* millimeters per unit of virtual space */
	int screenclear : 1;      /* set is we should clear the screen */
	int horizon : 1;          /* set if we should draw a horizon   */
	int set_palette: 1;       /* set if this world has a new palette */
	int movement_mode : 1;    /* non-zero if we can fly by looking up */
	Scalar movestep;          /* default movement step size */
	Angle rotstep;            /* default rotation step size */
	long horizoncolors[10];   /* entry 0 is ground, entry n is sky */
	int nhorizoncolors;       /* number of colors used in horizoncolors[] */
	vrl_Palette palette;      /* the palette, if one is used */
	vrl_List *named_objects;  /* objects that have been assigned names */
	vrl_List *named_lights;   /* lights that have been assigned names */
	vrl_List *named_cameras;  /* cameras that have been assigned names */
	};

vrl_World *vrl_WorldInit(vrl_World *world);
void vrl_WorldAddLight(vrl_Light *light);
void vrl_WorldRemoveLight(vrl_Light *light);
void vrl_WorldAddObject(vrl_Object *obj);
void vrl_WorldRemoveObject(vrl_Object *obj);
int vrl_WorldCountObjects(void);
int vrl_WorldCountLights(void);

#define vrl_NewWorld() vrl_WorldInit(malloc(sizeof(vrl_World)))
#define vrl_WorldSetScreenClear(n) (vrl_current_world->screenclear = (n) ? 1 : 0)
#define vrl_WorldGetScreenClear() (vrl_current_world->screenclear)
#define vrl_WorldToggleScreenClear() (vrl_current_world->screenclear = !vrl_current_world->screenclear)
#define vrl_WorldSetHorizon(n) (vrl_current_world->horizon = (n) ? 1 : 0)
#define vrl_WorldGetHorizon() (vrl_current_world->horizon)
#define vrl_WorldToggleHorizon() (vrl_current_world->horizon = !vrl_current_world->horizon)
#define vrl_WorldSetMovementMode(n) (vrl_current_world->movement_mode = (n) ? 1 : 0)
#define vrl_WorldGetMovementMode() (vrl_current_world->movement_mode)
#define vrl_WorldToggleMovementMode() (vrl_current_world->movement_mode = !vrl_current_world->movement_mode)
#define vrl_WorldSetMovestep(distance) (vrl_current_world->movestep = (distance))
#define vrl_WorldGetMovestep() (vrl_current_world->movestep)
#define vrl_WorldSetTurnstep(angle) (vrl_current_world->rotstep = (angle))
#define vrl_WorldGetTurnstep() (vrl_current_world->rotstep)
#define vrl_WorldSetScale(s) (vrl_current_world->scale = (s))
#define vrl_WorldGetScale() (vrl_current_world->scale)
#define vrl_WorldSetAmbient(amb) (vrl_current_world->ambient = (amb))
#define vrl_WorldGetAmbient() (vrl_current_world->ambient)
#define vrl_WorldSetGroundColor(color) (vrl_current_world->horizoncolors[0] = (color))
#define vrl_WorldGetGroundColor() (vrl_current_world->horizoncolors[0])
#define vrl_WorldSetSkyColor(color) (vrl_current_world->horizoncolors[vrl_current_world->nhorizoncolors-1] = (color))
#define vrl_WorldGetSkyColor() (vrl_current_world->horizoncolors[vrl_current_world->nhorizoncolors-1])
#define vrl_WorldGetPalette() (vrl_current_world->palette)
#define vrl_WorldUsePalette() (vrl_current_world->set_palette = 1)
#define vrl_WorldGetLights() (vrl_current_world->lights)
#define vrl_WorldGetObjectTree() (vrl_current_world->objects)
#define vrl_WorldSetCamera(cam) (vrl_current_world->camera = (cam))
#define vrl_WorldGetCamera() (vrl_current_world->camera)

extern vrl_World *vrl_current_world;   /* the currently active world */

#define vrl_WorldSetCurrent(world) (vrl_current_world = (world))
#define vrl_WorldGetCurrent() (vrl_current_world)

/* Objects */

struct _vrl_object
	{
	vrl_Shape *shape;           /* geometry information */
	vrl_Surface **surfmap;      /* array of pointers to surface descriptors */
	Matrix localmat;            /* transformation matrix relative to our parent */
	Matrix globalmat;           /* transformation matrix relative to the world */
	vrl_Object *parent;         /* pointer to our parent in the hierarchy */
	vrl_Object *children;       /* pointer to our children */
	vrl_Object *siblings;       /* pointers to our siblings */
	vrl_World *world;           /* the world this object is in, if any */
	unsigned char layer;        /* the layer we're on (0 for all, 1-255) */
	int highlight : 1;          /* set if object is highlighted */
	int invisible : 1;          /* set if object is invisible */
	int fixed : 1;              /* set if object is immobile */
	int moved : 1;              /* set when our local matrix has changed */
	int rotate_box : 1;         /* set if bounding box should rotate */
	Vector minbound, maxbound;  /* bounding box (world coords) */
	vrl_Object *contents;       /* points to objects contained by this one (not used) */
	vrl_Object *next;           /* points to next object on a list */
	vrl_Rep *forced_rep;        /* if not NULL, forces a rep to be used */
	};

vrl_Object *vrl_ObjectInit(vrl_Object *obj);
vrl_Object *vrl_NewObject(vrl_Shape *shape);
void vrl_ObjectMove(vrl_Object *obj, Scalar x, Scalar y, Scalar z);
void vrl_ObjectRelMove(vrl_Object *obj, Scalar x, Scalar y, Scalar z);
void vrl_ObjectRotX(vrl_Object *obj, Angle angle);
void vrl_ObjectRotY(vrl_Object *obj, Angle angle);
void vrl_ObjectRotZ(vrl_Object *obj, Angle angle);
void vrl_ObjectRotVector(vrl_Object *obj, Angle angle, Vector vector);
void vrl_ObjectRotReset(vrl_Object *obj);
vrl_Object *vrl_ObjectAttach(vrl_Object *obj, vrl_Object *newparent);
vrl_Object *vrl_ObjectDetach(vrl_Object *obj);
vrl_Object *vrl_ObjectUpdate(vrl_Object *object);
void vrl_ObjectTraverse(vrl_Object *object, int (*function)(vrl_Object *obj));
void vrl_ObjectMakeFixed(vrl_Object *object);
void vrl_ObjectMakeMovable(vrl_Object *object);

#define vrl_ObjectSetLayer(object, lay) ((obj)->layer = (lay))
#define vrl_ObjectGetLayer(object) ((object)->layer)
#define vrl_ObjectSetShape(object, shp) ((object)->shape = (shp))
#define vrl_ObjectGetShape(object) ((object)->shape)
#define vrl_ObjectSetSurfacemap(object, map) ((object)->surfmap = (map))
#define vrl_ObjectGetSurfacemap(object) ((object)->surfmap)
#define vrl_ObjectSetHighlight(object, high) ((object)->highlight = (high))
#define vrl_ObjectGetHighlight(object) ((object)->highlight)
#define vrl_ObjectToggleHighlight(object) ((object)->highlight = !(object)->highlight)
#define vrl_ObjectSetVisibility(object, vis) ((object)->invisible = ((vis) ? 1 : 0))
#define vrl_ObjectGetVisibility(object) ((object)->invisible)
#define vrl_ObjectToggleVisibility(object) ((object)->invisible = !(object)->invisible)
#define vrl_ObjectIsFixed(object) ((object)->fixed)
#define vrl_ObjectGetMinbounds(object) (&((object)->minbound))
#define vrl_ObjectGetMaxbounds(object) (&((object)->maxbound))
#define vrl_ObjectSetRep(object, r) ((object)->forced_rep = (r))
#define vrl_ObjectGetRep(object) ((object)->forced_rep)
#define vrl_ObjectGetX(object) ((object)->globalmat[3][X])
#define vrl_ObjectGetY(object) ((object)->globalmat[3][Y])
#define vrl_ObjectGetZ(object) ((object)->globalmat[3][Z])
#define vrl_ObjectGetForwardVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, Z)
#define vrl_ObjectGetRightVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, X)
#define vrl_ObjectGetUpVector(object, v) vrl_MatrixGetBasis((v), (object)->globalmat, Y)
#define vrl_ObjectGetLocation(object, v) vrl_VectorCopy((v), ((object)->globalmat[3])

/* Shapes */

struct _vrl_shape
	{
	Vector center;  /* center of bounding sphere */
	Scalar radius;  /* radius of bounding sphere */
	Vector minbound, maxbound;  /* bounding box */
	vrl_Surface **default_surfacemap;  /* default surface map for this shape */
	vrl_Rep *replist;  /* linked list of representations */
	};

void vrl_ShapeComputeBounds(vrl_Shape *shape);
void vrl_ShapeUpdate(vrl_Shape *shape);
void vrl_ShapeTransform(Matrix mat, vrl_Shape *shape);
vrl_Rep *vrl_ShapeGetRep(vrl_Shape *shape, Scalar size);
void vrl_ShapeTraverseReps(vrl_Shape *shape, int (*function(vrl_Rep *rep)));

#define vrl_ShapeGetRadius(shape) ((shape)->radius)
#define vrl_ShapeGetCenter(shape) (&((shape)->center))
#define vrl_ShapeGetMinbounds(shape) (&((shape)->minbound))
#define vrl_ShapeGetMaxbounds(shape) (&((shape)->maxbound))
#define vrl_ShapeGetSurfacemap(shape) ((shape)->default_surfacemap)

/* Representations */

typedef enum
		{
		SORT_NONE = 0, SORT_FARTHEST, SORT_NEAREST, SORT_AVERAGE, SORT_OTHER
		} vrl_SortingType;

struct _vrl_rep
	{
	Scalar size;        /* (radius/distance) value at which to use this rep */
	vrl_Rep *next;      /* next less-detailed rep */
	vrl_SortingType sorttype;  /* type of poly sorting to do on this object */
	int nvertices;      /* number of vertices (and normals if present) */
	Vector *vertices;   /* array of vertices */
	Vector *normals;    /* array of vertex normals; can be NULL */
	vrl_Facet *facets;  /* facets are kept in a linked list */
	};

void vrl_RepTraverseVertices(vrl_Rep *rep, int (*function)(Vector *vertex));
void vrl_RepTraverseFacets(vrl_Rep *rep, int (*function)(vrl_Facet *facet));
vrl_Facet *vrl_RepGetFacet(vrl_Rep *rep, int n);
int vrl_RepCountFacets(vrl_Rep *rep);

#define vrl_RepSetSorting(rep, type) ((rep)->sorttype = (type))
#define vrl_RepGetSorting(rep) ((rep)->sorttype)
#define vrl_RepGetSize(rep) ((rep)->size)
#define vrl_RepCountVertices(rep) ((rep)->nvertices)
#define vrl_RepGetVertex(rep, n) (&((rep)->vertices[n]))

/* Facets */

struct _vrl_facet
	{
	int surface;            /* index into object's surface array */
	Vector normal;          /* perpendicular to facet, left-hand rule */
	int highlight : 1;      /* facet is highlighted */
	int interior  : 1;      /* facet is on the interior of an object (not used) */
	vrl_Facet *details;     /* points to linked list of detail facets (not used) */
	vrl_Facet *nearside, *farside;  /* only farside is used */
	int npoints;  /* number of points in the facet */
	int *points;  /* indices into array of vertices of the facet points */
	};

void vrl_FacetTraverse(vrl_Facet *facet, int (*function)(vrl_Facet *f));
void vrl_FacetComputeNormal(vrl_Facet *facet, Vector *vertices);

#define vrl_FacetSetSurfnum(facet, n) ((facet)->surface = (n))
#define vrl_FacetGetSurfnum(facet) ((facet)->surface)
#define vrl_FacetCountPoints(facet) ((facet)->npoints)
#define vrl_FacetGetPoint(facet, n) ((facet)->points[n])
#define vrl_FacetGetVertex(rep, facet, n) (&((rep)->vertices[(facet)->points[n]]))
#define vrl_FacetSetHighlight(facet, high) ((facet)->highlight = ((high) ? 1 : 0))
#define vrl_FacetGetHighlight(facet) ((facet)->highlight)
#define vrl_FacetToggleHighlight(facet) ((facet)->highlight = !(facet)->highlight)

/* Surfaces */

typedef enum
		{
		SURF_SIMPLE = 0, SURF_FLAT, SURF_METAL, SURF_GLASS, SURF_GOURAUD
		} vrl_SurfaceType;

struct _vrl_surface
	{
	vrl_SurfaceType type;
	unsigned char hue;
	unsigned char brightness;
	};

vrl_Surface *vrl_SurfaceDesc(unsigned int desc, vrl_Surface *surf);
vrl_Surface *vrl_SurfaceInit(vrl_Surface *surf, unsigned char hue);

#define vrl_SurfaceSetType(surf, t) ((surf)->type = (t))
#define vrl_SurfaceGetType(surf) ((surf)->type)
#define vrl_SurfaceSetHue(surf, h) ((surf)->hue = (h))
#define vrl_SurfaceGetHue(surf) ((surf)->hue)
#define vrl_SurfaceSetBrightness(surf, b) ((surf)->brightness = (b))
#define vrl_SurfaceGetBrightness(surf) ((surf)->brightness)

#define vrl_SurfacemapSetSurface(map, surfnum, surf) ((map)[surfnum] = (surf))
#define vrl_SurfacemapGetSurface(map, surfnum) ((map)[surfnum])

#define vrl_NewSurface(hue) vrl_SurfaceInit(malloc(sizeof(vrl_Surface)), hue)
#define vrl_NewSurfacemap(n) calloc((n), sizeof(vrl_Surface *))

/* Lights */

typedef enum
	{
	LIGHT_AMBIENT = 0, LIGHT_DIRECTIONAL, LIGHT_POINTSOURCE
	} vrl_LightingType;

struct _vrl_light
	{
	vrl_LightingType type;  /* type of light source */
	int on : 1;             /* set if the light is on */
	Factor intensity;       /* how bright the light is */
	vrl_Object *object;     /* the object this light is associated with, if any */
	vrl_Light *next;        /* lights are kept in a linked list */
	};

vrl_Light *vrl_LightInit(vrl_Light *light);
vrl_Light *vrl_NewLight(void);

#define vrl_LightSetType(light, ltype) ((light)->type = (ltype))
#define vrl_LightGetType(light) ((light)->type)
#define vrl_LightOn(light) ((light)->on = 1)
#define vrl_LightOff(light) ((light)->on = 0)
#define vrl_LightToggle(light) ((light)->on = !((light)->on))
#define vrl_LightIsOn(light) ((light)->on)
#define vrl_LightSetIntensity(light, inten) ((light)->intensity = (inten))
#define vrl_LightGetIntensity(light) ((light)->intensity)
#define vrl_LightAssociate(light, obj) ((light)->object = (obj))
#define vrl_LightDisAssociate(light) ((light)->object = NULL)
#define vrl_LightGetObject(light) ((light)->object)

#define vrl_LightMove(light, x, y, z) vrl_ObjectMove((light)->object, x, y, z)
#define vrl_LightRelMove(light, x, y, z) vrl_ObjectMove((light)->object, (x), (y), (z))
#define vrl_LightRotX(light, angle) vrl_ObjectRotX((light)->object, (angle))
#define vrl_LightRotY(light, angle) vrl_ObjectRotY((light)->object, (angle))
#define vrl_LightRotZ(light, angle) vrl_ObjectRotZ((light)->object, (angle))
#define vrl_LightRotVector(light, angle, vector) vrl_ObjectRotVector((light)->object, (angle), (vector))
#define vrl_LightRotReset(light) vrl_ObjectRotReset((light)->object)
#define vrl_LightAttach(obj, newparent) vrl_ObjectAttach((light)->object, (newparent))
#define vrl_LightDetach(obj) vrl_ObjectAttach((light)->object)

#define vrl_LightGetLocation(light, v) vrl_ObjectGetLocation((light)->object, (v))

#define vrl_LightGetX(light) ((light)->object->globalmat[3][X])
#define vrl_LightGetY(light) ((light)->object->globalmat[3][Y])
#define vrl_LightGetZ(light) ((light)->object->globalmat[3][Z])

/* Cameras */

struct _vrl_camera
	{
	Scalar hither;       /* distance to near clipping plane */
	float zoom;          /* zoom factor (1/tan(FOV/2)) */
	float aspect;        /* aspect ratio */
	int ortho : 1;       /* set if we want orthographic projection (not used) */
	Scalar orthodist;    /* apparent "distance" for orthographic projection (not used) */
	vrl_Object *object;  /* the object this camera is attached to */
	unsigned char need_updating;  /* set when zoom or aspect is changed */
	/* these next four are only used internally, for object culling */
	Factor aright, cright, btop, ctop;
	};

vrl_Camera *vrl_CameraInit(vrl_Camera *camera);
vrl_Camera *vrl_NewCamera(void);

#define vrl_CameraSetZoom(camera, zf) (((camera)->zoom = (zf)), (camera)->need_updating = 1)
#define vrl_CameraGetZoom(camera) ((camera)->zoom)
#define vrl_CameraSetAspect(camera, asp) (((camera)->aspect = (asp)), (camera)->need_updating = 1)
#define vrl_CameraGetAspect(camera) ((camera)->aspect)
#define vrl_CameraSetHither(camera, h) ((camera)->hither = (h))
#define vrl_CameraGetHither(camera) ((camera)->hither)
#define vrl_CameraAssociate(camera, obj) ((camera)->object = (obj))
#define vrl_CameraGetObject(camera) ((camera)->object)

#define vrl_CameraMove(camera, x, y, z) vrl_ObjectMove((camera)->object, x, y, z)
#define vrl_CameraRelMove(camera, x, y, z) vrl_ObjectRelMove((camera)->object, (x), (y), (z))
#define vrl_CameraRotX(camera, angle) vrl_ObjectRotX((camera)->object, (angle))
#define vrl_CameraRotY(camera, angle) vrl_ObjectRotY((camera)->object, (angle))
#define vrl_CameraRotZ(camera, angle) vrl_ObjectRotZ((camera)->object, (angle))
#define vrl_CameraRotVector(camera, angle, vector) vrl_ObjectRotVector((camera)->object, (angle), (vector))
#define vrl_CameraRotReset(camera) vrl_ObjectRotReset((camera)->object)
#define vrl_CameraAttach(obj, newparent) vrl_ObjectAttach((camera)->object, (newparent))
#define vrl_CameraDetach(obj) vrl_ObjectAttach((camera)->object)

#define vrl_CameraGetX(camera) ((camera)->object->globalmat[3][X])
#define vrl_CameraGetY(camera) ((camera)->object->globalmat[3][Y])
#define vrl_CameraGetZ(camera) ((camera)->object->globalmat[3][Z])

#define vrl_CameraGetLocation(camera, v) vrl_ObjectGetLocation((camera)->object, (v))

#define vrl_CameraGetForwardVector(camera, v) vrl_ObjectGetForwardVector((camera)->object, (v))
#define vrl_CameraGetRightVector(camera, v) vrl_ObjectGetRightVector((camera)->object, (v))
#define vrl_CameraGetUpVector(camera, v) vrl_ObjectGetUpVector((camera)->object, (v))

/* Rendering functions */

typedef struct
	{
	int memory : 1;       /* set if the renderer ran out of memory */
	int objects : 1;      /* set if there were too many objects  */
	int facets : 1;       /* set if there were too many facets   */
	} vrl_RenderStatus;

int vrl_RenderInit(int maxvert, int maxf, int maxobjs, int maxlights, unsigned int mempoolsize);
void vrl_RenderQuit(void);
void vrl_RenderBegin(vrl_Camera *camera, vrl_Light *lights);
void vrl_RenderSetAmbient(Factor amb);
void vrl_RenderHorizon(void);
vrl_RenderStatus *vrl_RenderObjlist(vrl_Object *objects);

void vrl_RenderMonitorInit(int x, int y);  /* monitor the specified point */
int vrl_RenderMonitorRead(vrl_Object **obj, vrl_Facet **facet, int *vertnum);  /* returns zero if none */

/* Layer support */

void vrl_LayerOn(int n);
void vrl_LayerOff(int n);
void vrl_LayerToggle(int n);
int vrl_LayerIsOn(int n);
void vrl_LayerAllOn(void);

/* File i/o routines */

void vrl_FileSetLoadpath(char *path);
char *vrl_FileFixupFilename(char *fname);
vrl_Shape *vrl_ReadPLG(FILE *in);
void vrl_SetReadPLGscale(float x, float y, float z);
void vrl_SetReadPLGoffset(float x, float y, float z);
int vrl_ReadWLD(FILE *in);
vrl_Object *vrl_ReadFIG(FILE *in, vrl_Object *parent, char *rootname);
void vrl_SetFigurePartArray(vrl_Object **ptr, int maxparts);
void vrl_SetReadFIGscale(float x, float y, float z);
vrl_Object *vrl_ReadObjectPLG(FILE *in);

/* List management routines */

struct _vrl_listnode
	{
	char *name;
	void *value;
	vrl_List *next;
	};

void *vrl_AddToList(vrl_List **list, char *name, void *value);
void *vrl_FindOnList(vrl_List *list, char *name);

/* Display routines */

typedef struct _vrl_outvertex vrl_OutputVertex;

struct _vrl_outvertex
	{
	vrl_OutputVertex *next;   /* linked list */
	long coords[3];           /* X, Y screen coordinates and Z-depth */
	unsigned int intensity;   /* used for Gouraud shading */
	unsigned char outcode;    /* used for XY clipping */
	/* future versions of AVRIL may have additional info per vertex */
	};

typedef struct
	{
	int version;          /* driver version number */
	char *driver;         /* points to driver name */
	int zbuffer;          /* 0 = none, 1 = software, 2 = hardware */
	int gouraud;          /* 0 = none, 1 = software, 2 = hardware */
	} vrl_DisplayCap;

void vrl_DisplaySetDriver(char *name);
int vrl_DisplayInit(int mode);
void vrl_DisplayQuit(void);
vrl_DisplayCap *vrl_DisplayGetCapabilities(void);
void vrl_DisplayClear(long color);
void vrl_DisplayPoint(int x, int y, long color);
void vrl_DisplayLine(int x1, int y1, int x2, int y2, long color);
void vrl_DisplayBox(int x1, int y1, int x2, int y2, long color);
void vrl_DisplayPoly(vrl_OutputVertex *vertices, vrl_Surface *surf, unsigned int intensity, int highlight, int xyclip);
void vrl_DisplayText(int x, int y, long color, char *message);
void vrl_DisplayUseZbuffer(int flag);
void vrl_DisplayClearZbuffer(void);
void vrl_DisplaySetWindow(int x1, int y1, int x2, int y2);
void vrl_DisplayGetWindow(int *x1, int *y1, int *x2, int *y2);
void vrl_DisplaySetDrawPage(int page);
int vrl_DisplayGetDrawPage(void);
void vrl_DisplaySetViewPage(int page);
int vrl_DisplayGetViewPage(void);
int vrl_DisplayGetWidth(void);
int vrl_DisplayGetHeight(void);
int vrl_DisplayGetDepth(void);
int vrl_DisplayGetNpages(void);
int vrl_DisplayGetTextHeight(char *string);
int vrl_DisplayGetTextWidth(char *string);
void vrl_DisplayUpdate(void);  /* forces ram->framebuffer copy if applicable */
void vrl_DisplaySetPalette(int start, int end, vrl_Palette *palette);  /* updates the palette */
void vrl_DisplayGetPalette(int start, int end, vrl_Palette *palette);  /* updates the palette */
void vrl_DisplayReadScanline(int n, void *buffer);
void vrl_DisplayWriteScanline(int n, void *buffer);
long vrl_DisplaySetWireframeColor(long color);
long vrl_DisplaySetHighlightColor(long color);
void vrl_DisplaySetDrawMode(int wireframe);
int vrl_DisplayGetDrawMode(void);

/* Timer routines -- 1024 ticks per second resolution */

int vrl_TimerInit(void);
void vrl_TimerQuit(void);
unsigned long vrl_TimerRead(void);
int vrl_TimerAddHandler(void (*func)(unsigned long));
unsigned long vrl_TimerGetTickRate(void);

/* Mouse routines */

int vrl_MouseInit(void);
int vrl_MouseRead(int *x, int *y, unsigned int *buttons);
void vrl_MouseCursorHide(void);
void vrl_MouseCursorShow(void);

/* Keyboard routines */

int vrl_KeyboardCheck(void);
unsigned int vrl_KeyboardRead(void);

/* System routines */

int vrl_SystemStartup(void);
void vrl_SystemRun(void);
void vrl_SystemShutdown(void);
vrl_RenderStatus *vrl_SystemRender(vrl_Object *list);
unsigned long vrl_SystemGetRenderTime(void);
void vrl_SystemCommandLine(int argc, char *argv[]);

void vrl_SystemRequestRefresh(void);
int vrl_SystemQueryRefresh(void);

void vrl_SystemStartRunning(void);
void vrl_SystemStopRunning(void);
int vrl_SystemIsRunning(void);

/* Routines defined by application */

void vrl_ApplicationDrawUnder(void);
void vrl_ApplicationDrawOver(vrl_RenderStatus *stat);
void vrl_ApplicationKey(unsigned int c);
void vrl_ApplicationMouseUp(int x, int y, unsigned int buttons);

/* User interface routines */

void vrl_UserInterfaceBox(int w, int h, int *x, int *y);
void vrl_UserInterfacePopText(char *text[]);
void vrl_UserInterfacePopMsg(char *msg);
int vrl_UserInterfacePopMenu(char *text[]);
unsigned vrl_UserInterfacePopPrompt(char *prompt, char *buff, int n);
int vrl_UserInterfaceDismiss(void);
void vrl_DrawCompass(vrl_Camera *camera, int compass_x, int compass_y, int compass_armlen);
void vrl_DropText(int x, int y, int color, char *text);

/* Pseudo-tasking routines */

int vrl_TaskCreate(void (*function)(void), void *data, unsigned long period);
void vrl_TaskRun(void);
void *vrl_TaskGetData(void);
unsigned long vrl_TaskGetElapsed(void);
unsigned long vrl_TaskGetTimeNow(void);

/* Routines for creating primitive shapes */

vrl_Shape *vrl_PrimitiveBox(Scalar width, Scalar height, Scalar depth, vrl_Surface **map);
vrl_Shape *vrl_PrimitiveCone(Scalar radius, Scalar height, int nsides, vrl_Surface **map);
vrl_Shape *vrl_PrimitiveCylinder(Scalar bottom_radius, Scalar top_radius, Scalar height, int nsides, vrl_Surface **map);
vrl_Shape *vrl_PrimitivePrism(Scalar width, Scalar height, Scalar depth, vrl_Surface **map);
vrl_Shape *vrl_PrimitiveSphere(Scalar radius, int nsides, vrl_Surface **map);

/* End of avril.h */
