/* Simple keyboard and mouse support for AVRIL demos */

/* Written by Bernie Roehl, April 1994 */

/* Copyright 1994 by Bernie Roehl */

#include <stdio.h>
#include <math.h>
#include <ctype.h>   /* isupper() */
#include <alloc.h>   /* coreleft() */
#include "avril.h"
#include "avrilkey.h"

static vrl_Object *active_object = NULL;

static int showlocation = 0, showhud = 0, showcompass = 0, showframerate = 0;

static int turnmove = 0;       /* 0 for translation, 1 for rotation */

void vrl_ApplicationDrawOver(vrl_RenderStatus *stat)
	{
	vrl_Camera *cam = vrl_WorldGetCamera();
	char buff[100];
	if (showlocation)
		{
		sprintf(buff, "Position: %ld,%ld", vrl_CameraGetX(cam), vrl_CameraGetZ(cam));
		vrl_DropText(10, 10, 15, buff);
		}
	if (showcompass)
		vrl_DrawCompass(cam, 250, 40, 35);
	if (showhud)
		{
		sprintf(buff, "%c%c%c",
			stat->memory ?  'M' : ' ',
			stat->objects ? 'O' : ' ',
			stat->facets ?  'F' : ' ');
		vrl_DropText(10, 20, 15, buff);
		sprintf(buff, "Stepsize = %f",
			scalar2float(vrl_WorldGetMovestep()));
		vrl_DropText(10, 30, 15, buff);
		sprintf(buff,"Anglesize = %f",
			angle2float(vrl_WorldGetTurnstep()));
		vrl_DropText(10, 40, 15, buff);
		sprintf(buff, "Zoom = %f, hither = %f",
			vrl_CameraGetZoom(cam), scalar2float(vrl_CameraGetHither(cam)));
		vrl_DropText(10, 50, 15, buff);
		sprintf(buff, "%ld", coreleft());
		vrl_DropText(10, 60, 15, buff);
		}
	if (showframerate)
		{
		unsigned long rtime = vrl_SystemGetRenderTime();
		if (rtime)  /* make sure we don't divide by zero */
			{
			sprintf(buff, "Frames/sec: %ld", vrl_TimerGetTickRate() / rtime);
			vrl_DropText(5, 170, 15, buff);
			}
		}
	}

static void process_key(int c)
	{
	static int gathering = 0;  /* non-zero if we're inputting angle, zero for distance */
	vrl_Camera *cam = vrl_WorldGetCamera();
	if (active_object == NULL)
		active_object = vrl_CameraGetObject(cam);
	switch (c)
		{
		case 'm':
			if (vrl_WorldGetMovementMode())
				vrl_WorldSetMovementMode(0);
			else
				vrl_WorldSetMovementMode(1);
			break;
		case UPKEY:
			{
			Vector v;
			Scalar step = vrl_WorldGetMovestep();
			vrl_CameraGetForwardVector(cam, v);
			vrl_CameraRelMove(cam,
				vrl_FactorMultiply(v[X], step),
				vrl_WorldGetMovementMode() ? 
					vrl_FactorMultiply(v[Y], step) : 0,
				vrl_FactorMultiply(v[Z], step));
			}
			break;
		case DOWNKEY:
			{
			Vector v;
			Scalar step = vrl_WorldGetMovestep();
			vrl_CameraGetForwardVector(cam, v);
			vrl_CameraRelMove(cam,
				vrl_FactorMultiply(v[X], -step),
				vrl_WorldGetMovementMode() ? 
					vrl_FactorMultiply(v[Y], -step) : 0,
				vrl_FactorMultiply(v[Z], -step));
			}
			break;
		case LSHIFT+LEFTKEY:
			{
			Vector v;
			Scalar step = vrl_WorldGetMovestep();
			vrl_CameraGetRightVector(cam, v);
			vrl_CameraRelMove(cam,
				vrl_FactorMultiply(v[X], -step),
				vrl_WorldGetMovementMode() ? 
					vrl_FactorMultiply(v[Y], -step) : 0,
				vrl_FactorMultiply(v[Z], -step));
			}
			break;
		case LSHIFT+RIGHTKEY:
			{
			Vector v;
			Scalar step = vrl_WorldGetMovestep();
			vrl_CameraGetRightVector(cam, v);
			vrl_CameraRelMove(cam,
				vrl_FactorMultiply(v[X], step),
				vrl_WorldGetMovementMode() ? 
					vrl_FactorMultiply(v[Y], step) : 0,
				vrl_FactorMultiply(v[Z], step));
			}
			break;
		case LSHIFT+UPKEY:
		case PGUPKEY:
			{
			Vector v;
			Scalar step = vrl_WorldGetMovestep();
			vrl_CameraGetUpVector(cam, v);
			vrl_CameraRelMove(cam,
				vrl_FactorMultiply(v[X], step),
				vrl_WorldGetMovementMode() ? 
					vrl_FactorMultiply(v[Y], step) : 0,
				vrl_FactorMultiply(v[Z], step));
			}
			break;
		case LSHIFT+DOWNKEY:
		case PGDNKEY:
			{
			Vector v;
			Scalar step = vrl_WorldGetMovestep();
			vrl_CameraGetUpVector(cam, v);
			vrl_CameraRelMove(cam,
				vrl_FactorMultiply(v[X], -step),
				vrl_WorldGetMovementMode() ? 
					vrl_FactorMultiply(v[Y], -step) : 0,
				vrl_FactorMultiply(v[Z], -step));
			}
			break;
		case LEFTKEY:
			{
			Vector v;
			vrl_CameraGetUpVector(cam, v);
			vrl_CameraRotVector(cam, -vrl_WorldGetTurnstep(), v);
			}
			break;
		case RIGHTKEY:
			{
			Vector v;
			vrl_CameraGetUpVector(cam, v);
			vrl_CameraRotVector(cam, vrl_WorldGetTurnstep(), v);
			}
			break;
		case RSHIFT+UPKEY:
			{
			Vector v;
			vrl_CameraGetRightVector(cam, v);
			vrl_CameraRotVector(cam, -vrl_WorldGetTurnstep(), v);
			}
			break;
		case RSHIFT+DOWNKEY:
			{
			Vector v;
			vrl_CameraGetRightVector(cam, v);
			vrl_CameraRotVector(cam, vrl_WorldGetTurnstep(), v);
			}
			break;
		case RSHIFT+LEFTKEY:
		case CTRLPGUP:
			{
			Vector v;
			vrl_CameraGetForwardVector(cam, v);
			vrl_CameraRotVector(cam, vrl_WorldGetTurnstep(), v);
			}
			break;
		case RSHIFT+RIGHTKEY:
		case CTRLPGDN:
			{
			Vector v;
			vrl_CameraGetForwardVector(cam, v);
			vrl_CameraRotVector(cam, -vrl_WorldGetTurnstep(), v);
			}
			break;
		case 'w': vrl_DisplaySetDrawMode(!vrl_DisplayGetDrawMode()); break;
		case 'q': case 0x1B: vrl_SystemStopRunning(); break;
		case 'f': showframerate = !showframerate; break;
		case 'c': showcompass = !showcompass; break;
		case 'p': showlocation = !showlocation; break;
		case 'd': showhud = !showhud; break;
		case '_': vrl_WorldToggleHorizon(); break;
		case 't': turnmove = 0; break;
		case 'r': turnmove = 1; break;
		case 'v': active_object = vrl_CameraGetObject(cam); break;
		case 'l': active_object = vrl_CameraGetObject(vrl_WorldGetLights()); break;
		case '*':
			vrl_ObjectRotReset(active_object);
			break;
		case 'u': case 'U':
			vrl_ObjectRotY(active_object, float2angle(180));
			break;
		case 'x': case 'X':
			if (turnmove)
				vrl_ObjectRotX(active_object,
					isupper(c) ? vrl_WorldGetTurnstep() : -vrl_WorldGetTurnstep());
			else
				vrl_ObjectRelMove(active_object,
					isupper(c) ? vrl_WorldGetMovestep() : -vrl_WorldGetMovestep(), 0, 0);
			break;
		case 'y': case 'Y':
			if (turnmove)
				vrl_ObjectRotY(active_object,
					isupper(c) ? vrl_WorldGetTurnstep() : -vrl_WorldGetTurnstep());
			else
				vrl_ObjectRelMove(active_object,
					0, isupper(c) ? vrl_WorldGetMovestep() : -vrl_WorldGetMovestep(), 0);
			break;
		case 'z': case 'Z':
			if (turnmove)
				vrl_ObjectRotZ(active_object,
					isupper(c) ? vrl_WorldGetTurnstep() : -vrl_WorldGetTurnstep());
			else
				vrl_ObjectRelMove(active_object,
					0, 0, isupper(c) ? vrl_WorldGetMovestep() : -vrl_WorldGetMovestep());
			break;
		case 'h':
			{
			Scalar newhither = vrl_CameraGetHither(cam) - vrl_WorldGetMovestep();
			if (newhither < 1) newhither = 1;
			vrl_CameraSetHither(cam, newhither);
			}
			break;
		case 'H':
			{
			Scalar newhither = vrl_CameraGetHither(cam) + vrl_WorldGetMovestep();
			if (newhither < 1) newhither = 1;
			vrl_CameraSetHither(cam, newhither);
			}
			break;
		case '+': vrl_CameraSetZoom(cam, vrl_CameraGetZoom(cam) * 1.1); break;
		case '-': vrl_CameraSetZoom(cam, vrl_CameraGetZoom(cam) * 0.9); break;
		case '=': vrl_CameraSetZoom(cam, 1.0); break;
		case 'a': case 'A': gathering = 1; vrl_WorldSetTurnstep(0); break;
		case 's': case 'S': gathering = 0; vrl_WorldSetMovestep(0); break;
		case '0': case '1': case '2': case '3':
		case '4': case '5': case '6': case '7':
		case '8': case '9':
			if (gathering)
				{
				float ang = angle2float(vrl_WorldGetTurnstep());
				ang = ang * 10 + c - '0';
				vrl_WorldSetTurnstep(float2angle(ang));
				}
			else
				{
				float step = scalar2float(vrl_WorldGetMovestep());
				step = step * 10 + c - '0';
				vrl_WorldSetMovestep(float2scalar(step));
				}
			break;
		default: break;
		}
	vrl_SystemRequestRefresh();
	}
	
void vrl_ApplicationKey(unsigned int c)
	{
	static int lastkey = 0;
	if (c == INSKEY)
		{
		int i;
		for (i = 0; i < 100; ++i)
			{
			process_key(lastkey);
			vrl_SystemRender(vrl_ObjectUpdate(vrl_WorldGetObjectTree()));
			}
		}
	else
		process_key(lastkey = c);
	}

void vrl_ApplicationMouseUp(int x, int y, unsigned int buttons)
	{
	if (buttons & 0x07)
		{
		vrl_RenderMonitorInit(x, y);
		vrl_SystemRender(NULL);  /* redraw screen */
		if (vrl_RenderMonitorRead(&active_object, NULL, NULL))
			if (active_object)
				{
				vrl_ObjectToggleHighlight(active_object);
				vrl_SystemRequestRefresh();
				}
		}
	}
