/* The "vrl_System" functions for AVRIL applications */

/* Written by Bernie Roehl, January 1994 */

/* Copyright 1994 by Bernie Roehl */

/* You may use this code for your own non-commercial projects without
   paying any fees or royalties.  "Non-commercial", in this context,
   means that the software you write is given away for free to anyone
   who wants it.
   
   Commercial use, including shareware, requires a licensing
   fee and a specific written agreement with the author.

   All programs created using this software (both commercial and
   non-commercial) must acknowledge the use of the AVRIL library,
   both in the documentation and in a banner screen at the start or
   end of the program.

   For more information, contact Bernie Roehl (broehl@uwaterloo.ca).

*/

#include <stdio.h>
#include <string.h>  /* strstr() */
#include <stdlib.h>  /* getenv() */

#include "avril.h"

static int system_running = 0;   /* non-zero while system is initialized and running */

void vrl_SystemStartRunning(void)
	{
	system_running = 1;
	}

void vrl_SystemStopRunning(void)
	{
	system_running = 0;
	}

int vrl_SystemIsRunning(void)
	{
	return system_running;
	}

static int need_to_redraw = 0;   /* non-zero if the screen needs updating */

void vrl_SystemRequestRefresh(void)
	{
	need_to_redraw = 1;
	}

int vrl_SystemQueryRefresh(void)
	{
	return need_to_redraw;
	}

int vrl_SystemStartup(void)
	{
	vrl_FileSetLoadpath(getenv("AVRIL"));
	vrl_Mathinit();
	if (vrl_RenderInit(800, 800, 500, 5, 65000))
		{
		printf("Could not initialize rendering engine\n");
		exit(1);
		}
	atexit(vrl_RenderQuit);
	if (vrl_TimerInit()) return -2;
	atexit(vrl_TimerQuit);
	if (vrl_DisplayInit(0)) return -3;
	atexit(vrl_DisplayQuit);
	vrl_MouseInit();
	vrl_SystemStartRunning();
	vrl_SystemRequestRefresh();
	return 0;
	}

void vrl_SystemShutdown(void)
	{
	exit(0);
	}

void vrl_SystemRun(void)
	{
	while (vrl_SystemIsRunning())
		{
		vrl_Object *list;
		if (vrl_MouseRead(NULL, NULL, NULL))  /* mouse changed */
			{
			unsigned int mouse_buttons;
			vrl_MouseRead(NULL, NULL, &mouse_buttons);
			if (mouse_buttons)  /* button down */
				{
				int mouse_x, mouse_y, win_x, win_y;
				unsigned int down_buttons = mouse_buttons;
				vrl_DisplayGetWindow(&win_x, &win_y, NULL, NULL);
				while (mouse_buttons)  /* wait for button release */
					vrl_MouseRead(&mouse_x, &mouse_y, &mouse_buttons);
				vrl_ApplicationMouseUp(mouse_x - win_x, mouse_y - win_y, down_buttons);
				}
			}
		if (vrl_KeyboardCheck())
			vrl_ApplicationKey(vrl_KeyboardRead());
		vrl_TaskRun();
		list = vrl_ObjectUpdate(vrl_WorldGetObjectTree());
		if (vrl_SystemQueryRefresh())
			vrl_SystemRender(list);
		}
	}

static unsigned long last_render_ticks;

unsigned long vrl_SystemGetRenderTime(void)
	{
	return last_render_ticks;
	}

vrl_RenderStatus *vrl_SystemRender(vrl_Object *list)
	{
	static vrl_Object *lastlist = NULL;
	int pagenum;
	vrl_RenderStatus *stat;
	long render_start;
	if (list == NULL)
		{
		list = lastlist;
		if (list == NULL)
			return NULL;
		}
	else
		lastlist = list;
	pagenum = vrl_DisplayGetDrawPage();
	if (++pagenum >= vrl_DisplayGetNpages())
		pagenum = 0;
	vrl_DisplaySetDrawPage(pagenum);
	render_start = vrl_TimerRead();
	vrl_RenderBegin(vrl_WorldGetCamera(), vrl_WorldGetLights());
	vrl_RenderSetAmbient(vrl_WorldGetAmbient());
	if (vrl_WorldGetScreenClear())
		{
		if (vrl_WorldGetHorizon() && !vrl_DisplayGetDrawMode())
			vrl_RenderHorizon();
		else
			vrl_DisplayClear(vrl_WorldGetSkyColor());
		}
	vrl_ApplicationDrawUnder();
	stat = vrl_RenderObjlist(list);
	last_render_ticks = vrl_TimerRead() - render_start;
	vrl_ApplicationDrawOver(stat);
	vrl_MouseCursorHide();
	vrl_DisplayUpdate();
	vrl_DisplaySetViewPage(pagenum);
	vrl_MouseCursorShow();
	need_to_redraw = 0;
	return stat;
	}

void vrl_SystemCommandLine(int argc, char *argv[])
	{
	int i;
	vrl_Camera *cam;
	vrl_World *w = vrl_WorldGetCurrent();
	for (i = 1; i < argc; ++i)  /* i = 1 to skip argv[0] */
		{
		FILE *in = fopen(argv[i], "r");
		if (in == NULL) continue;
		if (strstr(argv[i], ".wld"))
			vrl_ReadWLD(in);
		else if (strstr(argv[i], ".fig"))
			vrl_ReadFIG(in, NULL, NULL);
		else if (strstr(argv[i], ".plg"))
			vrl_ReadObjectPLG(in);
		/* ignore anything else */
		fclose(in);
		}
	if (vrl_WorldGetCamera())   /* have a camera already */
		return;
	cam = vrl_FindOnList(w->named_cameras, "1");
	if (cam)
		vrl_WorldSetCamera(cam);
	else
		vrl_NewCamera();
	}

